"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCurve25519KeyPair = void 0;
exports.curve25519 = curve25519;
exports.elligator2 = elligator2;
let constants;
const getConstants = () => {
    if (constants) {
        return constants;
    }
    if (typeof BigInt === 'undefined') {
        throw new Error('curve25519: BigInt not supported');
    }
    const p = 2n ** 255n - 19n;
    const J = 486662n;
    const c3 = BigInt('19681161376707505956807079304988542015446066515923890162744021073123829784752');
    const c4 = (p - 5n) / 8n;
    const a24 = (J + 2n) / 4n;
    const ctx = {
        p,
        J,
        c3,
        c4,
        a24,
    };
    constants = ctx;
    return ctx;
};
function littleEndianBytesToBigInt(bytes) {
    let result = 0n;
    for (let i = 0; i < bytes.length; i++) {
        result += BigInt(bytes[i]) << (8n * BigInt(i));
    }
    return result;
}
function bigintToLittleEndianBytes(value, length = 32) {
    const byteArray = new Uint8Array(length);
    for (let i = 0; i < length; i++) {
        byteArray[i] = Number(value & 0xffn);
        value >>= 8n;
    }
    return byteArray;
}
function pow(base, exp, mod) {
    let result = 1n;
    base = base % mod;
    while (exp > 0) {
        if (exp % 2n === 1n) {
            result = (result * base) % mod;
        }
        exp = exp >> 1n;
        base = (base * base) % mod;
    }
    return result;
}
function decodeScalar(scalar) {
    if (scalar.length !== 32) {
        throw new Error('Invalid length of scalar');
    }
    const array = new Uint8Array(scalar);
    array[0] &= 248;
    array[31] &= 127;
    array[31] |= 64;
    return littleEndianBytesToBigInt(array);
}
function decodeCoordinate(coordinate) {
    if (coordinate.length !== 32) {
        throw new Error('Invalid length of coordinate');
    }
    const array = new Uint8Array(coordinate);
    array[array.length - 1] &= 0x7f;
    return littleEndianBytesToBigInt(array);
}
function encodeCoordinate(coordinate) {
    return bigintToLittleEndianBytes(coordinate);
}
function conditionalSwap(a, b, condition) {
    return condition ? [b, a] : [a, b];
}
function ladderOperation({ p, a24 }, x1, x2, z2, x3, z3) {
    const a = (x2 + z2) % p;
    const aa = (a * a) % p;
    const b = (x2 - z2 + p) % p;
    const bb = (b * b) % p;
    const e = (aa - bb + p) % p;
    const c = (x3 + z3) % p;
    const d = (x3 - z3 + p) % p;
    const da = (d * a) % p;
    const cb = (c * b) % p;
    const t0 = (da + cb) % p;
    const x5 = (t0 * t0) % p;
    const t1 = (da - cb + p) % p;
    const t2 = (t1 * t1) % p;
    const z5 = (x1 * t2) % p;
    const x4 = (aa * bb) % p;
    const t3 = (a24 * e) % p;
    const t4 = (bb + t3) % p;
    const z4 = (e * t4) % p;
    return [x4, z4, x5, z5];
}
function curve25519(privateKey, publicKey) {
    const ctx = getConstants();
    const { p } = ctx;
    const k = decodeScalar(privateKey);
    const u = decodeCoordinate(publicKey) % p;
    const x1 = u;
    let x2 = 1n;
    let z2 = 0n;
    let x3 = u;
    let z3 = 1n;
    let swap = 0;
    for (let i = 255; i >= 0; i--) {
        const bit = Number((k >> BigInt(i)) & 1n);
        swap ^= bit;
        [x2, x3] = conditionalSwap(x2, x3, Boolean(swap));
        [z2, z3] = conditionalSwap(z2, z3, Boolean(swap));
        swap = bit;
        [x2, z2, x3, z3] = ladderOperation(ctx, x1, x2, z2, x3, z3);
    }
    [x2, x3] = conditionalSwap(x2, x3, Boolean(swap));
    [z2, z3] = conditionalSwap(z2, z3, Boolean(swap));
    const x = (pow(z2, p - 2n, p) * x2) % p;
    return Buffer.from(encodeCoordinate(x));
}
function conditionalMove(first, second, condition) {
    return condition ? second : first;
}
function elligator2(point) {
    const ctx = getConstants();
    const { p, J, c4, c3 } = ctx;
    const u = decodeCoordinate(point) % p;
    let tv1 = (u * u) % p;
    tv1 = (2n * tv1) % p;
    const xd = (tv1 + 1n) % p;
    const x1n = (-J + p) % p;
    let tv2 = (xd * xd) % p;
    const gxd = (tv2 * xd) % p;
    let gx1 = (J * tv1) % p;
    gx1 = (gx1 * x1n) % p;
    gx1 = (gx1 + tv2) % p;
    gx1 = (gx1 * x1n) % p;
    let tv3 = (gxd * gxd) % p;
    tv2 = (tv3 * tv3) % p;
    tv3 = (tv3 * gxd) % p;
    tv3 = (tv3 * gx1) % p;
    tv2 = (tv2 * tv3) % p;
    let y11 = pow(tv2, c4, p);
    y11 = (y11 * tv3) % p;
    const y12 = (y11 * c3) % p;
    tv2 = (y11 * y11) % p;
    tv2 = (tv2 * gxd) % p;
    const e1 = tv2 == gx1;
    const y1 = conditionalMove(y12, y11, e1);
    const x2n = (x1n * tv1) % p;
    tv2 = (y1 * y1) % p;
    tv2 = (tv2 * gxd) % p;
    const e3 = tv2 == gx1;
    const xn = conditionalMove(x2n, x1n, e3);
    const x = (xn * pow(xd, p - 2n, p)) % p;
    return encodeCoordinate(x);
}
const getCurve25519KeyPair = (randomPriv) => {
    randomPriv[0] &= 248;
    randomPriv[31] &= 127;
    randomPriv[31] |= 64;
    const basepoint = Buffer.alloc(32).fill(0);
    basepoint[0] = 0x09;
    return {
        publicKey: curve25519(randomPriv, basepoint),
        privateKey: randomPriv,
    };
};
exports.getCurve25519KeyPair = getCurve25519KeyPair;
//# sourceMappingURL=curve25519.js.map