"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertMultisigPubKey = exports.xpubToHDNodeType = exports.convertBitcoinXpub = exports.convertXpub = void 0;
exports.xpubDerive = xpubDerive;
const utxo_lib_1 = require("@trezor/utxo-lib");
const constants_1 = require("../constants");
const pubNode2bjsNode = (node, network) => {
    const chainCode = Buffer.from(node.chain_code, 'hex');
    const publicKey = Buffer.from(node.public_key, 'hex');
    const res = utxo_lib_1.bip32.fromPublicKey(publicKey, chainCode, network);
    res.__DEPTH = node.depth;
    res.__INDEX = node.child_num;
    res.__PARENT_FINGERPRINT = node.fingerprint;
    return res;
};
const convertXpub = (xpub, originalNetwork, requestedNetwork) => {
    const node = utxo_lib_1.bip32.fromBase58(xpub, originalNetwork);
    if (requestedNetwork) {
        node.network = requestedNetwork;
    }
    return node.toBase58();
};
exports.convertXpub = convertXpub;
const convertBitcoinXpub = (xpub, network) => {
    if (network.bip32.public === 0x0488b21e) {
        return xpub;
    }
    const node = utxo_lib_1.bip32.fromBase58(xpub);
    node.network = network;
    return node.toBase58();
};
exports.convertBitcoinXpub = convertBitcoinXpub;
const pubKey2bjsNode = (key, network) => {
    const keyNode = key.node;
    const bjsNode = pubNode2bjsNode(keyNode, network);
    const bjsXpub = bjsNode.toBase58();
    const keyXpub = (0, exports.convertXpub)(key.xpub, network);
    if (bjsXpub !== keyXpub) {
        throw constants_1.ERRORS.TypedError('Runtime', `pubKey2bjsNode: Invalid public key transmission detected. Key: ${bjsXpub}, Received: ${keyXpub}`);
    }
    return bjsNode;
};
const checkDerivation = (parBjsNode, childBjsNode, suffix) => {
    const derivedChildBjsNode = parBjsNode.derive(suffix);
    const derivedXpub = derivedChildBjsNode.toBase58();
    const compXpub = childBjsNode.toBase58();
    if (derivedXpub !== compXpub) {
        throw constants_1.ERRORS.TypedError('Runtime', `checkDerivation: Invalid child cross-check public key. Derived: ${derivedXpub}, Received: ${compXpub}`);
    }
};
function xpubDerive(xpub, childXPub, suffix, network, _requestedNetwork) {
    const resNode = pubKey2bjsNode(xpub, network);
    const childNode = pubKey2bjsNode(childXPub, network);
    checkDerivation(resNode, childNode, suffix);
    return xpub;
}
const xpubToHDNodeType = (xpub, network) => {
    const hd = utxo_lib_1.bip32.fromBase58(xpub, network);
    return {
        depth: hd.depth,
        child_num: hd.index,
        fingerprint: hd.parentFingerprint,
        public_key: hd.publicKey.toString('hex'),
        chain_code: hd.chainCode.toString('hex'),
    };
};
exports.xpubToHDNodeType = xpubToHDNodeType;
const convertMultisigPubKey = (network, utxo) => {
    if ('multisig' in utxo && utxo.multisig?.pubkeys) {
        utxo.multisig.pubkeys.forEach(pk => {
            if (typeof pk.node === 'string') {
                pk.node = (0, exports.xpubToHDNodeType)(pk.node, network);
            }
        });
    }
    return utxo;
};
exports.convertMultisigPubKey = convertMultisigPubKey;
//# sourceMappingURL=hdnodeUtils.js.map