"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFirmwareType = exports.getFirmwareMode = exports.buildIntermediaryFirmwareFileName = exports.buildLocalFirmwareFileName = exports.findBestCompatibleRelease = exports.isStrictFeatures = void 0;
const utils_1 = require("@trezor/utils");
const types_1 = require("../types");
const isStrictFeatures = (extFeatures) => [1, 2].includes(extFeatures.major_version) &&
    (extFeatures.firmware_present === false ||
        extFeatures.bootloader_mode == null ||
        extFeatures.bootloader_mode === true);
exports.isStrictFeatures = isStrictFeatures;
const findBestCompatibleRelease = (releasesOfDevice, currentVesion, checkProperty) => {
    if (!releasesOfDevice || Object.keys(releasesOfDevice).length === 0) {
        return;
    }
    const availableFirmwares = Object.values(releasesOfDevice);
    const currentFirmwareVersion = currentVesion.firmwareVersion;
    const currentBootloaderVersion = currentVesion.bootloaderVersion;
    let versionToCompare = currentFirmwareVersion;
    if (checkProperty === 'min_bootloader_version' && currentBootloaderVersion) {
        versionToCompare = currentBootloaderVersion;
    }
    else if (checkProperty === 'min_bootloader_version' && currentFirmwareVersion) {
        const currentRelease = availableFirmwares.find(fw => utils_1.versionUtils.isEqual(currentFirmwareVersion, fw.version));
        if (!currentRelease?.bootloader_version) {
            return;
        }
    }
    if (!versionToCompare) {
        return;
    }
    const sortedFirmwares = availableFirmwares.sort((a, b) => utils_1.versionUtils.isNewer(b.version, a.version) ? 1 : -1);
    const compatibleFirmware = sortedFirmwares.find(fw => utils_1.versionUtils.isNewerOrEqual(versionToCompare, fw[checkProperty]));
    return compatibleFirmware;
};
exports.findBestCompatibleRelease = findBestCompatibleRelease;
const buildLocalFirmwareFileName = (firmwareType, deviceModel, version) => {
    const firmwareTypeFileString = firmwareType === types_1.FirmwareType.BitcoinOnly ? '-bitcoinonly' : '';
    return `trezor-${deviceModel.toLowerCase()}-${version.join('.')}${firmwareTypeFileString}.bin`;
};
exports.buildLocalFirmwareFileName = buildLocalFirmwareFileName;
const buildIntermediaryFirmwareFileName = (internalModel, version) => `trezor-${internalModel.toLowerCase()}-inter-v${version}.bin`;
exports.buildIntermediaryFirmwareFileName = buildIntermediaryFirmwareFileName;
const getFirmwareMode = (features) => {
    if (features.bootloader_mode)
        return 'bootloader';
    if (!features.initialized)
        return 'initialize';
    if (features.no_backup)
        return 'seedless';
    return 'normal';
};
exports.getFirmwareMode = getFirmwareMode;
const getFirmwareType = (features) => {
    let type = types_1.FirmwareType.Universal;
    if (features.fw_vendor === 'Trezor Bitcoin-only') {
        type = types_1.FirmwareType.BitcoinOnly;
    }
    else if (features.fw_vendor === 'Trezor') {
        type = types_1.FirmwareType.Universal;
    }
    else if ((0, exports.getFirmwareMode)(features) !== 'bootloader') {
        type =
            features.capabilities &&
                features.capabilities.length > 0 &&
                !features.capabilities.includes('Capability_Bitcoin_like')
                ? types_1.FirmwareType.BitcoinOnly
                : types_1.FirmwareType.Universal;
    }
    else if ((0, exports.getFirmwareMode)(features) === 'bootloader' && features.unit_btconly) {
        type = types_1.FirmwareType.BitcoinOnly;
    }
    return type;
};
exports.getFirmwareType = getFirmwareType;
//# sourceMappingURL=firmwareUtils.js.map