"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAllNetworks = exports.getUniqueNetworks = exports.parseCoinsJson = exports.ethereumNetworkInfoBase = exports.getCoinName = exports.getCoinInfo = exports.fixCoinInfoNetwork = exports.getBech32Network = exports.getSegwitNetwork = exports.getMiscNetwork = exports.getEthereumNetwork = exports.getBitcoinNetwork = void 0;
const utils_1 = require("@trezor/utils");
const defaultFeeLevels_1 = require("./defaultFeeLevels");
const pathUtils_1 = require("../utils/pathUtils");
const bitcoinNetworks = [];
const ethereumNetworks = [];
const miscNetworks = [];
const getBitcoinNetwork = (pathOrName) => {
    if (typeof pathOrName === 'string') {
        const name = pathOrName.toLowerCase();
        return bitcoinNetworks.find(n => n.name.toLowerCase() === name ||
            n.shortcut.toLowerCase() === name ||
            n.label.toLowerCase() === name);
    }
    const slip44 = (0, pathUtils_1.fromHardened)(pathOrName[1]);
    return bitcoinNetworks.find(n => n.slip44 === slip44);
};
exports.getBitcoinNetwork = getBitcoinNetwork;
const getEthereumNetwork = (pathOrNetworkSymbol) => {
    if (typeof pathOrNetworkSymbol === 'string') {
        const networkSymbol = pathOrNetworkSymbol.toLowerCase();
        return ethereumNetworks.find(network => network.shortcut.toLowerCase() === networkSymbol);
    }
    const slip44 = (0, pathUtils_1.fromHardened)(pathOrNetworkSymbol[1]);
    return ethereumNetworks.find(n => n.slip44 === slip44);
};
exports.getEthereumNetwork = getEthereumNetwork;
const getMiscNetwork = (pathOrName) => {
    if (typeof pathOrName === 'string') {
        const name = pathOrName.toLowerCase();
        return miscNetworks.find(n => n.name.toLowerCase() === name || n.shortcut.toLowerCase() === name);
    }
    const slip44 = (0, pathUtils_1.fromHardened)(pathOrName[1]);
    return miscNetworks.find(n => n.slip44 === slip44);
};
exports.getMiscNetwork = getMiscNetwork;
const getSegwitNetwork = (coin) => {
    if (coin.segwit && typeof coin.xPubMagicSegwit === 'number') {
        return {
            ...coin.network,
            bip32: {
                ...coin.network.bip32,
                public: coin.xPubMagicSegwit,
            },
        };
    }
    return null;
};
exports.getSegwitNetwork = getSegwitNetwork;
const getBech32Network = (coin) => {
    if (coin.segwit && typeof coin.xPubMagicSegwitNative === 'number') {
        return {
            ...coin.network,
            bip32: {
                ...coin.network.bip32,
                public: coin.xPubMagicSegwitNative,
            },
        };
    }
    return null;
};
exports.getBech32Network = getBech32Network;
const fixCoinInfoNetwork = (ci, path) => {
    const coinInfo = (0, utils_1.cloneObject)(ci);
    if (path[0] === (0, pathUtils_1.toHardened)(84)) {
        const bech32Network = (0, exports.getBech32Network)(coinInfo);
        if (bech32Network) {
            coinInfo.network = bech32Network;
        }
    }
    else if (path[0] === (0, pathUtils_1.toHardened)(49)) {
        const segwitNetwork = (0, exports.getSegwitNetwork)(coinInfo);
        if (segwitNetwork) {
            coinInfo.network = segwitNetwork;
        }
    }
    else {
        coinInfo.segwit = false;
    }
    return coinInfo;
};
exports.fixCoinInfoNetwork = fixCoinInfoNetwork;
const getCoinInfo = (currency) => (0, exports.getBitcoinNetwork)(currency) || (0, exports.getEthereumNetwork)(currency) || (0, exports.getMiscNetwork)(currency);
exports.getCoinInfo = getCoinInfo;
const getCoinName = (path) => {
    const slip44 = (0, pathUtils_1.fromHardened)(path[1]);
    const network = ethereumNetworks.find(n => n.slip44 === slip44);
    return network ? network.name : 'Unknown coin';
};
exports.getCoinName = getCoinName;
const BITCOIN_SHORTCUTS = ['BTC', 'TEST', 'REGTEST'];
const parseBitcoinNetworksJson = (json) => {
    Object.keys(json).forEach(key => {
        const coin = json[key];
        const shortcut = coin.coin_shortcut;
        const isBitcoin = BITCOIN_SHORTCUTS.includes(shortcut);
        const network = {
            messagePrefix: coin.signed_message_header,
            bech32: coin.bech32_prefix,
            bip32: {
                public: coin.xpub_magic,
                private: coin.xprv_magic,
            },
            pubKeyHash: coin.address_type,
            scriptHash: coin.address_type_p2sh,
            forkId: coin.fork_id,
            wif: 0,
        };
        bitcoinNetworks.push({
            type: 'bitcoin',
            blockchainLink: coin.blockchain_link,
            cashAddrPrefix: coin.cashaddr_prefix,
            label: coin.coin_label,
            name: coin.coin_name,
            shortcut,
            curveName: coin.curve_name,
            forceBip143: coin.force_bip143,
            hashGenesisBlock: coin.hash_genesis_block,
            maxAddressLength: coin.max_address_length,
            maxFeeSatoshiKb: coin.maxfee_kb,
            minAddressLength: coin.min_address_length,
            minFeeSatoshiKb: coin.minfee_kb,
            segwit: coin.segwit,
            slip44: coin.slip44,
            support: coin.support,
            xPubMagic: coin.xpub_magic,
            xPubMagicSegwitNative: coin.xpub_magic_segwit_native,
            xPubMagicSegwit: coin.xpub_magic_segwit_p2sh,
            taproot: coin.taproot,
            network,
            isBitcoin,
            decimals: coin.decimals,
            ...(0, defaultFeeLevels_1.getBitcoinFeeLevels)(coin),
        });
    });
};
exports.ethereumNetworkInfoBase = {
    type: 'ethereum',
    decimals: 18,
};
const parseEthereumNetworksJson = (json) => {
    Object.keys(json).forEach(key => {
        const network = json[key];
        ethereumNetworks.push({
            ...exports.ethereumNetworkInfoBase,
            ...(0, defaultFeeLevels_1.getEthereumFeeLevels)(network),
            blockchainLink: network.blockchain_link,
            chainId: network.chain_id,
            label: network.label,
            name: network.name,
            shortcut: network.shortcut,
            slip44: network.slip44,
            support: network.support,
        });
    });
};
const parseMiscNetworksJSON = (json, type) => {
    Object.keys(json).forEach(key => {
        const network = json[key];
        miscNetworks.push({
            type: type || 'misc',
            blockchainLink: network.blockchain_link,
            curve: network.curve,
            label: network.name,
            name: network.name,
            shortcut: network.shortcut,
            slip44: network.slip44,
            support: network.support,
            decimals: network.decimals,
            ...(0, defaultFeeLevels_1.getMiscFeeLevels)(network),
        });
    });
};
const parseCoinsJson = (json) => {
    Object.keys(json).forEach(key => {
        switch (key) {
            case 'bitcoin':
                return parseBitcoinNetworksJson(json[key]);
            case 'eth':
                return parseEthereumNetworksJson(json[key]);
            case 'misc':
                return parseMiscNetworksJSON(json[key]);
            case 'nem':
                return parseMiscNetworksJSON(json[key], 'nem');
        }
    });
};
exports.parseCoinsJson = parseCoinsJson;
const getUniqueNetworks = (networks) => networks.reduce((result, info) => {
    if (!info || result.find(i => i.shortcut === info.shortcut))
        return result;
    return result.concat(info);
}, []);
exports.getUniqueNetworks = getUniqueNetworks;
const getAllNetworks = () => [...bitcoinNetworks, ...ethereumNetworks, ...miscNetworks];
exports.getAllNetworks = getAllNetworks;
//# sourceMappingURL=coinInfo.js.map