"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kit_1 = require("@solana/kit");
const compute_budget_1 = require("@solana-program/compute-budget");
const system_1 = require("@solana-program/system");
const token_1 = require("@solana-program/token");
const schema_utils_1 = require("@trezor/schema-utils");
const utils_1 = require("@trezor/utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const solana_1 = require("../../../types/api/solana");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const additionalInfo_1 = require("../additionalInfo");
const solanaDefinitions_1 = require("../solanaDefinitions");
const solanaUtils_1 = require("../solanaUtils");
class SolanaSignTransaction extends AbstractMethod_1.AbstractMethod {
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.requiredDeviceCapabilities = ['Capability_Solana'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Solana'), this.firmwareRange);
        const { payload } = this;
        (0, schema_utils_1.AssertWeak)(solana_1.SolanaSignTransaction, payload);
        const path = (0, pathUtils_1.validatePath)(payload.path, 2);
        this.params = {
            address_n: path,
            serialized_tx: payload.serializedTx,
            additional_info: (0, additionalInfo_1.transformAdditionalInfo)(payload.additionalInfo),
            serialize: !!payload.serialize,
        };
    }
    async initAsync() {
        const token = this.params.additional_info?.token_accounts_infos?.[0];
        if (token) {
            const tokenDefinition = await (0, solanaDefinitions_1.getSolanaTokenDefinition)({
                mintAddress: token.token_mint,
            });
            if (!tokenDefinition)
                return;
            this.params.additional_info.encoded_token = tokenDefinition;
        }
    }
    get info() {
        return 'Sign Solana transaction';
    }
    payloadToPrecomposed() {
        try {
            let messageBytes;
            if (this.payload.serialize) {
                const transaction = (0, kit_1.pipe)(this.payload.serializedTx, (0, kit_1.getBase16Encoder)().encode, (0, kit_1.getTransactionDecoder)().decode);
                messageBytes = transaction.messageBytes;
            }
            else {
                messageBytes = (0, kit_1.getBase16Encoder)().encode(this.payload.serializedTx);
            }
            const compiledMessage = (0, kit_1.pipe)(messageBytes, (0, kit_1.getCompiledTransactionMessageDecoder)().decode);
            const message = (0, kit_1.decompileTransactionMessage)(compiledMessage);
            const baseFee = new utils_1.BigNumber(solanaUtils_1.SOLANA_BASE_FEE).multipliedBy(compiledMessage.header.numSignerAccounts);
            const outputs = [];
            let feePerUnit = new utils_1.BigNumber(0);
            let feeLimit = new utils_1.BigNumber(0);
            let rent = new utils_1.BigNumber(0);
            let sendAmount = new utils_1.BigNumber(0);
            let token;
            for (const instruction of message.instructions) {
                if (instruction.programAddress === compute_budget_1.COMPUTE_BUDGET_PROGRAM_ADDRESS &&
                    instruction.data) {
                    const data = instruction.data;
                    const type = (0, compute_budget_1.identifyComputeBudgetInstruction)({ data });
                    if (type === compute_budget_1.ComputeBudgetInstruction.SetComputeUnitPrice) {
                        const parsed = (0, compute_budget_1.parseSetComputeUnitPriceInstruction)({
                            data,
                            programAddress: compute_budget_1.COMPUTE_BUDGET_PROGRAM_ADDRESS,
                        });
                        feePerUnit = new utils_1.BigNumber(parsed.data.microLamports.toString());
                    }
                    if (type === compute_budget_1.ComputeBudgetInstruction.SetComputeUnitLimit) {
                        const parsed = (0, compute_budget_1.parseSetComputeUnitLimitInstruction)({
                            data,
                            programAddress: compute_budget_1.COMPUTE_BUDGET_PROGRAM_ADDRESS,
                        });
                        feeLimit = new utils_1.BigNumber(parsed.data.units.toString());
                    }
                }
                if (instruction.programAddress === system_1.SYSTEM_PROGRAM_ADDRESS &&
                    instruction.data &&
                    instruction.accounts) {
                    const instructionSafe = {
                        ...instruction,
                        data: instruction.data,
                        accounts: instruction.accounts,
                    };
                    const type = (0, system_1.identifySystemInstruction)(instructionSafe);
                    if (type === system_1.SystemInstruction.TransferSol) {
                        const parsed = (0, system_1.parseTransferSolInstruction)(instructionSafe);
                        outputs.push({
                            address: parsed.accounts.destination.address,
                            amount: parsed.data.amount.toString(),
                            script_type: 'PAYTOADDRESS',
                        });
                        sendAmount = sendAmount.plus(parsed.data.amount.toString());
                    }
                    else if (type === system_1.SystemInstruction.CreateAccount) {
                        const parsed = (0, system_1.parseCreateAccountInstruction)(instructionSafe);
                        outputs.push({
                            address: parsed.accounts.newAccount.address,
                            amount: parsed.data.lamports.toString(),
                            script_type: 'PAYTOADDRESS',
                        });
                        rent = rent.plus(parsed.data.lamports.toString());
                    }
                }
                if (instruction.programAddress === token_1.TOKEN_PROGRAM_ADDRESS &&
                    instruction.data &&
                    instruction.accounts) {
                    const instructionSafe = {
                        ...instruction,
                        data: instruction.data,
                        accounts: instruction.accounts,
                    };
                    const type = (0, token_1.identifyTokenInstruction)(instructionSafe);
                    if (type === token_1.TokenInstruction.TransferChecked) {
                        const parsed = (0, token_1.parseTransferCheckedInstruction)(instructionSafe);
                        const destinationATA = parsed.accounts.destination.address;
                        const tokenInfo = this.payload.additionalInfo?.tokenAccountsInfos?.find(t => t.tokenAccount === destinationATA &&
                            t.tokenMint === parsed.accounts.mint.address);
                        if (!sendAmount.isZero()) {
                            throw constants_1.ERRORS.TypedError('Runtime', 'Multiple token transfers in a single transaction are not supported');
                        }
                        outputs.push({
                            address: tokenInfo?.baseAddress || destinationATA,
                            amount: parsed.data.amount.toString(),
                            script_type: 'PAYTOADDRESS',
                        });
                        sendAmount = new utils_1.BigNumber(parsed.data.amount.toString());
                        token = {
                            type: 'SPL',
                            standard: 'SPL',
                            contract: parsed.accounts.mint.address,
                            decimals: parsed.data.decimals,
                            symbol: tokenInfo?.symbol,
                        };
                    }
                }
            }
            if (outputs.length === 0) {
                throw constants_1.ERRORS.TypedError('Runtime', 'No outputs decoded');
            }
            const fee = baseFee.plus(feePerUnit.multipliedBy(feeLimit).dividedBy(1e6));
            const totalSpent = sendAmount.plus(rent).plus(fee);
            return Promise.resolve({
                type: 'final',
                inputs: [],
                outputsPermutation: [0],
                outputs,
                totalSpent: token ? sendAmount.toString() : totalSpent.toString(),
                fee: fee.toString(),
                feePerByte: feePerUnit.toString(),
                feeLimit: feeLimit.toString(),
                bytes: 0,
                max: undefined,
                isTokenKnown: false,
                token,
                createdTimestamp: 'blockhash' in message.lifetimeConstraint ? new Date().getTime() : undefined,
            });
        }
        catch (e) {
            console.error('Error in payloadToPrecomposed', e);
            return Promise.resolve(undefined);
        }
    }
    async run() {
        const cmd = this.device.getCommands();
        if (this.params.serialize) {
            const tx = await (0, solanaUtils_1.createTransactionShimFromHex)(this.params.serialized_tx);
            const { message } = await cmd.typedCall('SolanaSignTx', 'SolanaTxSignature', {
                ...this.params,
                serialized_tx: tx.serializeMessage(),
            });
            const addressCall = await cmd.typedCall('SolanaGetAddress', 'SolanaAddress', {
                address_n: this.params.address_n,
                show_display: false,
                chunkify: false,
            });
            const { address } = addressCall.message;
            tx.addSignature(address, message.signature);
            const signedSerializedTx = tx.serialize();
            return { signature: message.signature, serializedTx: signedSerializedTx };
        }
        const { message } = await cmd.typedCall('SolanaSignTx', 'SolanaTxSignature', this.params);
        return { signature: message.signature };
    }
}
exports.default = SolanaSignTransaction;
//# sourceMappingURL=solanaSignTransaction.js.map