"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const utils_1 = require("@trezor/utils");
const verifyEntropy_1 = require("../api/firmware/verifyEntropy");
const constants_1 = require("../constants");
const errors_1 = require("../constants/errors");
const AbstractMethod_1 = require("../core/AbstractMethod");
const events_1 = require("../events");
const paramsValidator_1 = require("./common/paramsValidator");
const pathUtils_1 = require("../utils/pathUtils");
class ResetDevice extends AbstractMethod_1.AbstractMethod {
    init() {
        this.allowDeviceMode = [events_1.UI.INITIALIZE, events_1.UI.SEEDLESS];
        this.useDeviceState = false;
        this.requiredPermissions = ['management'];
        this.skipFinalReload = false;
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, null, this.firmwareRange);
        const { payload } = this;
        (0, schema_utils_1.Assert)(constants_1.PROTO.ResetDevice, payload);
        this.params = {
            strength: payload.strength || 256,
            passphrase_protection: payload.passphrase_protection,
            pin_protection: payload.pin_protection,
            language: payload.language,
            label: payload.label,
            u2f_counter: payload.u2f_counter || Math.floor(Date.now() / 1000),
            skip_backup: payload.skip_backup,
            no_backup: payload.no_backup,
            backup_type: payload.backup_type,
            entropy_check: typeof payload.entropy_check === 'boolean' ? payload.entropy_check : true,
        };
    }
    get info() {
        return 'Setup device';
    }
    get confirmation() {
        return {
            view: 'device-management',
            label: 'Do you really you want to create a new wallet?',
        };
    }
    async resetDeviceWorkflow() {
        const cmd = this.device.getCommands();
        const entropy = (0, verifyEntropy_1.generateEntropy)(32).toString('hex');
        await cmd.typedCall('ResetDevice', 'EntropyRequest', this.params);
        await cmd.typedCall('EntropyAck', 'Success', { entropy });
    }
    async entropyCheckWorkflow() {
        const cmd = this.device.getCommands();
        const paths = ["m/84'/0'/0'", "m/44'/60'/0'"];
        const handleErr = (error) => {
            throw error instanceof errors_1.TransportError
                ? error
                : constants_1.ERRORS.TypedError('Failure_EntropyCheck', error.message);
        };
        let entropy = (0, verifyEntropy_1.generateEntropy)(32).toString('hex');
        let commitment = await cmd
            .typedCall('ResetDevice', 'EntropyRequest', this.params)
            .then(response => response.message.entropy_commitment);
        await cmd.typedCall('EntropyAck', 'EntropyCheckReady', { entropy });
        const tries = (0, utils_1.getRandomInt)(1, 5);
        for (let i = 0; i < tries; i++) {
            const xpubs = {};
            for (const path of paths) {
                const pubKey = await cmd
                    .getPublicKey({ address_n: (0, pathUtils_1.validatePath)(path) })
                    .catch(handleErr);
                xpubs[path] = pubKey.xpub;
            }
            const { prev_entropy, entropy_commitment } = await cmd
                .typedCall('EntropyCheckContinue', 'EntropyRequest', {})
                .then(response => response.message)
                .catch(handleErr);
            const res = await (0, verifyEntropy_1.verifyEntropy)({
                type: this.params.backup_type,
                strength: this.params.strength,
                commitment,
                hostEntropy: entropy,
                trezorEntropy: prev_entropy,
                xpubs,
            });
            if (res.error) {
                await this.device.getCurrentSession().cancelCall();
                throw constants_1.ERRORS.TypedError('Failure_EntropyCheck', res.error);
            }
            entropy = (0, verifyEntropy_1.generateEntropy)(32).toString('hex');
            commitment = entropy_commitment;
            await cmd.typedCall('EntropyAck', 'EntropyCheckReady', { entropy }).catch(handleErr);
        }
        await cmd.typedCall('EntropyCheckContinue', 'Success', { finish: true });
    }
    async run() {
        if (this.params.entropy_check && this.device.unavailableCapabilities['entropyCheck']) {
            this.params.entropy_check = false;
        }
        if (this.params.entropy_check) {
            await this.entropyCheckWorkflow();
        }
        else {
            await this.resetDeviceWorkflow();
        }
        return { message: 'Success' };
    }
}
exports.default = ResetDevice;
//# sourceMappingURL=resetDevice.js.map