"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../constants");
const AbstractMethod_1 = require("../core/AbstractMethod");
const events_1 = require("../events");
const paramsValidator_1 = require("./common/paramsValidator");
const DataManager_1 = require("../data/DataManager");
const requestLogin_1 = require("../types/api/requestLogin");
class RequestLogin extends AbstractMethod_1.AbstractMethod {
    asyncChallenge;
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, null, this.firmwareRange);
        this.useEmptyPassphrase = true;
        const { payload } = this;
        const identity = {};
        const settings = DataManager_1.DataManager.getSettings();
        if (settings.origin) {
            const [proto, host, port] = settings.origin.split(':');
            identity.proto = proto;
            identity.host = host.substring(2);
            if (port) {
                identity.port = port;
            }
            identity.index = 0;
        }
        (0, schema_utils_1.Assert)(requestLogin_1.RequestLoginSchema, payload);
        this.params = {
            identity,
            challenge_hidden: payload.challengeHidden || '',
            challenge_visual: payload.challengeVisual || '',
        };
        this.asyncChallenge = !!payload.asyncChallenge;
    }
    get info() {
        return 'Login';
    }
    async run() {
        if (this.asyncChallenge) {
            const uiPromise = this.createUiPromise(events_1.UI.LOGIN_CHALLENGE_RESPONSE);
            this.postMessage((0, events_1.createUiMessage)(events_1.UI.LOGIN_CHALLENGE_REQUEST));
            const { payload } = await uiPromise.promise;
            if (typeof payload === 'string') {
                throw constants_1.ERRORS.TypedError('Runtime', `TrezorConnect.requestLogin callback error: ${payload}`);
            }
            (0, schema_utils_1.Assert)(schema_utils_1.Type.Object({
                challengeHidden: schema_utils_1.Type.String(),
                challengeVisual: schema_utils_1.Type.String(),
            }), payload);
            this.params.challenge_hidden = payload.challengeHidden;
            this.params.challenge_visual = payload.challengeVisual;
        }
        const cmd = this.device.getCommands();
        const { message } = await cmd.typedCall('SignIdentity', 'SignedIdentity', this.params);
        return {
            address: message.address,
            publicKey: message.public_key,
            signature: message.signature,
        };
    }
}
exports.default = RequestLogin;
//# sourceMappingURL=requestLogin.js.map