"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTx = void 0;
const constants_1 = require("../../constants");
const getCommon = (tx, address_n) => ({
    address_n,
    network: (tx.version >> 24) & 0xff,
    timestamp: tx.timeStamp,
    fee: tx.fee,
    deadline: tx.deadline,
    signer: address_n ? undefined : tx.signer,
});
const transferMessage = (tx) => {
    const mosaics = tx.mosaics
        ? tx.mosaics.map(mosaic => ({
            namespace: mosaic.mosaicId.namespaceId,
            mosaic: mosaic.mosaicId.name,
            quantity: mosaic.quantity,
        }))
        : undefined;
    return {
        recipient: tx.recipient,
        amount: tx.amount,
        payload: tx.message ? tx.message.payload : undefined,
        public_key: tx.message && tx.message.type === 0x02 ? tx.message.publicKey : undefined,
        mosaics,
    };
};
const importanceTransferMessage = (tx) => ({
    mode: tx.importanceTransfer.mode,
    public_key: tx.importanceTransfer.publicKey,
});
const aggregateModificationMessage = (tx) => {
    const modifications = tx.modifications
        ? tx.modifications.map(modification => ({
            type: modification.modificationType,
            public_key: modification.cosignatoryAccount,
        }))
        : undefined;
    return {
        modifications,
        relative_change: tx.minCosignatories.relativeChange,
    };
};
const provisionNamespaceMessage = (tx) => ({
    namespace: tx.newPart,
    parent: tx.parent,
    sink: tx.rentalFeeSink,
    fee: tx.rentalFee,
});
const mosaicCreationMessage = (tx) => {
    const { levy } = tx.mosaicDefinition;
    const definition = {
        namespace: tx.mosaicDefinition.id.namespaceId,
        mosaic: tx.mosaicDefinition.id.name,
        levy: levy && levy.type,
        fee: levy && levy.fee,
        levy_address: levy && levy.recipient,
        levy_namespace: levy && levy.mosaicId && levy.mosaicId.namespaceId,
        levy_mosaic: levy && levy.mosaicId && levy.mosaicId.name,
        description: tx.mosaicDefinition.description,
    };
    const { properties } = tx.mosaicDefinition;
    if (Array.isArray(properties)) {
        properties.forEach(property => {
            const { name, value } = property;
            switch (name) {
                case 'divisibility':
                    definition.divisibility = parseInt(value, 10);
                    break;
                case 'initialSupply':
                    definition.supply = parseInt(value, 10);
                    break;
                case 'supplyMutable':
                    definition.mutable_supply = value === 'true';
                    break;
                case 'transferable':
                    definition.transferable = value === 'true';
                    break;
            }
        });
    }
    return {
        definition,
        sink: tx.creationFeeSink,
        fee: tx.creationFee,
    };
};
const supplyChangeMessage = (tx) => ({
    namespace: tx.mosaicId.namespaceId,
    mosaic: tx.mosaicId.name,
    type: tx.supplyType,
    delta: tx.delta,
});
const createTx = (tx, address_n, chunkify) => {
    let transaction = tx;
    const message = {
        chunkify: typeof chunkify === 'boolean' ? chunkify : false,
        transaction: getCommon(tx, address_n),
        transfer: undefined,
        importance_transfer: undefined,
        aggregate_modification: undefined,
        provision_namespace: undefined,
        mosaic_creation: undefined,
        supply_change: undefined,
    };
    if (tx.type === constants_1.NEM.TxType.COSIGNING ||
        tx.type === constants_1.NEM.TxType.MULTISIG ||
        tx.type === constants_1.NEM.TxType.MULTISIG_SIGNATURE) {
        message.cosigning =
            tx.type === constants_1.NEM.TxType.COSIGNING || tx.type === constants_1.NEM.TxType.MULTISIG_SIGNATURE;
        transaction = tx.otherTrans;
        message.multisig = getCommon(transaction);
    }
    switch (transaction.type) {
        case constants_1.NEM.TxType.TRANSFER:
            message.transfer = transferMessage(transaction);
            break;
        case constants_1.NEM.TxType.IMPORTANCE_TRANSFER:
            message.importance_transfer = importanceTransferMessage(transaction);
            break;
        case constants_1.NEM.TxType.AGGREGATE_MODIFICATION:
            message.aggregate_modification = aggregateModificationMessage(transaction);
            break;
        case constants_1.NEM.TxType.PROVISION_NAMESPACE:
            message.provision_namespace = provisionNamespaceMessage(transaction);
            break;
        case constants_1.NEM.TxType.MOSAIC_CREATION:
            message.mosaic_creation = mosaicCreationMessage(transaction);
            break;
        case constants_1.NEM.TxType.SUPPLY_CHANGE:
            message.supply_change = supplyChangeMessage(transaction);
            break;
        default:
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Unknown transaction type');
    }
    return message;
};
exports.createTx = createTx;
//# sourceMappingURL=nemSignTx.js.map