"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFirmwareRange = exports.validateCoinPath = void 0;
exports.validateParams = validateParams;
const utils_1 = require("@trezor/utils");
const constants_1 = require("../../constants");
const config_1 = require("../../data/config");
const pathUtils_1 = require("../../utils/pathUtils");
const invalidParameter = (message) => constants_1.ERRORS.TypedError('Method_InvalidParameter', message);
function validateParams(params, schema) {
    schema.forEach(field => {
        const value = params[field.name];
        if (field.required && value == null) {
            throw invalidParameter(`Parameter "${field.name}" is missing.`);
        }
        if (!field.type || value == null)
            return;
        const { name, type } = field;
        if (Array.isArray(type)) {
            const p = {};
            p[name] = value;
            const success = type.reduce((count, t) => {
                try {
                    validateParams(p, [{ name: field.name, type: t }]);
                    return count + 1;
                }
                catch {
                    return count;
                }
            }, 0);
            if (!success) {
                throw invalidParameter(`Parameter "${name}" has invalid type. Union of "${type.join('|')}" expected.`);
            }
            return;
        }
        if (type === 'array') {
            if (!Array.isArray(value)) {
                throw invalidParameter(`Parameter "${name}" has invalid type. "${type}" expected.`);
            }
            if (!field.allowEmpty && value.length < 1) {
                throw invalidParameter(`Parameter "${name}" is empty.`);
            }
        }
        else if (type === 'uint') {
            if (typeof value !== 'string' && typeof value !== 'number') {
                throw invalidParameter(`Parameter "${name}" has invalid type. "string|number" expected.`);
            }
            if ((typeof value === 'number' && !Number.isSafeInteger(value)) ||
                !/^(?:[1-9]\d*|\d)$/.test(value.toString().replace(/^-/, field.allowNegative ? '' : '-'))) {
                throw invalidParameter(`Parameter "${name}" has invalid value "${value}". Integer representation expected.`);
            }
        }
        else if (type === 'array-buffer') {
            if (!(value instanceof ArrayBuffer)) {
                throw invalidParameter(`Parameter "${name}" has invalid type. "ArrayBuffer" expected.`);
            }
        }
        else if (typeof value !== type) {
            throw invalidParameter(`Parameter "${name}" has invalid type. "${type}" expected.`);
        }
    });
    return params;
}
const validateCoinPath = (path, coinInfo) => {
    if (coinInfo && coinInfo.slip44 !== (0, pathUtils_1.fromHardened)(path[1])) {
        throw invalidParameter('Parameters "path" and "coin" do not match.');
    }
};
exports.validateCoinPath = validateCoinPath;
const getFirmwareRange = (method, coinInfo, currentRange) => {
    const range = JSON.parse(JSON.stringify(currentRange));
    const models = (0, utils_1.typedObjectKeys)(range);
    if (coinInfo) {
        models.forEach(model => {
            const supportVersion = coinInfo.support ? coinInfo.support[model] : false;
            if (supportVersion === false) {
                range[model].min = '0';
            }
            else if (range[model].min !== '0' &&
                typeof supportVersion === 'string' &&
                utils_1.versionUtils.isNewer(supportVersion, range[model].min)) {
                range[model].min = supportVersion;
            }
        });
    }
    const coinType = coinInfo?.type;
    const shortcut = coinInfo?.shortcut.toLowerCase();
    const configRules = config_1.config.supportedFirmware
        .filter(rule => {
        if (rule.methods) {
            return rule.methods.includes(method);
        }
        if (rule.capabilities) {
            return rule.capabilities.includes(method);
        }
        return true;
    })
        .filter(rule => {
        if (rule.coinType) {
            return rule.coinType === coinType;
        }
        if (rule.coin) {
            return (typeof rule.coin === 'string' ? [rule.coin] : rule.coin).includes(shortcut);
        }
        return rule.methods || rule.capabilities;
    });
    configRules.forEach(rule => {
        if (rule.min) {
            models.forEach(model => {
                const modelMin = rule.min[model];
                if (modelMin) {
                    if (modelMin === '0' ||
                        range[model].min === '0' ||
                        !utils_1.versionUtils.isNewerOrEqual(range[model].min, modelMin)) {
                        range[model].min = modelMin;
                    }
                }
            });
        }
        if (rule.max) {
            models.forEach(model => {
                const modelMax = rule.max[model];
                if (modelMax) {
                    if (modelMax === '0' ||
                        range[model].max === '0' ||
                        !utils_1.versionUtils.isNewerOrEqual(range[model].max, modelMax)) {
                        range[model].max = modelMax;
                    }
                }
            });
        }
    });
    return range;
};
exports.getFirmwareRange = getFirmwareRange;
//# sourceMappingURL=paramsValidator.js.map