"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const cardano_1 = require("../../../types/api/cardano");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
class CardanoGetNativeScriptHash extends AbstractMethod_1.AbstractMethod {
    init() {
        this.requiredPermissions = ['read'];
        this.requiredDeviceCapabilities = ['Capability_Cardano'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Cardano'), this.firmwareRange);
        const { payload } = this;
        (0, schema_utils_1.Assert)(cardano_1.CardanoGetNativeScriptHash, payload);
        this.validateScript(payload.script);
        this.params = {
            script: this.scriptToProto(payload.script),
            display_format: payload.displayFormat,
            derivation_type: typeof payload.derivationType !== 'undefined'
                ? payload.derivationType
                : constants_1.PROTO.CardanoDerivationType.ICARUS_TREZOR,
        };
    }
    get info() {
        return 'Get Cardano native script hash';
    }
    validateScript(script) {
        if (script.keyPath) {
            (0, pathUtils_1.validatePath)(script.keyPath, 3);
        }
        if (script.scripts) {
            script.scripts.forEach(nestedScript => {
                this.validateScript(nestedScript);
            });
        }
    }
    scriptToProto(script) {
        let scripts = [];
        if (script.scripts) {
            scripts = script.scripts.map(nestedScript => this.scriptToProto(nestedScript));
        }
        let keyPath = [];
        if (script.keyPath) {
            keyPath = (0, pathUtils_1.validatePath)(script.keyPath, 3);
        }
        return {
            type: script.type,
            scripts,
            key_hash: script.keyHash,
            key_path: keyPath,
            required_signatures_count: script.requiredSignaturesCount,
            invalid_before: script.invalidBefore,
            invalid_hereafter: script.invalidHereafter,
        };
    }
    async run() {
        const { message } = await this.device
            .getCommands()
            .typedCall('CardanoGetNativeScriptHash', 'CardanoNativeScriptHash', {
            script: this.params.script,
            display_format: this.params.display_format,
            derivation_type: this.params.derivation_type,
        });
        return {
            scriptHash: message.script_hash,
        };
    }
}
exports.default = CardanoGetNativeScriptHash;
//# sourceMappingURL=cardanoGetNativeScriptHash.js.map