"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initIframeLogger = exports.clearTimeout = exports.postMessage = exports.init = exports.dispose = exports.error = exports.timeout = exports.initPromise = exports.origin = exports.instance = void 0;
const tslib_1 = require("tslib");
const ERRORS = tslib_1.__importStar(require("@trezor/connect/lib/constants/errors"));
const events_1 = require("@trezor/connect/lib/events");
const debug_1 = require("@trezor/connect/lib/utils/debug");
const urlUtils_1 = require("@trezor/connect/lib/utils/urlUtils");
const utils_1 = require("@trezor/utils");
const inlineStyles_1 = tslib_1.__importDefault(require("./inlineStyles"));
exports.initPromise = (0, utils_1.createDeferred)();
exports.timeout = 0;
const dispose = () => {
    if (exports.instance && exports.instance.parentNode) {
        try {
            exports.instance.parentNode.removeChild(exports.instance);
        }
        catch {
        }
    }
    exports.instance = null;
    exports.timeout = 0;
};
exports.dispose = dispose;
const handleIframeBlocked = () => {
    window.clearTimeout(exports.timeout);
    exports.error = ERRORS.TypedError('Init_IframeBlocked');
    (0, exports.dispose)();
    exports.initPromise.reject(exports.error);
};
const injectStyleSheet = () => {
    if (!exports.instance) {
        throw ERRORS.TypedError('Init_IframeBlocked');
    }
    const doc = exports.instance.ownerDocument;
    const head = doc.head || doc.getElementsByTagName('head')[0];
    const style = document.createElement('style');
    style.setAttribute('type', 'text/css');
    style.setAttribute('id', 'TrezorConnectStylesheet');
    if (style.styleSheet) {
        style.styleSheet.cssText = inlineStyles_1.default;
        head.appendChild(style);
    }
    else {
        style.appendChild(document.createTextNode(inlineStyles_1.default));
        head.append(style);
    }
};
const init = async (settings) => {
    exports.initPromise = (0, utils_1.createDeferred)();
    const existedFrame = document.getElementById('trezorconnect');
    if (existedFrame) {
        exports.instance = existedFrame;
    }
    else {
        exports.instance = document.createElement('iframe');
        exports.instance.frameBorder = '0';
        exports.instance.width = '0px';
        exports.instance.height = '0px';
        exports.instance.style.position = 'absolute';
        exports.instance.style.display = 'none';
        exports.instance.style.border = '0px';
        exports.instance.style.width = '0px';
        exports.instance.style.height = '0px';
        exports.instance.id = 'trezorconnect';
    }
    let src;
    if (settings.env === 'web') {
        const manifestString = settings.manifest ? JSON.stringify(settings.manifest) : 'undefined';
        const manifest = `version=${settings.version}&manifest=${encodeURIComponent(btoa(JSON.stringify(manifestString)))}`;
        src = `${settings.iframeSrc}?${manifest}`;
    }
    else {
        src = settings.iframeSrc;
    }
    if (!src.startsWith('http://') && !src.startsWith('https://')) {
        return;
    }
    exports.instance.setAttribute('src', src);
    if (navigator.usb) {
        exports.instance.setAttribute('allow', 'usb');
    }
    exports.origin = (0, urlUtils_1.getOrigin)(exports.instance.src);
    exports.timeout = window.setTimeout(() => {
        exports.initPromise.reject(ERRORS.TypedError('Init_IframeTimeout'));
    }, 10000);
    const onLoad = () => {
        if (!exports.instance) {
            exports.initPromise.reject(ERRORS.TypedError('Init_IframeBlocked'));
            return;
        }
        try {
            const iframeOrigin = exports.instance.contentWindow?.location.origin;
            if (!iframeOrigin || iframeOrigin === 'null') {
                handleIframeBlocked();
                return;
            }
        }
        catch {
        }
        let extension;
        if (typeof chrome !== 'undefined' &&
            chrome.runtime &&
            typeof chrome.runtime.onConnect !== 'undefined') {
            chrome.runtime.onConnect.addListener(() => { });
            extension = chrome.runtime.id;
        }
        exports.instance.contentWindow?.postMessage({
            type: events_1.IFRAME.INIT,
            payload: {
                settings,
                extension,
            },
        }, exports.origin);
        exports.instance.onload = null;
    };
    if (exports.instance.attachEvent) {
        exports.instance.attachEvent('onload', onLoad);
    }
    else {
        exports.instance.onload = onLoad;
    }
    if (document.body) {
        document.body.appendChild(exports.instance);
        injectStyleSheet();
    }
    try {
        await exports.initPromise.promise;
    }
    catch (e) {
        if (exports.instance) {
            if (exports.instance.parentNode) {
                exports.instance.parentNode.removeChild(exports.instance);
            }
            exports.instance = null;
        }
        throw e;
    }
    finally {
        window.clearTimeout(exports.timeout);
        exports.timeout = 0;
    }
};
exports.init = init;
const postMessage = (message) => {
    if (!exports.instance) {
        throw ERRORS.TypedError('Init_IframeBlocked');
    }
    exports.instance.contentWindow?.postMessage(message, exports.origin);
};
exports.postMessage = postMessage;
const clearTimeout = () => {
    window.clearTimeout(exports.timeout);
};
exports.clearTimeout = clearTimeout;
const initIframeLogger = () => {
    const logWriterFactory = () => ({
        add: (message) => {
            (0, exports.postMessage)({
                type: events_1.IFRAME.LOG,
                payload: message,
            });
        },
    });
    (0, debug_1.setLogWriter)(logWriterFactory);
};
exports.initIframeLogger = initIframeLogger;
//# sourceMappingURL=index.js.map