"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Stellar;
const tslib_1 = require("tslib");
const stellar_sdk_1 = require("@stellar/stellar-sdk");
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const utils = tslib_1.__importStar(require("@trezor/blockchain-link-utils/lib/stellar"));
const env_utils_1 = require("@trezor/env-utils");
const utils_1 = require("@trezor/utils");
const baseWorker_1 = require("../baseWorker");
const BASE_INFO = {
    BASE_RESERVE: utils.toStroops('0.5'),
    MINIMUM_RESERVE: utils.toStroops('1'),
};
const fetchLatestLedger = async (api) => {
    const latestLedgerInfo = await api.ledgers().order('desc').limit(1).call();
    if (latestLedgerInfo.records.length === 0) {
        throw new errors_1.CustomError('worker_invalid_horizon_response');
    }
    return latestLedgerInfo.records[0];
};
const getInfo = async (request, isTestnet) => {
    const api = await request.connect();
    const horizonServerInfo = await api.root();
    const { sequence: blockHeight, hash: blockHash, base_reserve_in_stroops: baseReserveInStroops, } = await fetchLatestLedger(api);
    BASE_INFO.BASE_RESERVE = new utils_1.BigNumber(baseReserveInStroops);
    BASE_INFO.MINIMUM_RESERVE = BASE_INFO.BASE_RESERVE.times(2);
    const serverInfo = {
        url: api.serverURL.toString(),
        name: 'Stellar',
        shortcut: isTestnet ? 'txlm' : 'xlm',
        network: isTestnet ? 'txlm' : 'xlm',
        testnet: isTestnet,
        version: horizonServerInfo.horizon_version,
        decimals: utils.STELLAR_DECIMALS,
        blockHeight,
        blockHash,
    };
    return {
        type: constants_1.RESPONSES.GET_INFO,
        payload: { ...serverInfo },
    };
};
const getAccountInfo = async (request) => {
    const { payload } = request;
    const account = {
        descriptor: payload.descriptor,
        balance: '0',
        availableBalance: '0',
        empty: true,
        history: {
            total: -1,
            unconfirmed: 0,
            transactions: undefined,
        },
        misc: {
            stellarSequence: '0',
            reserve: BASE_INFO.MINIMUM_RESERVE.toString(),
        },
    };
    const api = await request.connect();
    let info;
    try {
        info = await api.accounts().accountId(payload.descriptor).call();
    }
    catch {
        return {
            type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
            payload: account,
        };
    }
    const reserve = BASE_INFO.MINIMUM_RESERVE.plus(BASE_INFO.BASE_RESERVE.times(info.subentry_count));
    account.misc = {
        stellarSequence: info.sequence,
        reserve: reserve.toString(),
    };
    const nativeTokenBalance = info.balances.find(balance => balance.asset_type === 'native');
    if (!nativeTokenBalance) {
        throw new errors_1.CustomError('stellar_missing_native_balance');
    }
    const sellingLiabilities = utils.toStroops(nativeTokenBalance.selling_liabilities);
    account.balance = utils.toStroops(nativeTokenBalance.balance).toString();
    account.availableBalance = new utils_1.BigNumber(account.balance)
        .minus(reserve)
        .minus(sellingLiabilities)
        .minus(BASE_INFO.BASE_RESERVE.times(info.num_sponsoring))
        .plus(BASE_INFO.BASE_RESERVE.times(info.num_sponsored))
        .toString();
    const tokenMetadata = await request.getTokenMetadata();
    account.tokens = info.balances
        .filter(balanceInfo => balanceInfo.asset_type === 'credit_alphanum4' ||
        balanceInfo.asset_type === 'credit_alphanum12')
        .map(balanceInfo => {
        const contract = `${balanceInfo.asset_code}-${balanceInfo.asset_issuer}`;
        const balance = utils.toStroops(balanceInfo.balance);
        return {
            type: 'STELLAR-CLASSIC',
            standard: 'STELLAR-CLASSIC',
            contract,
            balance: balance.toString(),
            name: tokenMetadata[contract]?.name || balanceInfo.asset_code,
            symbol: tokenMetadata[contract]?.symbol || balanceInfo.asset_code,
            decimals: utils.STELLAR_DECIMALS,
        };
    });
    account.empty = false;
    if (payload.details !== 'txs') {
        return {
            type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
            payload: account,
        };
    }
    const requestBuilder = await api
        .transactions()
        .forAccount(payload.descriptor)
        .includeFailed(true)
        .limit(payload.pageSize || 20)
        .order('desc');
    if (payload.page && payload.page !== 1 && payload.pageCursor) {
        requestBuilder.cursor(payload.pageCursor);
    }
    const transactions = await requestBuilder.call();
    const cursor = transactions.records[transactions.records.length - 1]?.paging_token;
    account.history.transactions = transactions.records.map(record => utils.transformTransaction(record, payload.descriptor));
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
        payload: {
            ...account,
            stellarCursor: cursor,
        },
    };
};
const estimateFee = async (request) => {
    const api = await request.connect();
    const feeStats = await api.feeStats();
    const stroops = feeStats.fee_charged.p70;
    const payload = request.payload && Array.isArray(request.payload.blocks)
        ? request.payload.blocks.map(() => ({ feePerUnit: stroops }))
        : [{ feePerUnit: stroops }];
    return {
        type: constants_1.RESPONSES.ESTIMATE_FEE,
        payload,
    };
};
const BLOCK_SUBSCRIBE_INTERVAL_MS = 1000 * 15;
const subscribeBlock = async ({ state, connect, post }) => {
    if (state.getSubscription('block'))
        return { subscribed: true };
    const api = await connect();
    const fetchBlock = async () => {
        const { sequence: blockHeight, hash: blockHash } = await fetchLatestLedger(api);
        post({
            id: -1,
            type: constants_1.RESPONSES.NOTIFICATION,
            payload: {
                type: 'block',
                payload: {
                    blockHeight,
                    blockHash,
                },
            },
        });
    };
    fetchBlock();
    const interval = setInterval(fetchBlock, BLOCK_SUBSCRIBE_INTERVAL_MS);
    state.addSubscription('block', interval);
    return { subscribed: true };
};
const unsubscribeBlock = ({ state }) => {
    if (!state.getSubscription('block'))
        return { subscribed: false };
    const interval = state.getSubscription('block');
    clearInterval(interval);
    state.removeSubscription('block');
    return { subscribed: false };
};
const subscribe = async (request) => {
    let response;
    switch (request.payload.type) {
        case 'block':
            response = await subscribeBlock(request);
            break;
        case 'accounts':
        case 'addresses':
            response = { subscribed: false };
            break;
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
    return {
        type: constants_1.RESPONSES.SUBSCRIBE,
        payload: response,
    };
};
const unsubscribe = (request) => {
    let response;
    switch (request.payload.type) {
        case 'block':
            response = unsubscribeBlock(request);
            break;
        case 'accounts':
        case 'addresses':
            response = { subscribed: false };
            break;
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
    return {
        type: constants_1.RESPONSES.UNSUBSCRIBE,
        payload: response,
    };
};
const pushTransaction = async ({ connect, payload }) => {
    const api = await connect();
    const base64EncodedTx = Buffer.from(payload.hex, 'hex').toString('base64');
    const parsedTx = new stellar_sdk_1.Transaction(base64EncodedTx, stellar_sdk_1.Networks.PUBLIC);
    try {
        const resp = await api.submitTransaction(parsedTx, { skipMemoRequiredCheck: true });
        return {
            type: constants_1.RESPONSES.PUSH_TRANSACTION,
            payload: resp.hash,
        };
    }
    catch (e) {
        const txResultCode = e?.response?.data?.extras?.result_codes?.transaction || 'unknown';
        const opResultCode = e?.response?.data?.extras?.result_codes?.operations?.[0] || 'unknown';
        throw new Error(`transaction result code: ${txResultCode}, operation result code: ${opResultCode}`);
    }
};
const onRequest = (request, isTestnet) => {
    switch (request.type) {
        case constants_1.MESSAGES.GET_INFO:
            return getInfo(request, isTestnet);
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return getAccountInfo(request);
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return estimateFee(request);
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return pushTransaction(request);
        case constants_1.MESSAGES.SUBSCRIBE:
            return subscribe(request);
        case constants_1.MESSAGES.UNSUBSCRIBE:
            return unsubscribe(request);
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class StellarWorker extends baseWorker_1.BaseWorker {
    lazyTokens = (0, utils_1.createLazy)(() => utils.getTokenMetadata());
    isTestnet = false;
    isConnected(api) {
        return !!api;
    }
    async tryConnect(url) {
        const api = new stellar_sdk_1.Horizon.Server(url, {
            headers: {
                ...((0, env_utils_1.isDesktop)() || (0, env_utils_1.isNative)()
                    ? { 'User-Agent': `Trezor Suite ${(0, env_utils_1.getSuiteVersion)()}` }
                    : {}),
            },
        });
        if ((await api.root()).network_passphrase == stellar_sdk_1.Networks.TESTNET) {
            this.isTestnet = true;
        }
        return api;
    }
    disconnect() {
        if (!this.api) {
            return;
        }
        unsubscribeBlock({
            state: this.state,
            connect: () => this.connect(),
            post: (data) => this.post(data),
            getTokenMetadata: this.lazyTokens.getOrInit,
        });
        this.api = undefined;
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                post: (data) => this.post(data),
                state: this.state,
                getTokenMetadata: this.lazyTokens.getOrInit,
            };
            const response = await onRequest(request, this.isTestnet);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, error);
        }
    }
}
function Stellar() {
    return new StellarWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new StellarWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map