"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTransactions = void 0;
const utils_1 = require("@trezor/utils");
const transform_1 = require("./transform");
const transformOpReturn = (hex) => {
    const [, _len, data] = hex.match(/^6a(?:4c)?([0-9a-f]{2})([0-9a-f]*)$/i) ?? [];
    return data ? `OP_RETURN (${Buffer.from(data, 'hex').toString('ascii')})` : undefined;
};
const parseAddresses = ({ address, addresses, type, hex }) => {
    if (type === 'nulldata') {
        const opReturn = transformOpReturn(hex);
        return {
            addresses: opReturn ? [opReturn] : [],
            isAddress: false,
        };
    }
    const addrs = !address ? addresses || [] : [address];
    return {
        addresses: addrs,
        isAddress: addrs.length === 1,
    };
};
const isNotCoinbase = (item) => item.txid !== undefined;
const formatTransaction = (getVout, getSpent, currentHeight) => (tx) => {
    const { txid, version, vin, vout, hex, blockhash, confirmations, blocktime, locktime, vsize, size, } = tx;
    const vinRegular = vin.filter(isNotCoinbase);
    const value = vout.map(({ value }) => value).reduce((a, b) => a + b, 0);
    const valueIn = vinRegular
        .map(({ txid, vout }) => getVout(txid, vout).value)
        .reduce((a, b) => a + b, 0);
    return {
        txid,
        hex,
        version,
        confirmations: confirmations ?? 0,
        lockTime: locktime,
        blockTime: blocktime,
        blockHash: blockhash,
        blockHeight: currentHeight && confirmations ? currentHeight - confirmations + 1 : -1,
        value: (0, transform_1.btcToSat)(value),
        valueIn: (0, transform_1.btcToSat)(valueIn),
        fees: (0, transform_1.btcToSat)(valueIn - value),
        vsize,
        size,
        vin: vinRegular.map(({ txid, vout, sequence, n }, index) => ({
            txid,
            vout,
            sequence,
            n: n || index,
            value: (0, transform_1.btcToSat)(getVout(txid, vout).value),
            ...parseAddresses(getVout(txid, vout).scriptPubKey),
        })),
        vout: vout.map(({ value, n, scriptPubKey }) => ({
            value: (0, transform_1.btcToSat)(value),
            n,
            spent: getSpent(txid, n),
            hex: scriptPubKey.hex,
            ...parseAddresses(scriptPubKey),
        })),
    };
};
const getTransactions = async (client, history) => {
    const txids = history.map(({ tx_hash }) => tx_hash).filter(utils_1.arrayDistinct);
    const origTxs = await Promise.all(txids.map(txid => client.request('blockchain.transaction.get', txid, true))).then(txs => (0, utils_1.arrayToDictionary)(txs, ({ txid }) => txid));
    const prevTxs = await Promise.all(Object.values(origTxs)
        .flatMap(({ vin }) => vin.filter(isNotCoinbase).map(({ txid }) => txid))
        .filter(utils_1.arrayDistinct)
        .filter(txid => !origTxs[txid])
        .map(txid => client.request('blockchain.transaction.get', txid, true))).then(txs => (0, utils_1.arrayToDictionary)(txs, ({ txid }) => txid));
    const getSpent = () => false;
    const getTx = (txid) => origTxs[txid] || prevTxs[txid];
    const getVout = (txid, vout) => getTx(txid).vout[vout];
    const currentHeight = client.getInfo()?.block?.height || 0;
    return Object.values(origTxs).map(formatTransaction(getVout, getSpent, currentHeight));
};
exports.getTransactions = getTransactions;
//# sourceMappingURL=transaction.js.map