"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Electrum;
const tslib_1 = require("tslib");
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const baseWorker_1 = require("../baseWorker");
const caching_1 = require("./client/caching");
const L = tslib_1.__importStar(require("./listeners"));
const M = tslib_1.__importStar(require("./methods"));
const sockets_1 = require("./sockets");
const onRequest = async (request) => {
    const client = await request.connect();
    switch (request.type) {
        case constants_1.MESSAGES.GET_INFO:
            return {
                type: constants_1.RESPONSES.GET_INFO,
                payload: await M.getInfo(client),
            };
        case constants_1.MESSAGES.GET_BLOCK_HASH:
            return {
                type: constants_1.RESPONSES.GET_BLOCK_HASH,
                payload: await M.getBlockHash(client, request.payload),
            };
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return {
                type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
                payload: await M.getAccountInfo(client, request.payload),
            };
        case constants_1.MESSAGES.GET_ACCOUNT_UTXO:
            return {
                type: constants_1.RESPONSES.GET_ACCOUNT_UTXO,
                payload: await M.getAccountUtxo(client, request.payload),
            };
        case constants_1.MESSAGES.GET_TRANSACTION:
            return {
                type: constants_1.RESPONSES.GET_TRANSACTION,
                payload: await M.getTransaction(client, request.payload),
            };
        case constants_1.MESSAGES.GET_TRANSACTION_HEX:
            return {
                type: constants_1.RESPONSES.GET_TRANSACTION_HEX,
                payload: await client.request('blockchain.transaction.get', request.payload, false),
            };
        case constants_1.MESSAGES.GET_ACCOUNT_BALANCE_HISTORY:
            return {
                type: constants_1.RESPONSES.GET_ACCOUNT_BALANCE_HISTORY,
                payload: await M.getAccountBalanceHistory(client, request.payload),
            };
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return {
                type: constants_1.RESPONSES.ESTIMATE_FEE,
                payload: await M.estimateFee(client, request.payload),
            };
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return {
                type: constants_1.RESPONSES.PUSH_TRANSACTION,
                payload: await M.pushTransaction(client, request.payload),
            };
        case constants_1.MESSAGES.SUBSCRIBE:
            switch (request.payload.type) {
                case 'block':
                    return {
                        type: constants_1.RESPONSES.SUBSCRIBE,
                        payload: request.blockListener.subscribe(),
                    };
                case 'addresses':
                case 'accounts':
                    return {
                        type: constants_1.RESPONSES.SUBSCRIBE,
                        payload: await request.txListener.subscribe(request.payload),
                    };
                default:
                    throw new errors_1.CustomError(`Subscription ${request.payload.type} not implemented`);
            }
        case constants_1.MESSAGES.UNSUBSCRIBE:
            switch (request.payload.type) {
                case 'block':
                    return {
                        type: constants_1.RESPONSES.UNSUBSCRIBE,
                        payload: request.blockListener.unsubscribe(),
                    };
                case 'addresses':
                case 'accounts':
                    return {
                        type: constants_1.RESPONSES.UNSUBSCRIBE,
                        payload: await request.txListener.unsubscribe(request.payload),
                    };
                default:
                    throw new errors_1.CustomError(`Subscription ${request.payload.type} not implemented`);
            }
        case 'raw': {
            const { method, params } = request.payload;
            return client
                .request(method, ...params)
                .then((res) => ({ type: method, payload: res }));
        }
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class ElectrumWorker extends baseWorker_1.BaseWorker {
    blockListener;
    txListener;
    constructor() {
        super();
        this.blockListener = L.blockListener(this);
        this.txListener = L.txListener(this);
    }
    isConnected(api) {
        return api?.isConnected() ?? false;
    }
    async tryConnect(url) {
        const { debug, timeout, keepAlive, name } = this.settings;
        const socket = (0, sockets_1.createSocket)(url, {
            timeout,
            keepAlive,
            proxyAgent: this.proxyAgent,
        });
        const api = new caching_1.CachingElectrumClient();
        await api.connect(socket, {
            url,
            coin: name ?? 'BTC',
            debug,
            client: {
                name: 'blockchain-link',
                protocolVersion: '1.4',
            },
        });
        this.post({
            id: -1,
            type: constants_1.RESPONSES.CONNECTED,
        });
        return api;
    }
    disconnect() {
        if (this.api?.isConnected()) {
            this.api.close();
        }
    }
    cleanup() {
        if (this.api) {
            this.api.close();
        }
        super.cleanup();
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                post: (data) => this.post(data),
                state: this.state,
                blockListener: this.blockListener,
                txListener: this.txListener,
            };
            const response = await onRequest(request);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, error);
        }
    }
}
function Electrum() {
    return new ElectrumWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new ElectrumWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map