"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectrumClient = void 0;
const utxo_lib_1 = require("@trezor/utxo-lib");
const batching_1 = require("./batching");
const KEEP_ALIVE_INTERVAL = 120 * 1000;
const selectNetwork = (shortcut) => {
    switch (shortcut) {
        case 'REGTEST':
            return utxo_lib_1.networks.regtest;
        case 'TEST':
            return utxo_lib_1.networks.testnet;
        default:
            return utxo_lib_1.networks.bitcoin;
    }
};
class ElectrumClient extends batching_1.BatchingJsonRpcClient {
    options;
    network;
    version;
    lastBlock;
    timeLastCall = 0;
    async connect(socket, options) {
        this.timeLastCall = 0;
        this.options = options;
        this.network = selectNetwork(options.coin);
        const { name, protocolVersion } = options.client;
        await super.connect(socket, options);
        try {
            this.version = await this.request('server.version', name, protocolVersion);
            if (this.version[0]?.startsWith('ElectrumPersonalServer') ||
                this.version[0]?.startsWith('electrs-esplora')) {
                this.batchingDisabled = true;
            }
            this.on('blockchain.headers.subscribe', this.onBlock.bind(this));
            this.lastBlock = await this.request('blockchain.headers.subscribe');
        }
        catch (err) {
            this.socket = undefined;
            throw new Error(`Communication with Electrum server failed: [${err}]`);
        }
        this.keepAlive();
    }
    getInfo() {
        if (this.options?.url && this.version && this.lastBlock && this.network) {
            return {
                url: this.options.url,
                version: this.version,
                block: this.lastBlock,
                coin: this.options.coin,
                network: this.network,
            };
        }
    }
    onBlock(blocks) {
        const [last] = blocks.sort((a, b) => b.height - a.height);
        if (last)
            this.lastBlock = last;
    }
    request(method, ...params) {
        this.timeLastCall = new Date().getTime();
        return super.request(method, ...params);
    }
    keepAliveHandle;
    keepAlive() {
        if (!this.socket)
            return;
        this.keepAliveHandle = setInterval(async () => {
            if (this.timeLastCall !== 0 &&
                new Date().getTime() > this.timeLastCall + KEEP_ALIVE_INTERVAL / 2) {
                await this.request('server.ping').catch(err => {
                    console.error(`Ping to server failed: [${err}]`);
                    this.close();
                });
            }
        }, KEEP_ALIVE_INTERVAL);
    }
    onClose() {
        super.onClose();
        if (this.keepAliveHandle)
            clearInterval(this.keepAliveHandle);
    }
}
exports.ElectrumClient = ElectrumClient;
//# sourceMappingURL=electrum.js.map