"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BlockbookAPI = void 0;
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const env_utils_1 = require("@trezor/env-utils");
const baseWebsocket_1 = require("../baseWebsocket");
class BlockbookAPI extends baseWebsocket_1.BaseWebsocket {
    createWebsocket() {
        let { url } = this.options;
        if (typeof url !== 'string') {
            throw new errors_1.CustomError('websocket_no_url');
        }
        if (url.startsWith('http')) {
            url = url.replace('http', 'ws');
        }
        if (!url.endsWith('/websocket')) {
            const suffix = url.endsWith('/') ? 'websocket' : '/websocket';
            url += suffix;
        }
        return this.initWebsocket({
            url,
            agent: this.options.agent,
            headers: {
                'User-Agent': `Trezor Suite ${(0, env_utils_1.getSuiteVersion)()}`,
                ...this.options.headers,
            },
        });
    }
    ping() {
        return this.getBlockHash(1);
    }
    send = (method, params = {}) => this.sendMessage({ method, params });
    getServerInfo() {
        return this.send('getInfo');
    }
    getBlockHash(block) {
        return this.send('getBlockHash', { height: block });
    }
    getBlock(block) {
        return this.send('getBlock', { id: `${block}` });
    }
    getBlockFilter(blockHash, filterParams) {
        return this.send('getBlockFilter', {
            blockHash,
            scriptType: 'taproot-noordinals',
            ...filterParams,
        });
    }
    getBlockFiltersBatch(bestKnownBlockHash, pageSize, filterParams) {
        return this.send('getBlockFiltersBatch', {
            bestKnownBlockHash,
            pageSize,
            scriptType: 'taproot-noordinals',
            ...filterParams,
        });
    }
    getMempoolFilters(fromTimestamp, filterParams) {
        return this.send('getMempoolFilters', {
            fromTimestamp,
            scriptType: 'taproot-noordinals',
            ...filterParams,
        });
    }
    getAccountInfo(payload) {
        return this.send('getAccountInfo', payload);
    }
    getAccountUtxo(descriptor) {
        return this.send('getAccountUtxo', { descriptor });
    }
    getTransaction(txid) {
        return this.send('getTransaction', { txid });
    }
    pushTransaction(hex, disableAlternativeRPC) {
        return this.send('sendTransaction', { hex, disableAlternativeRPC });
    }
    estimateFee(payload) {
        return this.send('estimateFee', payload);
    }
    rpcCall(payload) {
        return this.send('rpcCall', payload);
    }
    getCurrentFiatRates(payload) {
        return this.send('getCurrentFiatRates', payload);
    }
    getAccountBalanceHistory(payload) {
        return this.send('getBalanceHistory', payload);
    }
    getFiatRatesForTimestamps(payload) {
        return this.send('getFiatRatesForTimestamps', payload);
    }
    getFiatRatesTickersList(payload) {
        return this.send('getFiatRatesTickersList', payload);
    }
    subscribeAddresses(addresses) {
        this.removeSubscription('notification');
        this.addSubscription('notification', result => this.emit('notification', result));
        return this.send('subscribeAddresses', { addresses });
    }
    unsubscribeAddresses() {
        const index = this.removeSubscription('notification');
        return index >= 0 ? this.send('unsubscribeAddresses') : { subscribed: false };
    }
    subscribeBlock() {
        this.removeSubscription('block');
        this.addSubscription('block', result => this.emit('block', result));
        return this.send('subscribeNewBlock');
    }
    unsubscribeBlock() {
        const index = this.removeSubscription('block');
        return index >= 0 ? this.send('unsubscribeNewBlock') : { subscribed: false };
    }
    subscribeFiatRates(currency) {
        this.removeSubscription('fiatRates');
        this.addSubscription('fiatRates', result => this.emit('fiatRates', result));
        return this.send('subscribeFiatRates', { currency });
    }
    unsubscribeFiatRates() {
        const index = this.removeSubscription('fiatRates');
        return index >= 0 ? this.send('unsubscribeFiatRates') : { subscribed: false };
    }
    subscribeMempool() {
        this.removeSubscription('mempool');
        this.addSubscription('mempool', result => this.emit('mempool', result));
        return this.send('subscribeNewTransaction');
    }
    unsubscribeMempool() {
        const index = this.removeSubscription('mempool');
        return index >= 0 ? this.send('unsubscribeNewTransaction') : { subscribed: false };
    }
}
exports.BlockbookAPI = BlockbookAPI;
//# sourceMappingURL=websocket.js.map