"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Analytics = void 0;
const utils_1 = require("./utils");
class Analytics {
    enabled;
    useQueue = false;
    queue = new Array();
    version;
    app;
    instanceId;
    sessionId;
    commitId;
    url;
    callbacks;
    constructor({ version, app, useQueue = false }) {
        this.version = version;
        this.app = app;
        this.useQueue = useQueue;
    }
    init = (enabled, options) => {
        this.enabled = enabled;
        this.instanceId = options.instanceId || (0, utils_1.getRandomId)();
        this.sessionId = options.sessionId || (0, utils_1.getRandomId)();
        this.commitId = options.commitId;
        this.url = (0, utils_1.getUrl)(this.app, options.isDev, options.environment);
        this.callbacks = options.callbacks;
        if (this.enabled !== undefined) {
            this.flushQueue();
        }
    };
    enable = () => {
        this.enabled = true;
        this.callbacks?.onEnable?.();
        this.flushQueue();
    };
    flushQueue = () => {
        if (this.useQueue) {
            this.useQueue = false;
            this.queue.map(data => this.report(data));
            this.queue = [];
        }
    };
    disable = () => {
        this.enabled = false;
        this.callbacks?.onDisable?.();
        if (this.useQueue) {
            this.useQueue = false;
            this.queue = [];
        }
    };
    isEnabled = () => !!this.enabled;
    report = (data, config) => {
        if (!data.timestamp) {
            data.timestamp = Date.now().toString();
        }
        const isMissingFields = !this.url || !this.instanceId || !this.sessionId || !this.commitId || !this.version;
        if (!this.useQueue && isMissingFields) {
            const listOfMissingFields = `${!this.url ? 'url, ' : ''}` +
                `${!this.instanceId ? 'instanceId, ' : ''}` +
                `${!this.sessionId ? 'sessionId, ' : ''}` +
                `${!this.commitId ? 'commitId, ' : ''}` +
                `${!this.version ? 'version, ' : ''}`;
            console.error(`Unable to report ${data.type}. Analytics is not initialized! Missing: ${listOfMissingFields}`);
            return;
        }
        const { anonymize, force } = config ?? {};
        if (this.useQueue && this.enabled === undefined && !force) {
            this.queue.push(data);
        }
        if ((!this.enabled && !force) || isMissingFields) {
            return;
        }
        const qs = (0, utils_1.encodeDataToQueryString)(anonymize ? (0, utils_1.getRandomId)() : this.instanceId, anonymize ? (0, utils_1.getRandomId)() : this.sessionId, this.commitId, this.version, data);
        (0, utils_1.reportEvent)({
            type: data.type,
            url: `${this.url}?${qs}`,
            options: {
                method: 'GET',
                keepalive: true,
            },
            retry: true,
        });
    };
}
exports.Analytics = Analytics;
//# sourceMappingURL=analytics.js.map