import type { Address } from '@solana/addresses';
import type { Signature } from '@solana/keys';
import type { Base58EncodedBytes, Base58EncodedDataResponse, Base64EncodedDataResponse, Blockhash, Commitment, Lamports, Reward, Slot, TokenBalance, TransactionError, TransactionStatus, UnixTimestamp } from '@solana/rpc-types';
import type { TransactionVersion } from '@solana/transaction-messages';
type ReturnData = {
    /** A tuple whose first element is the bytes of the return data as a base64-encoded string. */
    data: Base64EncodedDataResponse;
    /** The address of the program that generated the return data */
    programId: Address;
};
type TransactionMetaBase = Readonly<{
    /** Number of compute units consumed by the transaction */
    computeUnitsConsumed?: bigint;
    /** Error if transaction failed, `null` if transaction succeeded. */
    err: TransactionError | null;
    /** The fee this transaction was charged, in {@link Lamports} */
    fee: Lamports;
    /**
     * String log messages or `null` if log message recording was not enabled during this
     * transaction
     */
    logMessages: readonly string[] | null;
    /** Account balances after the transaction was processed */
    postBalances: readonly Lamports[];
    /**
     * List of token balances from after the transaction was processed or omitted if token balance
     * recording was not yet enabled during this transaction
     */
    postTokenBalances?: readonly TokenBalance[];
    /** Account balances from before the transaction was processed */
    preBalances: readonly Lamports[];
    /**
     * List of token balances from before the transaction was processed or omitted if token balance
     * recording was not yet enabled during this transaction
     */
    preTokenBalances?: readonly TokenBalance[];
    /** The most-recent return data generated by an instruction in the transaction */
    returnData?: ReturnData;
    /**
     * Transaction-level rewards; currently only `"Rent"`, but other types may be added in the
     * future
     */
    rewards: readonly Reward[] | null;
    /** @deprecated */
    status: TransactionStatus;
}>;
type AddressTableLookup = Readonly<{
    /** The address of the address lookup table account. */
    accountKey: Address;
    /** Indexes of accounts in a lookup table to load as read-only. */
    readonlyIndexes: readonly number[];
    /** Indexes of accounts in a lookup table to load as writable. */
    writableIndexes: readonly number[];
}>;
type TransactionBase = Readonly<{
    message: {
        /**
         * For transactions whose lifetime is specified by a recent blockhash, this is that
         * blockhash, and for transactions whose lifetime is specified by a durable nonce, this is
         * the nonce value.
         */
        recentBlockhash: Blockhash;
    };
    /**
     * An ordered list of signatures belonging to the accounts required to sign this transaction.
     *
     * Each signature is an Ed25519 signature of the transaction message using the private key
     * associated with the account required to sign the transaction.
     */
    signatures: readonly Base58EncodedBytes[];
}>;
type InstructionWithStackHeight = Readonly<{
    /**
     * A number indicating the height at which this instruction was called with respect to the
     * bottom of the call stack denoted by `1` or `null`.
     *
     * For instance, an instruction explicitly declared in the transaction message will have a `1`
     * or `null` height, the first instruction that it calls using a cross-program invocation (CPI)
     * will have a height of 2, an instruction called by that instruction using a CPI will have a
     * depth of 3, and so on.
     */
    stackHeight: number;
}>;
type InstructionWithData = Readonly<{
    /** The input to the invoked program */
    data: Base58EncodedBytes;
}>;
type TransactionInstruction = InstructionWithData & Partial<InstructionWithStackHeight> & Readonly<{
    /**
     * An ordered list of indices that indicate which accounts in the transaction message's
     * accounts list are loaded by this instruction.
     */
    accounts: readonly number[];
    /**
     * The index of the address in the transaction message's accounts list associated with the
     * program to invoke.
     */
    programIdIndex: number;
}>;
type TransactionJson = Readonly<{
    message: {
        /** An ordered list of addresses belonging to the accounts loaded by this transaction */
        accountKeys: readonly Address[];
        header: {
            /**
             * The number of read-only accounts in the static accounts list that must sign this
             * transaction.
             *
             * Subtracting this number from `numRequiredSignatures` yields the index of the first
             * read-only signer account in the static accounts list.
             */
            numReadonlySignedAccounts: number;
            /**
             * The number of accounts in the static accounts list that are neither writable nor
             * signers.
             *
             * Adding this number to `numRequiredSignatures` yields the index of the first read-only
             * non-signer account in the static accounts list.
             */
            numReadonlyUnsignedAccounts: number;
            /**
             * The number of accounts in the static accounts list that must sign this transaction.
             *
             * Subtracting `numReadonlySignedAccounts` from this number yields the number of
             * writable signer accounts in the static accounts list. Writable signer accounts always
             * begin at index zero in the static accounts list.
             *
             * This number itself is the index of the first non-signer account in the static
             * accounts list.
             */
            numRequiredSignatures: number;
        };
        instructions: readonly TransactionInstruction[];
    };
}> & TransactionBase;
type PartiallyDecodedTransactionInstruction = InstructionWithData & Partial<InstructionWithStackHeight> & Readonly<{
    /** An ordered list of addresses belonging to the accounts loaded by this instruction */
    accounts: readonly Address[];
    /** The address of the program to invoke */
    programId: Address;
}>;
type ParsedTransactionInstruction = Partial<InstructionWithStackHeight> & Readonly<{
    /** The output of the program's instruction parser */
    parsed: {
        /** The instruction, as interpreted the program's instruction parser. */
        info?: object;
        /**
         * A label that indicates the type of the instruction, as determined by the program's
         * instruction parser.
         */
        type: string;
    };
    /** The name of the program. */
    program: string;
    /** The address of the program */
    programId: Address;
}>;
type ParsedAccount = Readonly<{
    /** The address of the account */
    pubkey: Address;
    /** Whether this account is required to sign the transaction that it's a part of */
    signer: boolean;
    /**
     * Indicates whether the account was statically declared in the transaction message or loaded
     * from an address lookup table.
     */
    source: 'lookupTable' | 'transaction';
    /** Whether this account must be loaded with a write-lock */
    writable: boolean;
}>;
type TransactionJsonParsed = Readonly<{
    message: {
        /**
         * An ordered list of parsed accounts belonging to the accounts loaded by this transaction
         */
        accountKeys: readonly ParsedAccount[];
        instructions: readonly (ParsedTransactionInstruction | PartiallyDecodedTransactionInstruction)[];
    };
}> & TransactionBase;
type GetTransactionCommonConfig<TMaxSupportedTransactionVersion> = Readonly<{
    /**
     * Fetch the transaction details as of the highest slot that has reached this level of
     * commitment.
     *
     * @defaultValue Whichever default is applied by the underlying {@link RpcApi} in use. For
     * example, when using an API created by a `createSolanaRpc*()` helper, the default commitment
     * is `"confirmed"` unless configured otherwise. Unmitigated by an API layer on the client, the
     * default commitment applied by the server is `"finalized"`.
     */
    commitment?: Commitment;
    /**
     * Determines how the transaction should be encoded in the response.
     *
     * - `'base58'` returns a tuple whose first element is the bytes of the wire transaction as a
     *   base58-encoded string.
     * - `'base64'` returns a tuple whose first element is the bytes of the wire transaction as a
     *   base64-encoded string.
     * - `'json'` returns structured {@link TransactionJson}
     * - `'jsonParsed'` returns structured {@link TransactionJson} which the server will attempt to
     *   further process using account parsers and parsers specific to the transaction instructions'
     *   owning program. Whenever an instruction parser is successful, instruction will consist of
     *   parsed data as JSON. Otherwise, the instruction will materialize as a list of accounts, a
     *   program address, and base64-encoded instruction data.
     */
    encoding: 'base58' | 'base64' | 'json' | 'jsonParsed';
    /**
     * The newest transaction version that the caller wants to receive in the response.
     *
     * When not supplied, only legacy (unversioned) transactions will be returned, and no `version`
     * property will be returned in the response.
     *
     * If a transaction with the supplied signature is found with a version higher than this, the
     * server will throw
     * {@link SolanaErrorCode.SOLANA_ERROR__JSON_RPC__SERVER_ERROR_UNSUPPORTED_TRANSACTION_VERSION | SOLANA_ERROR__JSON_RPC__SERVER_ERROR_UNSUPPORTED_TRANSACTION_VERSION}.
     */
    maxSupportedTransactionVersion?: TMaxSupportedTransactionVersion;
}>;
type GetTransactionApiResponseBase = Readonly<{
    /**
     * The estimated production time at which the transaction was processed. `null` if not
     * available.
     */
    blockTime: UnixTimestamp | null;
    /** The slot during which this transaction was processed */
    slot: Slot;
}>;
type TransactionMetaLoadedAddresses = Readonly<{
    /** Addresses loaded from lookup tables */
    loadedAddresses: {
        /** Ordered list of base-58 encoded addresses for read-only accounts */
        readonly: readonly Address[];
        /** Ordered list of base-58 encoded addresses for writable accounts */
        writable: readonly Address[];
    };
}>;
type InnerInstructions<TInstructionType> = Readonly<{
    /** The index of the instruction in the transaction */
    index: number;
    /** The instructions */
    instructions: readonly TInstructionType[];
}>;
type TransactionMetaInnerInstructionsNotParsed = Readonly<{
    /** A list of instructions called by programs via cross-program invocation (CPI) */
    innerInstructions?: readonly InnerInstructions<TransactionInstruction>[] | null;
}>;
type TransactionMetaInnerInstructionsParsed = Readonly<{
    /** A list of instructions called by programs via cross-program invocation (CPI) */
    innerInstructions?: readonly InnerInstructions<ParsedTransactionInstruction | PartiallyDecodedTransactionInstruction>[] | null;
}>;
type TransactionAddressTableLookups = Readonly<{
    message: Readonly<{
        /** A list of address tables and the accounts that this transaction loads from them */
        addressTableLookups: readonly AddressTableLookup[];
    }>;
}>;
export type GetTransactionApi = {
    /**
     * Returns details of the confirmed transaction identified by the given signature.
     *
     * @param signature A 64 byte Ed25519 signature, encoded as a base-58 string, that uniquely
     * identifies a transaction by virtue of being the first or only signature in its list of
     * signatures.
     *
     * Materializes the transaction as structured {@link TransactionJson} which the server will
     * attempt to further process using account parsers and parsers specific to the transaction
     * instructions' owning program. Whenever an instruction parser is successful, instruction will
     * consist of parsed data as JSON. Otherwise, the instruction will materialize as a list of
     * accounts, a program address, and base64-encoded instruction data.
     *
     * {@label parsed}
     * @see https://solana.com/docs/rpc/http/gettransaction
     */
    getTransaction<TMaxSupportedTransactionVersion extends TransactionVersion | void = void>(signature: Signature, config: GetTransactionCommonConfig<TMaxSupportedTransactionVersion> & Readonly<{
        encoding: 'jsonParsed';
    }>): (GetTransactionApiResponseBase & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : {
        version: TransactionVersion;
    }) & {
        meta: (TransactionMetaBase & TransactionMetaInnerInstructionsParsed) | null;
        transaction: TransactionJsonParsed & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : TransactionAddressTableLookups);
    }) | null;
    /**
     * Returns details of the confirmed transaction identified by the given signature.
     *
     * @param signature A 64 byte Ed25519 signature, encoded as a base-58 string, that uniquely
     * identifies a transaction by virtue of being the first or only signature in its list of
     * signatures.
     *
     * Materializes the transaction as a tuple whose first element is the bytes of the wire
     * transaction as a base64-encoded string.
     *
     * {@label base64}
     * @see https://solana.com/docs/rpc/http/gettransaction
     */
    getTransaction<TMaxSupportedTransactionVersion extends TransactionVersion | void = void>(signature: Signature, config: GetTransactionCommonConfig<TMaxSupportedTransactionVersion> & Readonly<{
        encoding: 'base64';
    }>): (GetTransactionApiResponseBase & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : {
        version: TransactionVersion;
    }) & {
        meta: (TransactionMetaBase & TransactionMetaInnerInstructionsNotParsed & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : TransactionMetaLoadedAddresses)) | null;
        transaction: Base64EncodedDataResponse;
    }) | null;
    /**
     * Returns details of the confirmed transaction identified by the given signature.
     *
     * @param signature A 64 byte Ed25519 signature, encoded as a base-58 string, that uniquely
     * identifies a transaction by virtue of being the first or only signature in its list of
     * signatures.
     *
     * Materializes the transaction as a tuple whose first element is the bytes of the wire
     * transaction as a base58-encoded string.
     *
     * {@label base58}
     * @see https://solana.com/docs/rpc/http/gettransaction
     */
    getTransaction<TMaxSupportedTransactionVersion extends TransactionVersion | void = void>(signature: Signature, config: GetTransactionCommonConfig<TMaxSupportedTransactionVersion> & Readonly<{
        encoding: 'base58';
    }>): (GetTransactionApiResponseBase & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : {
        version: TransactionVersion;
    }) & {
        meta: (TransactionMetaBase & TransactionMetaInnerInstructionsNotParsed & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : TransactionMetaLoadedAddresses)) | null;
        transaction: Base58EncodedDataResponse;
    }) | null;
    /**
     * Returns details of the confirmed transaction identified by the given signature.
     *
     * @param signature A 64 byte Ed25519 signature, encoded as a base-58 string, that uniquely
     * identifies a transaction by virtue of being the first or only signature in its list of
     * signatures.
     *
     * Materializes the transaction as structured {@link TransactionJson}.
     *
     * {@label json}
     * @see https://solana.com/docs/rpc/http/gettransaction
     */
    getTransaction<TMaxSupportedTransactionVersion extends TransactionVersion | void = void>(signature: Signature, config?: GetTransactionCommonConfig<TMaxSupportedTransactionVersion> & Readonly<{
        encoding?: 'json';
    }>): (GetTransactionApiResponseBase & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : {
        version: TransactionVersion;
    }) & {
        meta: (TransactionMetaBase & TransactionMetaInnerInstructionsNotParsed & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : TransactionMetaLoadedAddresses)) | null;
        transaction: TransactionJson & (TMaxSupportedTransactionVersion extends void ? Record<string, never> : TransactionAddressTableLookups);
    }) | null;
};
export {};
//# sourceMappingURL=getTransaction.d.ts.map