import { type GetAccountInfoApi, type Rpc, Address } from '@solana/kit';
/**
 * Convert amount to UiAmount for a mint with interest bearing extension without simulating a transaction
 * This implements the same logic as the CPI instruction available in /token/program-2022/src/extension/interest_bearing_mint/mod.rs
 * In general to calculate compounding interest over a period of time, the formula is:
 * A = P * e^(r * t) where
 * A = final amount after interest
 * P = principal amount (initial investment)
 * r = annual interest rate (as a decimal, e.g., 5% = 0.05)
 * t = time in years
 * e = mathematical constant (~2.718)
 *
 * In this case, we are calculating the total scale factor for the interest bearing extension which is the product of two exponential functions:
 * totalScale = e^(r1 * t1) * e^(r2 * t2)
 * where r1 and r2 are the interest rates before and after the last update, and t1 and t2 are the times in years between
 * the initialization timestamp and the last update timestamp, and between the last update timestamp and the current timestamp.
 *
 * @param amount                   Amount of tokens to be converted
 * @param decimals                 Number of decimals of the mint
 * @param currentTimestamp         Current timestamp in seconds
 * @param lastUpdateTimestamp      Last time the interest rate was updated in seconds
 * @param initializationTimestamp  Time the interest bearing extension was initialized in seconds
 * @param preUpdateAverageRate     Interest rate in basis points (1 basis point = 0.01%) before last update
 * @param currentRate              Current interest rate in basis points
 *
 * @return Amount scaled by accrued interest as a string with appropriate decimal places
 */
export declare function amountToUiAmountForInterestBearingMintWithoutSimulation(amount: bigint, decimals: number, currentTimestamp: number, lastUpdateTimestamp: number, initializationTimestamp: number, preUpdateAverageRate: number, currentRate: number): string;
/**
 * Convert an amount with interest back to the original amount without interest
 * This implements the same logic as the CPI instruction available in /token/program-2022/src/extension/interest_bearing_mint/mod.rs
 *
 * @param uiAmount                  UI Amount (principal plus continuously compounding interest) to be converted back to original principal
 * @param decimals                  Number of decimals for the mint
 * @param currentTimestamp          Current timestamp in seconds
 * @param lastUpdateTimestamp       Last time the interest rate was updated in seconds
 * @param initializationTimestamp   Time the interest bearing extension was initialized in seconds
 * @param preUpdateAverageRate      Interest rate in basis points (hundredths of a percent) before the last update
 * @param currentRate              Current interest rate in basis points
 *
 * In general to calculate the principal from the UI amount, the formula is:
 * P = A / (e^(r * t)) where
 * P = principal
 * A = UI amount
 * r = annual interest rate (as a decimal, e.g., 5% = 0.05)
 * t = time in years
 *
 * In this case, we are calculating the principal by dividing the UI amount by the total scale factor which is the product of two exponential functions:
 * totalScale = e^(r1 * t1) * e^(r2 * t2)
 * where r1 is the pre-update average rate, r2 is the current rate, t1 is the time in years between the initialization timestamp and the last update timestamp,
 * and t2 is the time in years between the last update timestamp and the current timestamp.
 * then to calculate the principal, we divide the UI amount by the total scale factor:
 * P = A / totalScale
 *
 * @return Original amount (principal) without interest
 */
export declare function uiAmountToAmountForInterestBearingMintWithoutSimulation(uiAmount: string, decimals: number, currentTimestamp: number, lastUpdateTimestamp: number, initializationTimestamp: number, preUpdateAverageRate: number, currentRate: number): bigint;
/**
 * Convert amount to UiAmount for a mint with scaled UI amount extension
 * @param amount     Amount of tokens to be converted
 * @param decimals   Number of decimals of the mint
 * @param multiplier Multiplier to scale the amount
 * @return Scaled UI amount as a string
 */
export declare function amountToUiAmountForScaledUiAmountMintWithoutSimulation(amount: bigint, decimals: number, multiplier: number): string;
/**
 * Convert a UI amount back to the raw amount for a mint with a scaled UI amount extension
 * @param uiAmount       UI Amount to be converted back to raw amount
 * @param decimals       Number of decimals for the mint
 * @param multiplier     Multiplier for the scaled UI amount
 *
 * @return Raw amount
 */
export declare function uiAmountToAmountForScaledUiAmountMintWithoutSimulation(uiAmount: string, decimals: number, multiplier: number): bigint;
/**
 * Convert amount to UiAmount for a mint without simulating a transaction
 * This implements the same logic as `process_amount_to_ui_amount` in
 * solana-labs/solana-program-library/token/program-2022/src/processor.rs
 * and `process_amount_to_ui_amount` in solana-labs/solana-program-library/token/program/src/processor.rs
 *
 * @param rpc     Rpc to use
 * @param mint    Mint to use for calculations
 * @param amount  Amount of tokens to be converted to Ui Amount
 *
 * @return Ui Amount generated
 */
export declare function amountToUiAmountForMintWithoutSimulation(rpc: Rpc<GetAccountInfoApi>, mint: Address, amount: bigint): Promise<string>;
/**
 * Convert a UI amount back to the raw amount
 *
 * @param rpc      Rpc to use
 * @param mint     Mint to use for calculations
 * @param uiAmount UI Amount to be converted back to raw amount
 *
 * @return Raw amount
 */
export declare function uiAmountToAmountForMintWithoutSimulation(rpc: Rpc<GetAccountInfoApi>, mint: Address, uiAmount: string): Promise<bigint>;
//# sourceMappingURL=amountToUiAmount.d.ts.map