import { ZCASH_ACTIVATION_HEIGHTS } from "../src/constants";
import { getDefaultVersions } from "../src/createTransaction";
import { openTransportReplayer, RecordStore } from "@ledgerhq/hw-transport-mocker";
import Btc from "../src/Btc";

describe("createTransaction", () => {
  describe("createTransaction", () => {
    test("zcash", async () => {
      const transport = await openTransportReplayer(
        RecordStore.fromString(`
        => b001000000
        <= 01055a6361736805322e332e3101029000
        => e04200001100000000050000800a27a726b4d0d6c201
        <= 9000
        => e0428000251d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f4010000006a
        <= 9000
        => e0428000324730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879
        <= 9000
        => e042800032cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff
        <= 9000
        => e04280000a75673c50650c00000000
        <= 9000
        => e04280000102
        <= 9000
        => e04280002250c30000000000001976a914de3542c396924ada3c5850225770f6dd3e2249b988ac
        <= 9000
        => e0428000223df2c202000000001976a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac
        <= 9000
        => e042800003000000
        <= 9000
        => e042800009000000000400000000
        <= 3200ed40989caa0731d6526fe4e03d49c54720be148f911924129e15d7ecf6e190829edb0000000050c3000000000000d53c396d2f5c98619000
        => e04200001100000000050000800a27a7265510e7c801
        <= 9000
        => e0428000258c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d11000000006b
        <= 9000
        => e04280003248304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb
        <= 9000
        => e042800032146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903
        <= 9000
        => e04280000ba42b5aea50327900000000
        <= 00009000
        => e04280000102
        <= 9000
        => e042800022404b4c00000000001976a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac
        <= 9000
        => e0428000220e532a00000000001976a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac
        <= 9000
        => e042800003000000
        <= 9000
        => e042800009000000000400000000
        <= 320071857203407d7ff3f5832db7c37973ac01569a832cd78a6dbd3627e4f9ee7458532200000000404b4c0000000000aa2ea74bcab3a20a9000

        => e04000000d03800000000000000000000000
        <= 9000
        => e04000000100
        <= 9000
        => e044000009010000005510e7c802
        <= 9000

        => e04480003b01383200ed40989caa0731d6526fe4e03d49c54720be148f911924129e15d7ecf6e190829edb0000000050c3000000000000d53c396d2f5c986119
        <= 9000
        => e04480001d76a914de3542c396924ada3c5850225770f6dd3e2249b988ac00000000
        <= 009000
        => e04480003b0138320071857203407d7ff3f5832db7c37973ac01569a832cd78a6dbd3627e4f9ee7458532200000000404b4c0000000000aa2ea74bcab3a20a19
        <= 00009000

        => e04480001d76a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac00000000
        <= 00009000

        => e04a0000320210270000000000001976a9140fef7d9e0afcc8e198ac049945b6499b6fe7aef288ac1a314700000000001976a914d83e71
        <= 00009000

        => e04a800013f7a39b28a617c7bcedbd925c2a621952b288ac
        <= 00009000

        => e04800001303800000000000000000000000000000000001
        <= 00009000


        => e044008009010000005510e7c802
        <= 9000

        => e04480803b01383200ed40989caa0731d6526fe4e03d49c54720be148f911924129e15d7ecf6e190829edb0000000050c3000000000000d53c396d2f5c986100
        <= 9000
        => e04480800400000000
        <= 9000

        => e04480803b0138320071857203407d7ff3f5832db7c37973ac01569a832cd78a6dbd3627e4f9ee7458532200000000404b4c0000000000aa2ea74bcab3a20a19
        <= 9000

        => e04480801d76a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac00000000
        <= 314402200e4952c0763be7e81f342e80042ea0858911aa3c460b90698195f014ddc8bf120220723b14f186ad7a66829982f00210e8ae6289284a146ed90b706d9e0e3313ae12019000

        => e04a0000320210270000000000001976a9140fef7d9e0afcc8e198ac049945b6499b6fe7aef288ac1a314700000000001976a914d83e71
        <= 9000
        => e04a800013f7a39b28a617c7bcedbd925c2a621952b288ac
        <= 9000

        => e04800000700000000000001
        <= 9000

        => e04800001f058000002c8000008580000001000000000000000a00000000000100000000
        <= 3045022100fd7bcbec98aaa7768774db3a289e07d951a727cb65b6fbfb48e23c745c3a7d16022046f2246519789af567d970b396c9c8c5c462e5a13709ea93481b512479bdf2b3019000
        `),
      );
      /*

splitTransaction 050000800a27a726b4d0d6c20000000000000000011d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f4010000006a4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c000000000250c30000000000001976a914de3542c396924ada3c5850225770f6dd3e2249b988ac3df2c202000000001976a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac000000:
TX version 05000080 locktime 00000000 timestamp  nVersionGroupId 0a27a726 nExpiryHeight 00000000 extraData 
input 0: prevout 1d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f401000000 script 4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c sequence 00000000
output 0: amount 50c3000000000000 script 76a914de3542c396924ada3c5850225770f6dd3e2249b988ac
output 1: amount 3df2c20200000000 script 76a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac
      */
      // This test covers the old bitcoin Nano app 1.6 API, before the breaking changes that occurred in v2.1.0 of the app
      const btc = new Btc({ transport, currency: "zcash" });
      const tx1 = btc.splitTransaction(
        "050000800a27a726b4d0d6c20000000000000000011d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f4010000006a4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c000000000250c30000000000001976a914de3542c396924ada3c5850225770f6dd3e2249b988ac3df2c202000000001976a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac000000",
        true,
        true,
        ["zcash", "sapling"],
      );
      /*
       splitTransaction 050000800a27a7265510e7c80000000000000000018c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d11000000006b48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea5032790000000002404b4c00000000001976a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac0e532a00000000001976a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac000000:
TX version 05000080 locktime 00000000 timestamp  nVersionGroupId 0a27a726 nExpiryHeight 00000000 extraData 
input 0: prevout 8c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d1100000000 script 48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea503279 sequence 00000000
output 0: amount 404b4c0000000000 script 76a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac
output 1: amount 0e532a0000000000 script 76a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac
      */
      const tx2 = btc.splitTransaction(
        "050000800a27a7265510e7c80000000000000000018c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d11000000006b48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea5032790000000002404b4c00000000001976a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac0e532a00000000001976a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac000000",
        true,
        true,
        ["zcash", "sapling"],
      );
      const result = await btc.createPaymentTransaction({
        inputs: [
          [tx1, 0, null, 0, 1806010],
          [tx2, 0, null, 0, 2733284],
        ],
        associatedKeysets: ["0'/0/0"],
        changePath: undefined,
        blockHeight: 2812009,
        sigHashType: 1,
        outputScriptHex:
          "0210270000000000001976a9140fef7d9e0afcc8e198ac049945b6499b6fe7aef288ac1a314700000000001976a914d83e71f7a39b28a617c7bcedbd925c2a621952b288ac",
        additionals: ["zcash", "sapling"],
      });
      expect(result).toEqual(
        "010000005510e7c8000000000000000002989caa0731d6526fe4e03d49c54720be148f911924129e15d7ecf6e190829edb00000000050230000102000000007203407d7ff3f5832db7c37973ac01569a832cd78a6dbd3627e4f9ee745853220000000003000102000000000210270000000000001976a9140fef7d9e0afcc8e198ac049945b6499b6fe7aef288ac1a314700000000001976a914d83e71f7a39b28a617c7bcedbd925c2a621952b288ac000000",
      );
    });

    test("zcash Sapling deshielded input", async () => {
      const transport = await openTransportReplayer(
        RecordStore.fromString(`
        => b001000000
        <= 01055a6361736805322e342e3201029000
        => e04200001100000001050000800a27a7265510e7c801
        <= 9000
        => e042800025e1360c957489515ddfb5c564962e2c8cb2dc3c651c4a219e25e0b5e569f49d33000000006b
        <= 9000
        => e0428000324830450221008844cfb8d9983226f74cdd20cb63ee282360374def5de88d093df7f340775d65022072673cea8cd2092484c1
        <= 9000
        => e0428000321c6e8c35ab765a9501024a96265bdd3b80d0c46f9190012102495e50ff5127b9b74083bad438208c7a39ddd83301cd04e40b
        <= 9000
        => e04280000bff5556d3351ab300000000
        <= 9000
        => e04280000102
        <= 9000
        => e042800022a0860100000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac
        <= 9000
        => e042800022cedb0e00000000001976a9142495eecd3d7ea979d2066da533f45956a3a6b5c888ac
        <= 9000
        => e042800003000000
        <= 9000
        => e042800009000000000400000000
        <= 320039a3a9a27d42321c7ace2884a65a343abb9755f3eba881e53834bdb4a3fed4432a1301000000cedb0e0000000000a725fe2e190a317a9000
        => e04200001100000000050000800a27a7265510e7c800
        <= 9000
        => e04280000101
        <= 9000
        => e04280002240951600000000001976a9142767cba450ee8c78f42c151c2b0a89673686fb0988ac
        <= 9000
        => e042800003030200
        <= 9000
        => e04280002860e31600000000003355b4c7f891cc2ff3f6ea75caf6885eba72c75e94fee5a076afe0394ea1966d
        <= 9000
        => e0428000608b8d5de736a24745b0854cbe0e191a5f39bdae2bc9ae3c43be411f2b549d20738f8c66cac5f3470abeac19d39940f7edeed994add401a209a6c816a2bf259de7fbafb44e3ba4b1aa861a188d273d0fef368b9a2ef9f7f7b0d880abec65a72723
        <= 9000
        => e0428000606471cef68f92c7761bd75fdb6c584193760a16dab8cd2f510753c651b7a8e89ca7f67e513641b7ec3dd9a72785719c7c306e703f2589e8562c659d5d34b170839658294c84eb1a2efd77923a790e68b6268948cb603f851afe60d3b96f738224
        <= 9000
        => e042800060eb9a9088dac9a59155320936e61a0143e22822b6afd53c20f029994d0784d747f8374eeda6f60d31ab1a7462c9d3b92578b01a8e79994103026687713680bb972337d979d4318e5bc2f6d7035e7635b4d5526e1b04ba4089cc330c7348732e9d
        <= 9000
        => e04280007455888f46bee4da6970177672bffc486e4543fe5306104029cd26649e1d60581930885b3146de5204a21819f6665c26e449e23cca619fe58097df1bbc8197842f2aba5bef1fd1d65acec1f26eb8e8b7658b87232aaad3eca72ec2f066ac647401614d4844c0892e4cc012f8b0e2983c39672c80ec
        <= 9000
        => e042800074a96342d24ad5ec914387ec50c31d15b64c9a125261dc0e498caebe4187a2c51e70e63c0f4f431f336592b0a17bc64e455fc6ae20147ee34d1d13fd6b11bc52c78def9e7d58ec2967539f8bdf223a56b6fa2c1595f207945e7dda0612b94a5f2dda4d797e000effdb5f2e712c714b290de19da38e
        <= 9000
        => e042800080ee532afd277b2cce50da918b54a8d4dc0746b9ecf756987aa75914cd9ddf00c890fa02afd62d229f3e039fffbb52693df89de9431975551afd5fd5c9b80a76f769357a7a5c271ec15a2a4dd6ce6de3608139d332db94fb4c71ac13c2226cc81d2271598110aabfb28b884cedb27253aa0df15014a3dc3fbfc859b56ae0657dea
        <= 9000
        => e0428000803d4df68cd5ec183422974c4df25290ef7618b9c6b8e97e6dc7017d1d2db464187f7dd4069c27c3b58b2bc60da03eebb015180bf94ee13aaf2c5dee3aae42e6184ede66141eb0cc26fa8f96fb8ea6f825dc582d072a1534364eea3f9df423a55795b3bd5814d48eca709f8b87c1660ade69b2d4a6ff1dbda23da6c8a0d7f892a0
        <= 9000
        => e042800080663f3ccd63d0063ac80a8d2652c4c422c817e39578570e4aee360a2429bc5476bb23896f652283553b315e28c4c5c94195b823c9c7b24becfe21af74295bfe633bd646c743d1b0d5868e3a71b833316bac88f3656dd2ecafb066158a05ab008dbf923e33786be07446b41613253f75f31f1a4b2cb684be69b69ac20046522b60
        <= 9000
        => e042800080d0193490a849c1ae4faf89740b6c331b09eb8f2c27cce750ef507114e6df584fefd7e2d4df1affe7d3480559b5149114803b8d99570833bdaa80845332cf9a7c17d6d86c32f4af265a1c58251b47ba822527cb89ab456112ca484acd3213d895f23214e5e870b0d3d15632a613812fb755adf420eade22f4b852a752f4619e07
        <= 9000
        => e04280008036a5b044ff66bf7b95a26e6f12b85f111ba48a3f2114867049700bab9921df438a261d9e67e850c96b4aab9c273f9f64a00df0266e6de34ff06d542a8d7525345fd88d13bd02c472e97f4ce21eab32112e26c462c4f8856e1e84d126faf81963bd1796dbbed3fb37237143b9b38f3e2b4e398b29c386c1a5e3ece72b3302c042
        <= 9000
        => e0428000809fb583512da35e7d18bb0fe4a8d37c16c41e62c377112f0c05eebd8d2702b29c10c4695ed28dd517d8a9866991d3265e3b77340ff103c468048484e636dc92d1228aa1e99c91030e2a8a7727fff53fc9bb67055f8c3b188bb3d945006d401645cec07c2e646f6ceeb05640e3bc05b83aa7f2f6de1680aec5e72c2d782163fd01
        <= 9000
        => e0428000806f778f154226cf309c90ba2529f4111bfa360eae8d3d75319fc5e69dbfca29700ac39fb1ffc56ae22c2d0e1a9df3b9376c01944728377cec74286ab0b3c94795ed2eb65d117521c9d0928949c0cc4c782059b257bd547756077d76362754206d884a27122f477888cf950981a19673931c633fb1d13e52e104d93002b408979b
        <= 9000
        => e042800080b5f58dc29bc36394ab96874f12d8d4eccb36fd2cb20151604d106af87613e5ca9ecf46825ba88ac548ad0bd68494c912a022a38e6a903fe2db62e4f1f798b343b4b5f2843f0ca683c91b04f421050e6a9479f5280ed98766451873575be596e25d437ab7dbc347cc015da1c88b7336a1d7956e5d4632496143c14eabe023f26f
        <= 9000
        => e04280008010a595cb47f9aac32e7047109c56a8d49c601d79ec4a6776e8f5f6a3021ce86c8e05fe69ca0d99a4d96a6285ac2079a57c9d40b91b09f832bb47cf7f027a8720214b40cff41f6d3671ccfe63b6f312ee6709b3657cdbdfc99bd276033dd22d255971f256750ce2661f93596373a5664f2a4657b50cb544aac8d36bb9d9d17ddb
        <= 9000
        => e042800080ec224bd72791daed879010322da1614878dacca61370576f0d5a12566f23db806c6ee3e71bb929f9d217a26d692cc4f5a18c400d2de6ce7208190d50be41f5478d076ff42bfb0e4e7e8021f1618f8700966e1e73041683980bd46768647d5ca2a3c706cb470f77fcd1547f8d7e25234b77332220d523813900d562f7f5dbcbb4
        <= 9000
        => e0428000090000000004d87c2d00
        <= 32001a8635f812ce32a94bbd7679a7f4a71c08b9fb263462352157a5c87fa569ad1d7814000000004095160000000000d2a01509f84bc7b69000
        => e040000015058000002c80000085800000000000000100000000
        <= 4104a374dc40ad9e6aeca5f10aa1fec30c00335016ee628679b49965360a79cdc9bcc4b26e3efed19cf36c7eb8a8c07193c2c240c538b984176327dd8fbfa471fe572374314d4433776b706a4351775039586358345345503144555a767a7745344c4e506d660cd558bd1f787b8e13c5786f298ca36015c2dcf28d791256f21da29b05352d4b9000
        => e040000015058000002c80000085800000000000000000000001
        <= 4104c81f0705e847a0e765dd1b829d110aa3b21a4dc77f9e4626f03b40b8547c8fb19c572b6b25deb01064df2924af3e000bdee66b7bce26cd8d2b30316a078c2aca2374314d5478686571425432624b7a4d384d67794878576f543159356d58776a4d53376e9d4811b3d3b57c541271f400cbe38b33f5a121272a919b2bd00e250175c48cfa9000
        => e04400050d050000800a27a7265510e7c802
        <= 9000
        => e04480053b0138320039a3a9a27d42321c7ace2884a65a343abb9755f3eba881e53834bdb4a3fed4432a1301000000cedb0e0000000000a725fe2e190a317a19
        <= 9000
        => e04480051d76a9142495eecd3d7ea979d2066da533f45956a3a6b5c888ac00000000
        <= 9000
        => e04480053b013832001a8635f812ce32a94bbd7679a7f4a71c08b9fb263462352157a5c87fa569ad1d7814000000004095160000000000d2a01509f84bc7b619
        <= 9000
        => e04480051d76a9142767cba450ee8c78f42c151c2b0a89673686fb0988ac00000000
        <= 9000
        => e04aff0015058000002c80000085800000000000000100000001
        <= 9000
        => e04a0000320280841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac28360100000000001976a9141ed023
        <= 009000
        => e04a800013c5e3414784b975c5ee2ec5378ed7c4fcd488ac
        <= 00009000
        => e04800000b0000000000000100000000
        <= 9000
        => e04400800d050000800a27a7265510e7c801
        <= 9000
        => e04480803b0138320039a3a9a27d42321c7ace2884a65a343abb9755f3eba881e53834bdb4a3fed4432a1301000000cedb0e0000000000a725fe2e190a317a19
        <= 9000
        => e04480801d76a9142495eecd3d7ea979d2066da533f45956a3a6b5c888ac00000000
        <= 9000
        => e04800001f058000002c8000008580000000000000010000000000000000000100000000
        <= 3145022100a75b5a5a700c148f2eeb7e900304f88b81a54b34a2f9aba4b37fd2ba016e9b6f02203bfeb391bccef75620ff3f98b25b6503aba1b397108ab11691e0d268f577760e019000
        => e04400800d050000800a27a7265510e7c801
        <= 9000
        => e04480803b013832001a8635f812ce32a94bbd7679a7f4a71c08b9fb263462352157a5c87fa569ad1d7814000000004095160000000000d2a01509f84bc7b619
        <= 9000
        => e04480801d76a9142767cba450ee8c78f42c151c2b0a89673686fb0988ac00000000
        <= 9000
        => e04800001f058000002c8000008580000000000000000000000100000000000100000000
        <= 3145022100886a6c227090e2bd3a2b511909213209117d1d3386773dc27dd68ec41dd8fbb5022019ccb19c48e705bb0887da8a7f3b32d0e08e239d6aeb363284a841fe0fa2adc9019000
        `),
      );

      const btc = new Btc({ transport, currency: "zcash" });
      // sapling transperent
      const tx1 = btc.splitTransaction(
        "050000800a27a7265510e7c8000000000000000001e1360c957489515ddfb5c564962e2c8cb2dc3c651c4a219e25e0b5e569f49d33000000006b4830450221008844cfb8d9983226f74cdd20cb63ee282360374def5de88d093df7f340775d65022072673cea8cd2092484c11c6e8c35ab765a9501024a96265bdd3b80d0c46f9190012102495e50ff5127b9b74083bad438208c7a39ddd83301cd04e40bff5556d3351ab30000000002a0860100000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388accedb0e00000000001976a9142495eecd3d7ea979d2066da533f45956a3a6b5c888ac000000",
        true,
        true,
        ["zcash", "sapling"],
      );

      // sapling deshielded
      const tx2 = btc.splitTransaction(
        "050000800a27a7265510e7c800000000d87c2d00000140951600000000001976a9142767cba450ee8c78f42c151c2b0a89673686fb0988ac038b8d5de736a24745b0854cbe0e191a5f39bdae2bc9ae3c43be411f2b549d20738f8c66cac5f3470abeac19d39940f7edeed994add401a209a6c816a2bf259de7fbafb44e3ba4b1aa861a188d273d0fef368b9a2ef9f7f7b0d880abec65a727236471cef68f92c7761bd75fdb6c584193760a16dab8cd2f510753c651b7a8e89ca7f67e513641b7ec3dd9a72785719c7c306e703f2589e8562c659d5d34b170839658294c84eb1a2efd77923a790e68b6268948cb603f851afe60d3b96f738224eb9a9088dac9a59155320936e61a0143e22822b6afd53c20f029994d0784d747f8374eeda6f60d31ab1a7462c9d3b92578b01a8e79994103026687713680bb972337d979d4318e5bc2f6d7035e7635b4d5526e1b04ba4089cc330c7348732e9d0210a595cb47f9aac32e7047109c56a8d49c601d79ec4a6776e8f5f6a3021ce86c55888f46bee4da6970177672bffc486e4543fe5306104029cd26649e1d60581930885b3146de5204a21819f6665c26e449e23cca619fe58097df1bbc8197842f2aba5bef1fd1d65acec1f26eb8e8b7658b87232aaad3eca72ec2f066ac647401614d4844c0892e4cc012f8b0e2983c39672c80ecee532afd277b2cce50da918b54a8d4dc0746b9ecf756987aa75914cd9ddf00c890fa02afd62d229f3e039fffbb52693df89de9431975551afd5fd5c9b80a76f769357a7a5c271ec15a2a4dd6ce6de3608139d332db94fb4c71ac13c2226cc81d2271598110aabfb28b884cedb27253aa0df15014a3dc3fbfc859b56ae0657dea3d4df68cd5ec183422974c4df25290ef7618b9c6b8e97e6dc7017d1d2db464187f7dd4069c27c3b58b2bc60da03eebb015180bf94ee13aaf2c5dee3aae42e6184ede66141eb0cc26fa8f96fb8ea6f825dc582d072a1534364eea3f9df423a55795b3bd5814d48eca709f8b87c1660ade69b2d4a6ff1dbda23da6c8a0d7f892a0663f3ccd63d0063ac80a8d2652c4c422c817e39578570e4aee360a2429bc5476bb23896f652283553b315e28c4c5c94195b823c9c7b24becfe21af74295bfe633bd646c743d1b0d5868e3a71b833316bac88f3656dd2ecafb066158a05ab008dbf923e33786be07446b41613253f75f31f1a4b2cb684be69b69ac20046522b60d0193490a849c1ae4faf89740b6c331b09eb8f2c27cce750ef507114e6df584fefd7e2d4df1affe7d3480559b5149114803b8d99570833bdaa80845332cf9a7c17d6d86c32f4af265a1c58251b47ba822527cb89ab456112ca484acd3213d895f23214e5e870b0d3d15632a613812fb755adf420eade22f4b852a752f4619e078e05fe69ca0d99a4d96a6285ac2079a57c9d40b91b09f832bb47cf7f027a8720214b40cff41f6d3671ccfe63b6f312ee6709b3657cdbdfc99bd276033dd22d255971f256750ce2661f93596373a5664f2a4657b50cb544aac8d36bb9d9d17ddbec224bd72791daed879010322da1614878dacca61370576f0d5a12566f23db80a96342d24ad5ec914387ec50c31d15b64c9a125261dc0e498caebe4187a2c51e70e63c0f4f431f336592b0a17bc64e455fc6ae20147ee34d1d13fd6b11bc52c78def9e7d58ec2967539f8bdf223a56b6fa2c1595f207945e7dda0612b94a5f2dda4d797e000effdb5f2e712c714b290de19da38e36a5b044ff66bf7b95a26e6f12b85f111ba48a3f2114867049700bab9921df438a261d9e67e850c96b4aab9c273f9f64a00df0266e6de34ff06d542a8d7525345fd88d13bd02c472e97f4ce21eab32112e26c462c4f8856e1e84d126faf81963bd1796dbbed3fb37237143b9b38f3e2b4e398b29c386c1a5e3ece72b3302c0429fb583512da35e7d18bb0fe4a8d37c16c41e62c377112f0c05eebd8d2702b29c10c4695ed28dd517d8a9866991d3265e3b77340ff103c468048484e636dc92d1228aa1e99c91030e2a8a7727fff53fc9bb67055f8c3b188bb3d945006d401645cec07c2e646f6ceeb05640e3bc05b83aa7f2f6de1680aec5e72c2d782163fd016f778f154226cf309c90ba2529f4111bfa360eae8d3d75319fc5e69dbfca29700ac39fb1ffc56ae22c2d0e1a9df3b9376c01944728377cec74286ab0b3c94795ed2eb65d117521c9d0928949c0cc4c782059b257bd547756077d76362754206d884a27122f477888cf950981a19673931c633fb1d13e52e104d93002b408979bb5f58dc29bc36394ab96874f12d8d4eccb36fd2cb20151604d106af87613e5ca9ecf46825ba88ac548ad0bd68494c912a022a38e6a903fe2db62e4f1f798b343b4b5f2843f0ca683c91b04f421050e6a9479f5280ed98766451873575be596e25d437ab7dbc347cc015da1c88b7336a1d7956e5d4632496143c14eabe023f26f6c6ee3e71bb929f9d217a26d692cc4f5a18c400d2de6ce7208190d50be41f5478d076ff42bfb0e4e7e8021f1618f8700966e1e73041683980bd46768647d5ca2a3c706cb470f77fcd1547f8d7e25234b77332220d523813900d562f7f5dbcbb460e31600000000003355b4c7f891cc2ff3f6ea75caf6885eba72c75e94fee5a076afe0394ea1966d90015aa410991258ac7786d256ee0fa76584aa57fd45ba1055f8a4e19cfbbfbfa4070f3b7c5bb68a2437b1310e18218e854cabc7bac92d97a974a471e0de550d167e82e2363314bb68c1a3a4e2877aa7c236817954ec51ed196a2bb97ebf16d70101d0e56f49206f01c407eb55674a6eab5a0731aec169f3f641d515576ec525200b04c93281e4966d0d82122e70ccc18d84fad25fc5d5318a5ed8e71c738df0074db07753ed20281a268d4557b5890c02b1f4f050e5dc099258810f3d17c690a47425f08f56a1c59f7556781094d156a88469432a8893044e956501a07da27bdf5b3941a67e557067f046d351a78670b666c952e9fd09cfc4712aad3de487814f1ab9dd71649b8a45184f97929965ec359147ae763815eb7c43675d68b274e107730156a0268673a6fa58d07abe6fae9d34ba0ffa31d0bb78dd8844b58a96d743e5448e26d4fcbcba6b6a332774d6c78b1bb0721087ebcc2423940cda12a955187f68048f5e362542b2f6129fbe892afed1de8f6ff02642cf8f9547629a847cb45ff0353dc8e608140d1554ea05694d88f76f4f8dab1765f31681b2887f8b2582c1cea613813870cb75ca7dbdd3647ca9fa6990597debf2934f62490bb2440ff0b7d72af82155927e11e94e22ff836678cc8eb651db28d390c3832b3f176f471916b51ef0b05ea45635170281fb839dc80dbd0d3280166881782c0dc31a173bae8d74bc8aad1b02918c49233c810ace95cadf7d5cf1dcf89ceed4732a1a8a6011a67ecb10adbb2fc42f317fc7c663f5b828c69037a3efd80e4278b16fbf696584e251a39335a55f8ddbb66d1eebbd3e52378192e0f3342c5ea29e613db467714bf3b00eaafbac5762e9a96b44294e7052fc1245f6e4fd17ebbb7c4d07c38102349d8f45bfc3851c28f528ba5dad01cf8bd413e8b15bec20106289f55b1b7fa6aa950373da9a291d94823258ca8561b4559883c6bc16d804f284b72cb7542102d329eae6cad25195ad2aa39ec1f59954a667ccf5d7a2213a6115eef6000159bbb7a0c239a40b78caacba4256a60689ec474fc59c4ef8d4220f9df62bb4c78a0b8c12fef8b77902b03c60e3625f97b8eeedac8baea41a18ca9738f341bffd55a2574169302d3e647bfa0a404437d6339ba65a9f892892d94500d3d2c0e27de824941e7ead0d8d08c5430b2384bb07f3f795c83b16ca28012002588b7c8de5a8c808b716cf3e672ff08d47a872eca41f81565c24835a5857ed54e4e97c290dcefa3f21702bbe9300db0e86661462a055e59624f15ad9046f6f456372e16a9dc811724b71a9b6c5e6c5b79ff2cc5f6567a49e71542609c4b9202e465941abe935f485ed8e54cf70d047f5373e3b91d5163c290d99dec3c60b0512e318da5c47fedeb7cd12a61b8dc2523d1057f77f70f0fb95779c2ba1a84df1c8403782807b1b477419f2f8b2e7ee510b055ce616639d66fa401ee9c23f9a8ab2a394cc24f25e8806dca5bf31e35ebc00b8a28c2c1e56e1623357e2cdce2f2af08c768a3f78f5d23bd3e43163e6174edbbdfbe5a7cf4433b8b319877916e31fba0855aa6a41410d27a7af9d009d3f80ae6d685801e71c64aff4792ff9eab3c77e4772883c139a955b9ea810ae2be12380cd3aa9473c6e2b6f95ef2250189320089c1d2fb696154c09017d9377bd1030ad44f0ab46256da4bffb13552f37a924928a23257829dd0300",
        true,
        true,
        ["zcash", "sapling"],
      );

      const result = await btc.createPaymentTransaction({
        inputs: [
          [tx1, 1, null, 0, 2980249],
          [tx2, 0, null, 0, 2981031],
        ],
        associatedKeysets: ["44'/133'/0'/1/0", "44'/133'/0'/0/1"],
        changePath: "44'/133'/0'/1/1",
        blockHeight: 2984328,
        sigHashType: 1,
        outputScriptHex:
          "0280841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac28360100000000001976a9141ed023c5e3414784b975c5ee2ec5378ed7c4fcd488ac",
        additionals: ["zcash", "sapling"],
        expiryHeight: Buffer.from([0, 0, 0, 0]),
      });

      expect(result).toEqual(
        "050000800a27a7265510e7c8000000000000000002a9a27d42321c7ace2884a65a343abb9755f3eba881e53834bdb4a3fed4432a13010000006b483045022100a75b5a5a700c148f2eeb7e900304f88b81a54b34a2f9aba4b37fd2ba016e9b6f02203bfeb391bccef75620ff3f98b25b6503aba1b397108ab11691e0d268f577760e012103a374dc40ad9e6aeca5f10aa1fec30c00335016ee628679b49965360a79cdc9bc0000000035f812ce32a94bbd7679a7f4a71c08b9fb263462352157a5c87fa569ad1d7814000000006b483045022100886a6c227090e2bd3a2b511909213209117d1d3386773dc27dd68ec41dd8fbb5022019ccb19c48e705bb0887da8a7f3b32d0e08e239d6aeb363284a841fe0fa2adc9012102c81f0705e847a0e765dd1b829d110aa3b21a4dc77f9e4626f03b40b8547c8fb1000000000280841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac28360100000000001976a9141ed023c5e3414784b975c5ee2ec5378ed7c4fcd488ac000000",
      );
    });

    test("zcash Orchard deshielded input", async () => {
      const transport = await openTransportReplayer(
        RecordStore.fromString(`
=> b001000000
<= 01055a6361736805322e342e3201029000
=> e04200001100000000050000800a27a7265510e7c800
<= 9000
=> e04280000101
<= 9000
=> e042800022488e1000000000001976a914e58749ee655c0e39ae3ce063a33fb9edc86d23dd88ac
<= 9000
=> e042800003000002
<= 9000
=> e042800094310746abef4d8a77fdd55fc303a306f3953f794a95bb8f14d9c94b59d63f792ba9856f1ec074084e8a91d7008cbed27a00a4e4a816f66e2772c5e04b3f876c08671af32842982bcc7f344410d8eb50a2b166e7b00f6a4e7ff64cb368d84fa789ff76ee84fae0912e178422fff4dc220610589d344a461acaf9487f231d5091575a7deec69f9cdccec92a537b5c996e60a6c16f85
<= 9000
=> e042800094e16a84f5840e498b935bce15d2900a592d33446c14e9b9bfd8d05b4e24d0142ad3bec60fed5d37c0d46a2384516ecb02520a8dc06f83f747504078569c55a0290ea78f4a3b765cd00d69a3f60ad695b42d652b285caf3aea639aadb694e24f374ee493445fdb428c95de6016203993655595c7785d3a0c912bfd6af29601429fb773514ed6eb27f376f56384f52c65f9090fa462
<= 9000
=> e04280008002f1e4c873bc88dd2eb2036ba2bfa9cc7b515d721844ce251df61955f78c4e84bff6fdc62404cfbd3bd360b5020bae564e47594ef6cbb4687065749579ab477623d2752f38f30eda3d2231a513b0238b4f45101b2378b50acf8877b182bce2b994ff166cb95f6d319e2b50a0eb432ef7fc556a0e66752fe8ebb0b4184c38820c
<= 9000
=> e042800080733fccbeee6e78f3b72f8d61838721f689c4a6a6b1cf8f0fb6bfa1bbdf860ac1631567de443b9cdde8ccdb458b12413f1a1cc90966b40d34249e60eb49d24d9476c89e19721a2d32ce4dbf125a6ffc0a3a9bf3423328ae97e7a2227a08da8e4d1d5031e9d2a6ee04c8d800b1ce42e223d6a16afb88657a46663d7256288e6949
<= 9000
=> e042800080b751b4df845a9f544a2366e7976f9ff491e4779dc41431393c7ad99747b2470f24302b28df856506a16ec7b88e77e2c37e2a7a7580d0d6122c63e2f34b63de3011526e6ce4b16bfe43eed07a5364f230d0689c7cc97a2945a7a7e8e8535fa147e665e20bce1d8370e96be7629f34aa72d2a9031c90faf92b8383ed3c51065378
<= 9000
=> e042800080b01e143fa64a2eb8322fcf01207c740437b9a9470a67815e541ee1cc3da71349c9092bc296e2d20da040120345717b18df4bb6559caca87349389d9b5567267c640f8289f6caf7c701dc11d0decd9996c47227cc6418c3b13bedd9f4086c7b874b96f0e15bbd6793ba29bd74a4f969f75697e1a817ddba599b2878885724727a
<= 9000
=> e0428000807cf17b3c4c577685b1aa7f19ddb6c0596f48447d7aa8253d8faca7bf9bfdb76217500e6a9f1fb794c2c5dba36a4d7efafdbc74d28b7b6f36234312a1e3a99bb412bbe14ebc9e94287994e7846e14ac765809eef7e87289665d0089ed4f3399d547f9156ed34a0598852d52a4daff4096ebd3998e748de36a262e718514577baf
<= 9000
=> e0428000807ad16d6be978d382ff83fceefd37cc18940c9f9d7f88e1797023fdd4a017721e876088c86121667780e9a32ed92c749bb018e06157b1c7dcd81d7d35de08e3886338b1c6309b15b099c98281ebb79662526753596f9c5784ef6306c04cadf30f3fa650c2ce49bea66bbbe67580585f7e21cc4d04a16b0f2a3729881307020ee3
<= 9000
=> e0428000805efac01ae6a79a209dfdc911d4119a57ee9184ac38ccba8475d39c40010d2ae5fbb61fa0b0846cdff83afb9f35e63331743fe8b7098da2e58e0ce97593e9cab30bf362537a49ebe30469fc08aad3e75c9ccdd067d73afa042656a1395d15197f98d3ed1034e0552d852202e4eb34b89b801b10d0402a9d698897e6f9d650f685
<= 9000
=> e042800080064794d16c7f93755da6db19cd37de31a41746bbb43199e832f0e7f5f675470e9df5d3afc38716873cb0ade0c4b0e1b5e6275d3adf97e260b10fb74fc379407ccb329c4e06645d80c4ab188a7d952a3e631dc16f1ec82e2c5cafd39005006737d047c276aa7ef03325d4405c9e2bd86ad616d9d343dc8cf2b4543a437be4de0d
<= 9000
=> e0428000a0bd730701f6e2e1feffc2021707919aaced34d5e43c13a5010695cd9d333c19bf9e6f9133be659f9cc8b53fc1a26f2e1a9e4e132dd0e78df56be5d06688f103b232624758df64f86e09d78667e26b4adad05f7d8a539a15e7e19eff71fd29a71c6f58bf920456d8a1348f1225fc1d0b1abbc25bce80a645839d272389faf7a0d4367dcff461a64922b85186de7534bdac9739c57bd8dccd9c0b543a2c1cb1a2ed
<= 9000
=> e0428000a0bdbcc7541c03dab9665eda1bc07682ce3dcf4f2293e1bba1aabdbfeb2464739a9027b8a15f11e25d9efd7886e96a214344bbd44c046b223bd1ee5f0b636c5e07544dd47d91d7e218a16ee75597137e5f4d95e868a9a0939cc41f3d36bae7c2e769f5a6c5607b6f314d667d1a7dd951ae5351c98c4595cce67cec410a6609a15472670b6841cc12719e645b97aad11afce304cb7ffe866cd9297717de348d25c6
<= 9000
=> e04280002903e0c81000000000006cfe8df49885ea5ba8005b075db957a0f11d524a1f226a2da39e912e990ca93e
<= 9000
=> e04280000900000000046fe32c00
<= 3200e7984e1a6be05020adef1760a35aec3671402b57a3693d13e7d7dc8b73f6561c974100000000488e1000000000008013e033d0e5a3ef9000
=> e04200001100000000050000800a27a7265510e7c800
<= 9000
=> e04280000101
<= 9000
=> e042800022889b1300000000001976a914b108019156ddd648b20242a5999f87011f35eeff88ac
<= 9000
=> e042800003000002
<= 9000
=> e042800094ca5139024815c6296469c30ef59b66cdd37b2a11aa930347dc2521c25bdee421181b8f500abc6a82a1db9f08a787360d6bb2432da41edc6fdbbffd1cedd6ab1e49c056803a8796288364e4252348c429b1fd0aefd0b634fa12ba6b555008d914566d9c9a68dbb01fac38901c8520c020615495e999df1487e3753beee72d4c3db1a5428ab9c0b1404e0a001582ee7bdba917841f
<= 9000
=> e042800094f956cdcf2fa5bb7117fcf0884fe8708c984cb253b958d2608eaf152005218234e3e2fd82e8df37edc6582037be1c95f9d695eab9ad55cd9c57837660b9fafc19d8e757a29ae8e66dabdb1b3ad96eaabd9be90fcda7c5e2ad64b48c362e41f2223cf290180ebcd67ab9c4167c61e73a00ccb78f14f3f0f0fa1b5db1c1429919c6b5895d5387fa0badc26163eb5f740453f338b35d
<= 9000
=> e042800080fbb75890b2e1c8d7bf7224eee6f46f9c731c0abd44cb2f939a956e77f32b247e804824c04e079c4cc6bed256340f1f95d80b568dbbecf6db640491b8684dcbc65ecded2216c7921875daca1de3af2b25fb0393184417f58f4914bb6272c09cee129496a1d5f6635489aab581b8673abe9a36838c9317ad99c03e0c6bb4e29d7f
<= 9000
=> e04280008015f2440ac46978b580c232a1e310ccfd99638d215117a258c7b7e6e840ec75f6e78a7ade6679ddb7ef54eb7575294084b25d0cfecd951d4ddfde2e767b85543a6803d9db40209ce9909aa2849df954bbeb81a1374d7b84273af77cf72297297d2a9666f087bb970016d77e7880badd0fcef096f9f9d5864ae967789d24d5e8d4
<= 9000
=> e042800080077b47d03fdb6adc8b4bd6d0c177b221c1bc558cee14ed3fa47b3f4c33d099236239516507d286ec4a7941dd12abd04aeb4272d7bfed042bbc99ffacd88cc29ed6635418ddf68c994d9ea1513f7c348181e0c30183df70e6f35b62ca7e29c8ec718b5117c3d4aa7f72856d4c6e0c8d6152a520d05f8227e7e0844cd7cdadc3cd
<= 9000
=> e04280008097a4135135f7b390fa0bd6a05a0d5ab01cbf74fa4f3af5f7db6766d82539647b598040dac5009b13349903c310930e706dfd183262f260c067f851851f190b29f2c84649015755d9e9e30e2a026e43dec40352fa14c601613382182eb4e4b0f2ce9806945b1e83d3418be389152cf64da78c9d12abe2c44d684bcb528608c034
<= 9000
=> e042800080675257f965cca27fa1afb8bea4b6355159e4e8a5fac8ad39af3eaa41a57c49321988551ae50af3fa9faa2ba8cafa213c90102c05b0d4f3885161cba2cce37ae6dee50732a151a2a17b0e574dab458b3088e83d8e2641e8dc2ad044aad56e94b6f7e9e30f262df05858bf6be71080bed08a8e43e53ba682587e0dde9083409add
<= 9000
=> e042800080d9040f49b2e96d1d140a838ad5469a982474b8b4fd882f5557fd145bf6ccb2124f9bcf1e804acfe520ae316c2d290d85c25f59bb897d9df6e303f9dcfc86a343073867f671088b2f662e3e3efc39b568b6b01d9ddd56400ca3be54653587275c272e5ef2003096dda32ca5c75b515d9ad909d0f22fafa2c97400a83d7d981018
<= 9000
=> e0428000808d776908b3249969bfecbae60c85c18df28e5f8001410280282aae185710659dd917b60fc39fb107136ec5e455e0421f646a2959d2253b7052f1168d2f83c7d41958eec1e5efaa1044c052f379582d75d498157732651b21704a9e03af4df5c5f81cab17f2521b30d116160f94b0624f01570418b36a473e5d3f9d5adca69fc0
<= 9000
=> e04280008015653e6b7e00da3de4b970fcc7e92985ef7799177896c247b2729ecd3a548142dbd99d1210959e26fa7253f185ffa1e8da9c387e2c44764816f88cb449df673f3b14cce9eb8e6d795eca0f019a47cc3fea4cb170c3fd6f782418f6cc53459f8735bc8e1e1a04acccbd1578985f90fac17e8563b2e39f75eedf502cd5ea62f863
<= 9000
=> e0428000a0cc8c73241a05204762bd083eb7d36844c04792753537af8428722c2f390ebe8dbc51bf1f6bff58b27044a789f9672f3bba98df27dbd7a81b158c089c4221b6b53b2b2c864bcb7c1c301c4a5e3536484617e838afd3e9657f5ef79a0be781eec612f41d750ec5f74c5987ab3ca87e8712fed35db0e24240983ec04c90bf73692add4ac52f4efbf301a98638a7e2f62f312257357276813262899f8430bd8bd3d5
<= 9000
=> e0428000a02ff46a378a2e80e012e2b3741df234adc1e3fce1257f194c3c9b96c1eb472cba8067d8e1bcef6276c84d2368e84fabc9c3741a4a49ea70672deec5d92ddeb5982c2fe88d0b01dfcec1380f439bb11d54f41f5c74e550199a8df9ce1c5e1f90edb25c55c99add00c5aa7d2ff55f04abd61f86783c09c05ac911f3beb3d1f6ecffd0a2390f0e918aa9376145eb8bc649fb05f052ee116032911048b4fe31f7a19f
<= 9000
=> e0428000290320d6130000000000b36c2720c8fb7224de9121340e6b0a04725dfa76b1a95364bb5ef0cc89bbc737
<= 9000
=> e0428000090000000004da7c2d00
<= 32009d52244536b0ee6b70afe9fe3a72394a10ce9b57d558cf563bfeed2a3c51ccfbf69800000000889b130000000000b2f06f0a2e87a8859000
=> e040000015058000002c80000085800000010000000000000000
<= 41043ae1203a68dc462d508414743e1b06308f06d61604368bc83bd3bfe580adb80ee5a6f7e8366cf1d85f494481324d98f016424c49d00301f30b97cdcf705ba417237431656f45704a34524e4a476f313144714b5348704c32364a78666b5445323537553162907186e91952df6e922a4ec72ff97e80087a35b839ee2c4d1e4c27b17b088d9000
=> e040000015058000002c80000085800000010000000000000001
<= 41048ddaf0918c79a22c095af0c7700923d80d07b9d614ce132ae75bc4fbeecebddb809e284082059bf69992b18d398a328dd917773c0d5fa6b5948d3b7a9bfd924b237431613166435a69474539467833426679316865457847334c7a3261755566666e7a79f159e77b829e0715d27d0b4b9e5f3a51d1d959188ecc6d38bf85f15bf6292ed69000
=> e04400050d050000800a27a7265510e7c802
<= 9000
=> e04480053b01383200e7984e1a6be05020adef1760a35aec3671402b57a3693d13e7d7dc8b73f6561c974100000000488e1000000000008013e033d0e5a3ef19
<= 9000
=> e04480051d76a914e58749ee655c0e39ae3ce063a33fb9edc86d23dd88ac00000000
<= 9000
=> e04480053b013832009d52244536b0ee6b70afe9fe3a72394a10ce9b57d558cf563bfeed2a3c51ccfbf69800000000889b130000000000b2f06f0a2e87a88519
<= 9000
=> e04480051d76a914b108019156ddd648b20242a5999f87011f35eeff88ac00000000
<= 9000
=> e04a8000230180841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac
<= 00009000
=> e04800000b0000000000000100000000
<= 9000
=> e04400800d050000800a27a7265510e7c801
<= 9000
=> e04480803b01383200e7984e1a6be05020adef1760a35aec3671402b57a3693d13e7d7dc8b73f6561c974100000000488e1000000000008013e033d0e5a3ef19
<= 9000
=> e04480801d76a914e58749ee655c0e39ae3ce063a33fb9edc86d23dd88ac00000000
<= 9000
=> e04800001f058000002c8000008580000001000000000000000000000000000100000000
<= 3044022078b76b236fed88b376d7a8a7203a976835c7a8f7bd1d1bfaafb505821060871c0220056c96222f69666f4e23085ad9959053a4c94ddccba3c313112c46fada30d38d019000
=> e04400800d050000800a27a7265510e7c801
<= 9000
=> e04480803b013832009d52244536b0ee6b70afe9fe3a72394a10ce9b57d558cf563bfeed2a3c51ccfbf69800000000889b130000000000b2f06f0a2e87a88519
<= 9000
=> e04480801d76a914b108019156ddd648b20242a5999f87011f35eeff88ac00000000
<= 9000
=> e04800001f058000002c8000008580000001000000000000000100000000000100000000
<= 3044022079ce087bdc737664f9adc83589810ece58538a22a0944837a0696a3390cfadae022030ca76dc22dc9cdc7771d43e3a7a8d15ed949e8f0f800414a10c306e3bf58f8b019000
        `),
      );

      const btc = new Btc({ transport, currency: "zcash" });
      // orchard deshielded
      const tx1 = btc.splitTransaction(
        "050000800a27a7265510e7c8000000006fe32c000001488e1000000000001976a914e58749ee655c0e39ae3ce063a33fb9edc86d23dd88ac000002bd730701f6e2e1feffc2021707919aaced34d5e43c13a5010695cd9d333c19bf310746abef4d8a77fdd55fc303a306f3953f794a95bb8f14d9c94b59d63f792b9e6f9133be659f9cc8b53fc1a26f2e1a9e4e132dd0e78df56be5d06688f103b2a9856f1ec074084e8a91d7008cbed27a00a4e4a816f66e2772c5e04b3f876c08671af32842982bcc7f344410d8eb50a2b166e7b00f6a4e7ff64cb368d84fa789ff76ee84fae0912e178422fff4dc220610589d344a461acaf9487f231d5091575a7deec69f9cdccec92a537b5c996e60a6c16f8502f1e4c873bc88dd2eb2036ba2bfa9cc7b515d721844ce251df61955f78c4e84bff6fdc62404cfbd3bd360b5020bae564e47594ef6cbb4687065749579ab477623d2752f38f30eda3d2231a513b0238b4f45101b2378b50acf8877b182bce2b994ff166cb95f6d319e2b50a0eb432ef7fc556a0e66752fe8ebb0b4184c38820c733fccbeee6e78f3b72f8d61838721f689c4a6a6b1cf8f0fb6bfa1bbdf860ac1631567de443b9cdde8ccdb458b12413f1a1cc90966b40d34249e60eb49d24d9476c89e19721a2d32ce4dbf125a6ffc0a3a9bf3423328ae97e7a2227a08da8e4d1d5031e9d2a6ee04c8d800b1ce42e223d6a16afb88657a46663d7256288e6949b751b4df845a9f544a2366e7976f9ff491e4779dc41431393c7ad99747b2470f24302b28df856506a16ec7b88e77e2c37e2a7a7580d0d6122c63e2f34b63de3011526e6ce4b16bfe43eed07a5364f230d0689c7cc97a2945a7a7e8e8535fa147e665e20bce1d8370e96be7629f34aa72d2a9031c90faf92b8383ed3c51065378b01e143fa64a2eb8322fcf01207c740437b9a9470a67815e541ee1cc3da71349c9092bc296e2d20da040120345717b18df4bb6559caca87349389d9b5567267c640f8289f6caf7c701dc11d0decd9996c47227cc6418c3b13bedd9f4086c7b874b96f0e15bbd6793ba29bd74a4f969f75697e1a817ddba599b2878885724727a32624758df64f86e09d78667e26b4adad05f7d8a539a15e7e19eff71fd29a71c6f58bf920456d8a1348f1225fc1d0b1abbc25bce80a645839d272389faf7a0d4367dcff461a64922b85186de7534bdac9739c57bd8dccd9c0b543a2c1cb1a2edbdbcc7541c03dab9665eda1bc07682ce3dcf4f2293e1bba1aabdbfeb2464739ae16a84f5840e498b935bce15d2900a592d33446c14e9b9bfd8d05b4e24d0142a9027b8a15f11e25d9efd7886e96a214344bbd44c046b223bd1ee5f0b636c5e07d3bec60fed5d37c0d46a2384516ecb02520a8dc06f83f747504078569c55a0290ea78f4a3b765cd00d69a3f60ad695b42d652b285caf3aea639aadb694e24f374ee493445fdb428c95de6016203993655595c7785d3a0c912bfd6af29601429fb773514ed6eb27f376f56384f52c65f9090fa4627cf17b3c4c577685b1aa7f19ddb6c0596f48447d7aa8253d8faca7bf9bfdb76217500e6a9f1fb794c2c5dba36a4d7efafdbc74d28b7b6f36234312a1e3a99bb412bbe14ebc9e94287994e7846e14ac765809eef7e87289665d0089ed4f3399d547f9156ed34a0598852d52a4daff4096ebd3998e748de36a262e718514577baf7ad16d6be978d382ff83fceefd37cc18940c9f9d7f88e1797023fdd4a017721e876088c86121667780e9a32ed92c749bb018e06157b1c7dcd81d7d35de08e3886338b1c6309b15b099c98281ebb79662526753596f9c5784ef6306c04cadf30f3fa650c2ce49bea66bbbe67580585f7e21cc4d04a16b0f2a3729881307020ee35efac01ae6a79a209dfdc911d4119a57ee9184ac38ccba8475d39c40010d2ae5fbb61fa0b0846cdff83afb9f35e63331743fe8b7098da2e58e0ce97593e9cab30bf362537a49ebe30469fc08aad3e75c9ccdd067d73afa042656a1395d15197f98d3ed1034e0552d852202e4eb34b89b801b10d0402a9d698897e6f9d650f685064794d16c7f93755da6db19cd37de31a41746bbb43199e832f0e7f5f675470e9df5d3afc38716873cb0ade0c4b0e1b5e6275d3adf97e260b10fb74fc379407ccb329c4e06645d80c4ab188a7d952a3e631dc16f1ec82e2c5cafd39005006737d047c276aa7ef03325d4405c9e2bd86ad616d9d343dc8cf2b4543a437be4de0d544dd47d91d7e218a16ee75597137e5f4d95e868a9a0939cc41f3d36bae7c2e769f5a6c5607b6f314d667d1a7dd951ae5351c98c4595cce67cec410a6609a15472670b6841cc12719e645b97aad11afce304cb7ffe866cd9297717de348d25c603e0c81000000000006cfe8df49885ea5ba8005b075db957a0f11d524a1f226a2da39e912e990ca93efd601cd5c5ad704c15a04d2ac2359779c862da0f836c50496a8f453bcf553a7c4d62a37fee67ec8a20a36f81af6aba26e3d9696bf00ed53ee920b749bdc33b851f41aed3dd6487f52cc7263f6a32375325505d5ccaa2f5b4beec689469eef841d21487db101eb42638a141e0cf1d40ea7e25d577e5f21365e2bb51f7a21512c2a47bbd37a6f5f83d4deccc7751c2530d2d155d82d6476dd011affc56387722ec908730a29085e5990bcfde4cab6146b4e2826b6fd9c2995f251acf5ef68949d32789b09a55c11cbed7b01af860cd329d5d5ec9a23e869ab2632dc34a7f3ac9735eefa22300bb996a9ef5b307cc451f1b4e99cfe1fe53a1b985699d8b255899e9647eb3b12095175c378ccf525fdba70abb1def63b46caf43aeaad7ee8bc042ea241f191747475d7878319168767b24b59cccf338e6093ccccb7b3c16b8b90e1910e1206245f97f3f0078d66021ce66b6071d207ea411765258432fa2ea05b25bddcfa1f1a6b5d2900dae28c2bddc6f9a19e0ef43abf1616a2f6cab86e69bccb79d491e81b10e00680ebaec48565fb31e236e998a6a97cbc57bb21d6a5c86aba37b1a9bbf196ce20c7f2242b4c92a742e9d98da45374f63c9b7b2b2e59c60d6dfd54e2a34821f60a767772288b9b23b28444bf4117d4e8651c48f9131ecaab2ddcd6c9e46aaae1e307f8406e6bb3361b3cf3bdf19341b4365a3c34f6cde36c1dbe6daaacfbbff6187d52600c4c0c52468c3ff4bed76dc6303b836511c109aa8bc999f0af3c09630c43ab401d39dc5342b2958f3d82920e8ebbba512fed1169b63c11197caf4cd9f4be729c61f3100545ad8ce3e4cd4c2cd70fc1de379bfb8c278020a9af20a8355a778a34410c7bfd06704fa80384c017e4a5c3ebc4890ce9154c1e11fe6b3576d16b1e7d3e8869d1a1076b30c94e92c7df3506420a649483f7470e6972c31257bb276fcbf66dbf0c26cf4269d08f05437cf731635a579765ca8d036140373b133d01c02a8ebee4f28bc59de4ed82eedf5c261bd83deb4dc782fd79e8c5cd2de4137a0f4c9078efc4712ec89effa7c2f319dfee6a7f24e50ca37969dab0a59f2ffd09620dba8281496370aba99b897739f556f7862ec61faae13a57528ec038977e55d5d4f6eaa8af4a5ed07e777f04ffe23e696da446cc837db724f8262ac150f9b7591d4ec4757a9f6ff6cc051933e270a0d7b5dd4dff2d5511d8e1b2627561acc636524e0861e42cd461cf064908ee09664a20b34f717b4f8081014149cdc13dc1ec3921c515e2f4095dcc63b4fa4167527d2da7458b6b86ebf2e1fe7762fc07df50a31998bf8635b093687afdb91e7603e73dc92eaab38f95aa2bb22e7f5404440880b828b4c07f58210a35c600364afc87065eac9ad971cb3deb446c3a4287657737c870e5575af7dd4d9ffa4ce4c240431fc3f64d0ed0a3f102e7b37e98ee68bb3e549b5b466613571ab4f05ef939dfd3ff4e6c3748c924bb0a429202a697dc7231aaf2f14a690546cb6d2b53a648bd1943543f7840e48ff33914f2813b872d02bc2b8a5d595f6a71a932bf7f3deb4ee014ce963470f8ae73a9ceb9c35004c20161626ad7546624837e7afd777f03a8ac908ae5fc57b02284bb190ee5708e94f62af336eba956288b3cdd7e85408b36a078090079bba6c287eaa70646278ec2a56491cccbd6f737417fc555233a5ef54d44d69283da5d291dea4b5e0f3a6fb1a9b887eb3c0f2b9e47cf7cf232815b3326624e97026dfdf1f84bccc30502a1930d966ca2012c9e617f9b349c14bbb2dbb3107188f37eb8e5219bd606665df1c03aa2b6a25369e0cb09343a8d0982dafb23468668e21f436c17b361460874df27c18cca020bee6d6b0db86be0bc8611b03c8e599b1dae4b2b4b40c6ef37b3139ebfedf99b7540314cbbcc435c94c45c0dc7e21f6dc871f652cfc1a970a0b56986e513927dd2211fc05875303667cb9adc9e0eee42f2706a81e66b4bcd51fb27cc4d1c95fbe87e0835c1e77a19c7ecd141435d9379e765ea7ad720388b4448b50736a5d49eeb2f7b78d09ade267fc01fcab778d776973ce278c603780652b9ef734db74ba3b6eed35bb8fba1e1e3da7c12a73bee0487f452b78611cf412375971ec669646a4550bd5aeaa2a6799a36233d42d80547ce842b7fa1abe794950f4cdb6b365b3b2c769643f1bb82c147bb5389f25517f8268cd60e13babd06b69c1b828526a5500b8b1f7d681fd1b4feca57eac252dc061914887b3ab86e5c63de170650eb0ec3745af205df4453cde141c89e5fa100892ec26569700a37c597fa277eec6953ac9afc3f0f2d64d9d3661e5e28534f5918a2534a89a111d7880e9dc8ea0beff8321b68d2cf4c6c9171eb0da4136026d0403a590dd5f108042aabdfb73790f4795e6c6dac200f1f4c75a9b55aec182048b033cb9c71b08185919dee9a8e82af7793681a1a84944bfdee76d96eebe79a4dd45df0f1f3b8e105089beecac213b5fee8e7cab1f0ee916b8d6eece63a8d20c3202d9205aa20c3d1ffa692e3a95708d9f514590050eb7ce283905cf7493e998073e450132a1bc3950bed82beea95966f6b8204394f6427687ab6de5e83f12a6370902b2032b7831cc74ae508e071327ad01d93035abbef6e58ef9a13b3ee316099cd1d243b1df05560a2eaf615edcc8479a976e529830e972a9d9f7315f3233b7c37f021550a411338dea755498964ffca3451002cdc154a008bb52ffa53963b8f25906798f7c32340d97ea857f902ecf5d3e681b233c75388da7dc11d524c603797eb02bf98e063f5fe0f2fb5c96c8301cfb309ed819391a870cdebef0f22440fddf2487c19e3949126efefd5dbeee6dc36cebfb57c54530f18a4d5f93d931244099874ea0db0f0ff45d77e21e2003e276f43dd851aacd1fd07afe25189dfbbe30ab28548eb3255afbff73bda03e499062d8643df50420306091df3238e68a4cd30dc9b7296a3244c5735ea40d4ef0daf27506537f5abcc09f068f7e69f6473a3253acafb814064fe00d39a80e103949bf80aa6041e87232a7be02215bda1f39fbf0360c70422f44157f9ecf678891ea5aa43ef905c2ddb1d637317eeea25907b3163fa6adc70a604ba7f76316ca8a1bacd6d5e2a4ae23661dc182bf920cc649bd6f5818416d3f6bab8e597e029c3d776f6b39b0f35e73d2d0178e1d23f2068987d5f91b884f29515868b6a8a301428e59756466379ebcc281f0d68f5dc8cd96dbe62bf8d6691329d124113e21d95208e56deb4c6f4c993474741c6d0ab6588229a5e615f90935082a171d07777da118dd0efc1f07fe7cde872062a1948369980481998896ec258671f14cecb4755ec6a44c9513002ffe0b02949cbf40ea18e451d936954f752eb80cd99628b8b38bed40ecf9084aa469ee53108fb5ce438cc149b92ca6c58c12e749d2ed3b5861f07fedd07403a71e3578571f6e5c21c044e8125d796a5093184589324ed4cf0293104a47fa858bdbef26d421682ab04b2b74779d23135e3d22e14941a07f49b78645078999a9d67eb844b2d6544ab6b14990440418e619b401ea2ff98ecbba72e8a98206cf82658e16a11a884867a51c51e72d32e015642c0690dae0b061cd48784120565b3e034ba605974aec9a8bc2917c2f940ce2a00a0f7e54ac3fbb8766a07e1fd5261258d672c25f520495df32ff1f6866641e52870a614e352b3907e5db7c2e6b00dc11f81268ca68888d7f2feb3f943d000fb3bd350223d17de58a677eb905977bbde952b0b5bebaec4ed0d14d3dee2f079a62e5087261319974b1ffa5bc3fadec9684985246e94ca7bdaadfd13a7e6795fbbfcb2988d1f2cbc009ed550d4bc6260058b1532f196191f6ce6b28cc9b0d058728903501369d927b34f5ccd7b9b13ccd9f62e89825bed84f9a5e6c413f6ee824926b06f02642e13654a55e2fb6d640065d737654426a3235192791fe455e7354b66c09686055731179a9d21bab30677a48e06798f4601ce226678f0880919fd3df9e1a623a129d7f3ee39105e2a1bda3119723bf1f47846cbe45a502a8c0e6bff68c13c618d3c4d95e5d75b61a38acbad936077f2799694f92012592506db5787e83364a5065d545a58d965f6661717574a82b87f97f39733fde5ba7ea21c93e6f712f286c7fc6b4378165b2c8fe8ad2d4bb7a37c744bca19e9ae07f3b34c59a17970f57dfa8f2a157b91b4ed7126c3180c16b22f8f8ba4b8cac786811066303c74a22a9c22f5f07e5789a1f7ff7592f21dd5361e00ee1fc31c580f274ac5a78239f151ff9792bfff8411debd1edd5bd550861f5f208bfb08c414e2834cc69d1e3b9015c28766b903f130c23c070185d1054b9ba9313eace3fa86e47649171ee62122568f7fba88e6a6428e94b0714be9500980d8e91ab903c043b8433c3c8ebea760dd3273cc6f383764b0710addb2777230e6d51cd7deaf426c7f1ee7127f9d0e23767c86dc5dbcd4e88aa8c0bfeaf94cc1de1971a3352e4a0aae1325fd1103458202d7debc24348433b06752aceeea49b3118d51cf327909fdaf41812e0311e4021624d6c5f87657a34b9caa616ad28c86ebf0da68b7bfe1fe5969294c9b113562f4655178f5c9bf8d4b684c50766a21c3a984fe58001530f54f2ef1532ba0d5f1d514d8fe81665dcc95e5cb153b9705bbc78761e29eb6b0ff09c3b2c5bcdc6dd3d926ab2b2f653b9005e9d1433ab43572cfdbbf611444d8091627b640939d52f1292b3ced5de416f8c60bd64ccaa20fa7d8cb94259d7e88830a0691b9a22f7082d11689d8998b8e047aed7b184645a14f3830d179d0d35d88d3b8bf1cf8669a433094a089149c334b77587c0e41dc54987e9df45903a856f2c9703beef4e0be435247a446b38ab68f5c5f63e020085540e66097321062a8a74e294893bc5a0fe13460ad4b39f09d9bf49be2d60f4d200822797995b713445c9d7fce361351d4713560a8c5d15a30b7902818ee95e1c2ff0c6e74b36862f462477503090b124ff3e21eea237384dc9b10c8910e4682bf322e2cf388a5e561c9c50fe894f26c8e31039f3419ee9919baa6dddba19b45cf71c11bcee972ac18166b3d7f5577716e229901eb9430f83f822023cf1dd7a310b47c31a9494d7c38cecd702167d272d4116643ca94736daf257834dcbb552df1d0fd831da3805d5777f806d879637aacb30af22c2de3f6bd824c7700f01cc94ac2a0fbb17a5a21bbf7e517c87bc57c594292ca253704afcc982d3be8aa28b64201b48b46471262def9f322c3393b5dbf2014ba9bcc66d51f51b91db3008c8c770667dc94def246a8472446d6e9fb3636d0f1b812a22d0944a91ac33bf99fb2cd4a5486bfafd70203f821bbdb79a80ba1d1814de24df9a61f00fa046cf611e1486d5606b8bc2ff108ace735d65c7696e1813c561033352397360bce27d0e113e77defa1e77adc2400e1ee4b4e272b5873a2ac5454b79a2a7e06f1912218049d4e215d6b2ebc455a216782a0cfbff50956a3120875a49855c1ff02adfd86741bb2331e7dd75f23955424ca6015171c2169214a320aedab0d3d8a5fba05417bab631bdfa3eed7087292ae0e6ceb497f535400eef28c74559ebc516039804a116a17b04fa75fcc535b4143e21adab455df2c616f20fab76f5cf8f8217461945e9af871ecd4632e8cb34fe5d910d68bca784912e130d3c5effd5edf46f4e3f8e317f569ed9a7b2195a19d29187871c55c38cac04111d6ded0a3cd44fb479bb976009977f55b17e92e4422e79b602cddc5596443135716e75d3714bbaf5b476ab8c93f9342979a38e88dcd289fab2a07c61f7ab229ff02d4544c8226a6de1b8946def21e719a22379ccabee41c85085662f5da201947be59450eecd88e841537690aeab3cd98a2ae9897705c77b96a8fa6eda2c2ec5b3206e08ccdc204d011abbfc7b577902bccd5f5f6422173b4f7fe975f4c93ef95e43cd0fdad843606b2710b07d5da288b2fde5ff93de35eb0b0c2ce1560c2cc316c6fab0f7a60ee7dc20eca539b8c2bf85cecad5fd13a6c0a29273f6cdc01a011a8441c79ff4beb3016c2677a641938e8b95d6771a40673e0c79489dfe32329fc852d1453e04dc3163a3c1a3eb71a3b98dbba4e4e51fac00a8242485d40329247513b0839dfc095cf65a17dab8acf67d3789a975bb03edeb32d6119f0a5b3cdb2b92c50a3b6d66a7f8a5f7d4b60c83237d0fa27615f696d73ff5dd49dfe626ec3962e1d65c4a4b373442c34946144d2d383c043bdc5874808b21be3dffe320a1e7061c3aaf181b82721d3b7764a719fafc5d23d05aabd33f3edf6d403aa526025d1fc491c446750334ab4fbd9fb1cba986a300d3d81fbe64bd07d122a17e3ecd8e24ab6e3c37b87d2701653dec669b48d96fde97d9090e9ea332a4f6c76b1d4ec8a5c5ccf3df6cb6a2cafb840a2d981d2873bba28c7e45e98a7825e824523dd52bd35bdfc236e2001cf3bfaacc4c60530e9805eaf633000f7ca39aa60d861826d94c3dbf897468fffe88755f4492555e06c2d58758c102e4538488bfe95b3b74a448e72b1959fd7b19f2b52bc7ee08dc85c1231a5fd3c1dfc2d9e279c46008a84a8aa8be6f959bd3b0e96416a7a7b40ce4c8fbd27ca9fde72b2064dc693a258e853ad12a0f9be8aa724b5d4ce6819f20a200ac89d3c15ad6db02810c782d19a6b5b7e9b62fffc71dee4bb6bf121452ef3cacc9637948b5b2ee04848e755629a3f730b100f11bf8e095a17b78b570ec7ec54e1ba2446d26c0bb151fd171cd3e5b8c161af853051f967d9c96149b683c2c1dbc249d7208f9f10c58d463bd713f0d8997a334c35911a960cd57e1b8fcdf291ba69b90e3249daa6d3e2141a7691c0307150b953981c7861f22481a4f03fa769d1157ddc93f6bdc149b7023c31114496ba83514254b466a0b349d2007c21fc547aff700a389268d76327676c9873fe760b71d62f785c737820fe5d3356eda490a8ffb7562f11e2b0dfa7a3bcf7f2fcf2a801b263226c91031f8edbcb935731556aeccdb4d97893ca29e946d05803b016433dc20beaf83b9b60f9fbde490e1a0618cfed53407024890ead6991d811ddb5cd14d21982a1d99e9d72bb27b11d3102fd02d3d7027308073639a1ff1ec39ca038c5eeb8cf5603f19e83be3bfc574842e85935a20a414a851efed6998cb00ae827d33c2eb2b0f51616cbfcf0cd02e8bc00559c3827d43aca756b7e36407254c3a107674a41989d69c0cd25cf220eb3a4228468b9693668e2e946d9aa9bf2e8105c738fececb3486787007a292d4bad2a760015933c855b555be96b2e2622256bda6255e2ca7969696199b919ac7e22568387efec32f29270c36771c8be834fa1aac834db84a72dc1584299ed9ce45083ea53f074a9b5d55ad8c8ccc7e181666ac1e8f7a469b68a90a4b93384ac3b01a4304e1b07d66632a88d3abc620c436b4e815fc33ebacf7304889b98dccaf28b8028b7bc4c934345b0a97ab79db601379a32b057b96e5334ea15d2af585c7cbf568c28ed622ad96c728e523e228310b9b661ccaf9bc5dc5edd625aaff6ad87de4f4be746c56e774303e12a2f7fac1310622e009b611c5bfe9653bcf8c01982c57ff41bf8f4e5ac7628f99fdae138a1e66f684cbbbfa218d9d7104781e227262d50bd284f2096977e276e11fe107ec150bb605f614dbcbb6fb08d9c8184dbdbe82699f83701c088f183a507c6477c00093a19b36d9dbd36ff25536dad0705158bda21cc9050f7523cb209fc6d79f310f01ccc74c567654af502b57622b4db5203fef095dd55653035f889f03001b3a06e9c231eb35ef006a64f46cd7710db19075e49d7237bffbbcd6896d91d994d91e787b723e28a93e423e5a325371b290d39f9fe4647217a9661616006b6d4bb82098d7c269e9deb3d97c74c011e06a6a0ccb56de92c98e96dc19bb723626e1d8030346f9a7dfb040e76e6b1d6500bfc425cdd224f4112e16b2754026e4c05a7a2c9ae96e1ba8791975ef756d9703b19d6f86da0142ebe2c90971560f32940fae0590c25f255a69f91c75dd44b4fc3b84934c71c12fc1f6d16acbddb69bbb048331567efd6763c060f8ab2bf6651cd06dbab9219205c83c36b8de23b2de076d383587a18f5f01e76b2a6fa9d604bf3e98db4c70daaeba79fe44b91af0c2ef1fc53340852ddd0b47dffea8ad65cfdfc8d159515965133b6481294e75c2855857f604fa4fd35891ecccd20289b6ae9ddb60586949156d624a29ea3c6358ec25544302564b8e9b00deccab5cdb9820e4767ec6826b33314ca067699e6c69da9d5fad0ea62213285d486355305db1d5b2391392217e38d4d87d4243c83afe2e26aa7609b4e95ff8dc4f41bc6430d76430d7bdfdf5a32e897b41ac3496c3745b86fdd62dd9c53d8fe84b0aff97416e68f880fea9d0536880a99c8268013eda4619d636067017fcad5ad979f2255fa57a574c4792cfaf017a45d2519d4018c64a00e6c42352c84e4fc4ff9c2cac9eb59b7ec56f28faaf249d9cfbc22e3653db54cba6ce27d6d8867c6bbbd6d07fa1c3c816038ad534ad63c6e9a112f5534a0f390bdc6c05dc8d19f3502003dbbee74c074254945c7232e994df7cdfbbd0c255fb520c7e07115c8d44baea44595ceabd857d690b049cddbadaad43c33a624fd764dd90862b997b149f2305364a4ef9eefdc670e59092e7fbe38915266946a4ab6f07f0d42b03306cf678ee6802f5ab6df79101413f581cfa56b429d59096eaf1399b303532522688ca0ae363c770c70ffc1764230b61648ed1e14d742cb19bd70611cfbe3c0ca0c2654e2f1777e8bed1b49bd9e51d46dd6634123f0c7a0d7314fd311e973b415b5d3436ceb5c2060f629522d102b2b48b596cbc19530387929d136ee6953da9ab7617e040e534c4a880bfb7d13722df1696e358582b519d402fa05a61911458f6efff4f32669a6a8b948b42a6f0354a9c9659d118c8495ee25a5b2243643073b4a7abc0ff40fad3daa844445c0c909147682a07a4964c6cd9372795d555083d3b2885770371d5fbb78a797badd7554d1659c2dc7fbb00b107b39f2d562701d8c86d2f77946d257fd65f72a3536d1c76d80f8728927bf0c53be34a9dcb2c1dabcca9cfa9f84e36dd3b5039fcf106e4489715d6aa6ef387d8c38391ad61c21111493cb5080461b7cbe686777f0e619764b3a9be2414c7ec1ee212d474ca45256e4ed49637644e812d06151bbe44124ad4ef5287a8ac15b74f84c1c74dbc86ad368631c85e1eb74cfb97fb9320e0ad1a8c224f9cd415f364146a1ceb95e1d40cb8c15cd38c8a9f0684903439a5b76d9ac5247a3607082b56e44b9c3f63f785b0f1ad2b241cb2a3dac103a9b0de108b8b67ce4302b57036e62f589cbe480503a839e0915b4bef0a77356125ed548f8c84aee8e5f7e9bb268b0536d44b0a87363cc191262f52c972f8273ad17bf229c2c4a943362f3438723255e488cf89c555a0f777064de46030fc625d7ee4de4b6ee3ab3dade0b6a52dcaf6209150dba2450d1fe8eea8fcd70c9d097be7cf5e24dadf3cb733a2e28b6736d6398e0a3bef0a3f095c276b8bfd13d3e26ccc9bbfc6c666c93b7c2400b64f0a35a00a60d2cb1931d861e38fd79c430f9fe170528ef6e6a71ef3b409a2e2c8fd6435c8d6ba914a19e852dc3768bb27322eb99047dff1e290e89234b0351b15d2ebf86b61831cbd2ac23dc5e9f083c3bdd87538e66e6f5f1516704762b8682366a6c166b73622f59da30b2c09f3d4255888abb6b51c3637eeaf6cd4554603e18c01f96f1cdd1afabfb449cf1104c92d651a8ce98ada871074e6bdd4ba0c23c35e043a2bbadad8cc2017dcd0c3bc2b2ec18b9458661febda501634272de858af18a67b17d22a46eb3ceb5379a1f9228cd88179aa118eba2bc15d311087881e19904cbb967945eb7d0e22875d854b37e3a6b6c7f16d6791dfec212bbd902936f8c1e9f59d8093b56494cab299f7cb68f0aba306a598c0ce08b873e5c2c9841dec93abdd209bf2cbce89de0d6b8f5ec9716a891c198f15e5b6f7e14aee20e4b33924f430e381785f72b1edf025900551ecbeaf42381a8f02b815e9944a2d987a429d704ca191c778373d0357cbff6a814947bf2188ef16810dd1ab2abfd8fa8e8e55e5d50d508658fd0e7986e1939b96894a895d419022731078be994dee157ab6e06dd578e0d665490c23f51b139b9802bffc0baff177652e37d1b7c8cf744feff1ef1acb003f803530c89d60b899bc3e54079da17c5cc487ccc555ab83ebd36d44b401f772169dea21bd14563007373e0e55ca1a084d35fa557ef0c66a57f708912193a6cb1f924d28037c98e42a44153fe28ddb7023aa8bc77a60806c4a0d8be104a42004dda77aa056165106b0b13510dd97514e3bd5f4059b2f9b525b565bc798020e4582c6060248c8075628cd39d663497ade707a4be75a347b27ef86fb9d73f1fec0d2c877b42f28b999406f533b158ed8c2b05180966c375922eaffb8a096c4ae9eed2b073d",
        true,
        true,
        ["zcash", "sapling"],
      );

      // orchard deshielded
      const tx2 = btc.splitTransaction(
        "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",
        true,
        true,
        ["zcash", "sapling"],
      );

      const result = await btc.createPaymentTransaction({
        inputs: [
          [tx1, 0, null, 0, 2941758],
          [tx2, 0, null, 0, 2981033],
        ],
        associatedKeysets: ["44'/133'/1'/0/0", "44'/133'/1'/0/1"],
        changePath: undefined,
        blockHeight: 2984376,
        sigHashType: 1,
        outputScriptHex: "0180841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac",
        additionals: ["zcash", "sapling"],
        expiryHeight: Buffer.from([0, 0, 0, 0]),
      });

      expect(result).toEqual(
        "050000800a27a7265510e7c80000000000000000024e1a6be05020adef1760a35aec3671402b57a3693d13e7d7dc8b73f6561c9741000000006a473044022078b76b236fed88b376d7a8a7203a976835c7a8f7bd1d1bfaafb505821060871c0220056c96222f69666f4e23085ad9959053a4c94ddccba3c313112c46fada30d38d0121033ae1203a68dc462d508414743e1b06308f06d61604368bc83bd3bfe580adb80e00000000244536b0ee6b70afe9fe3a72394a10ce9b57d558cf563bfeed2a3c51ccfbf698000000006a473044022079ce087bdc737664f9adc83589810ece58538a22a0944837a0696a3390cfadae022030ca76dc22dc9cdc7771d43e3a7a8d15ed949e8f0f800414a10c306e3bf58f8b0121038ddaf0918c79a22c095af0c7700923d80d07b9d614ce132ae75bc4fbeecebddb000000000180841e00000000001976a914a96e684ec46cd8a2f98d6ef4b847c0ee88395e9388ac000000",
      );
    });
  });

  describe("getDefaultVersions", () => {
    it("should return default versions for non-Zcash and non-Decred with no expiryHeight", () => {
      const result = getDefaultVersions({
        isZcash: false,
        sapling: false,
        isDecred: false,
        expiryHeight: undefined,
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(1);
    });

    it("should return Zcash versions with expiryHeight and blockHeight below activation height", () => {
      const result = getDefaultVersions({
        isZcash: true,
        sapling: false,
        isDecred: false,
        expiryHeight: Buffer.alloc(4),
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(0x80000005);
    });

    it("should return Zcash versions with expiryHeight and blockHeight above activation height", () => {
      const blockHeight = 3_000_000;
      expect(blockHeight > ZCASH_ACTIVATION_HEIGHTS.NU6).toBe(true);
      const result = getDefaultVersions({
        isZcash: true,
        sapling: false,
        isDecred: false,
        expiryHeight: Buffer.alloc(4),
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(0x80000005);
    });

    it("should return Sapling versions with expiryHeight", () => {
      const result = getDefaultVersions({
        isZcash: false,
        sapling: true,
        isDecred: false,
        expiryHeight: Buffer.alloc(4),
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(0x80000004);
    });

    it("should return non-Sapling versions with expiryHeight", () => {
      const result = getDefaultVersions({
        isZcash: false,
        sapling: false,
        isDecred: false,
        expiryHeight: Buffer.alloc(4),
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(0x80000003);
    });

    it("should return default versions for Decred with expiryHeight", () => {
      const result = getDefaultVersions({
        isZcash: false,
        sapling: false,
        isDecred: true,
        expiryHeight: Buffer.alloc(4),
      });
      expect(result.defaultVersion.readUInt32LE(0)).toBe(1);
    });
  });
});
