"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createKey = exports.WalletPolicy = void 0;
const bitcoinjs_lib_1 = require("bitcoinjs-lib");
const bip32_1 = require("../bip32");
const buffertools_1 = require("../buffertools");
const merkle_1 = require("./merkle");
/**
 * The Bitcon hardware app uses a descriptors-like thing to describe
 * how to construct output scripts from keys. A "Wallet Policy" consists
 * of a "Descriptor Template" and a list of "keys". A key is basically
 * a serialized BIP32 extended public key with some added derivation path
 * information. This is documented at
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/master/doc/wallet.md
 */
class WalletPolicy {
    descriptorTemplate;
    keys;
    /**
     * For now, we only support default descriptor templates.
     */
    constructor(descriptorTemplate, key) {
        this.descriptorTemplate = descriptorTemplate;
        this.keys = [key];
    }
    getWalletId() {
        // wallet_id (sha256 of the wallet serialization),
        return bitcoinjs_lib_1.crypto.sha256(this.serialize());
    }
    serialize() {
        const keyBuffers = this.keys.map(k => {
            return Buffer.from(k, "ascii");
        });
        const m = new merkle_1.Merkle(keyBuffers.map(k => (0, merkle_1.hashLeaf)(k)));
        const buf = new buffertools_1.BufferWriter();
        buf.writeUInt8(0x01); // wallet type (policy map)
        buf.writeUInt8(0); // length of wallet name (empty string for default wallets)
        buf.writeVarSlice(Buffer.from(this.descriptorTemplate, "ascii"));
        buf.writeVarInt(this.keys.length), buf.writeSlice(m.getRoot());
        return buf.buffer();
    }
}
exports.WalletPolicy = WalletPolicy;
function createKey(masterFingerprint, path, xpub) {
    const accountPath = (0, bip32_1.pathArrayToString)(path);
    return `[${masterFingerprint.toString("hex")}${accountPath.substring(1)}]${xpub}/**`;
}
exports.createKey = createKey;
//# sourceMappingURL=policy.js.map