"use strict";
// Copyright (c) 2023 Jose-Luis Landabaso - https://bitcoinerlab.com
// Distributed under the MIT software license
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, state, value, kind, f) {
    if (kind === "m") throw new TypeError("Private method is not writable");
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a setter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
    return (kind === "a" ? f.call(receiver, value) : f ? f.value = value : state.set(receiver, value)), value;
};
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, state, kind, f) {
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
    return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DescriptorsFactory = void 0;
const bitcoinjs_lib_1 = require("bitcoinjs-lib");
const { p2sh, p2wpkh, p2pkh, p2pk, p2wsh, p2tr } = bitcoinjs_lib_1.payments;
const bip32_1 = require("bip32");
const ecpair_1 = require("ecpair");
const psbt_1 = require("./psbt");
const checksum_1 = require("./checksum");
const keyExpressions_1 = require("./keyExpressions");
const RE = __importStar(require("./re"));
const miniscript_1 = require("./miniscript");
//See "Resource limitations" https://bitcoin.sipa.be/miniscript/
//https://lists.linuxfoundation.org/pipermail/bitcoin-dev/2019-September/017306.html
const MAX_SCRIPT_ELEMENT_SIZE = 520;
const MAX_STANDARD_P2WSH_SCRIPT_SIZE = 3600;
const MAX_OPS_PER_SCRIPT = 201;
function countNonPushOnlyOPs(script) {
    const decompile = bitcoinjs_lib_1.script.decompile(script);
    if (!decompile)
        throw new Error(`Error: cound not decompile ${script}`);
    return decompile.filter(op => typeof op === 'number' && op > bitcoinjs_lib_1.script.OPS['OP_16']).length;
}
/*
 * Returns a bare descriptor without checksum and particularized for a certain
 * index (if desc was a range descriptor)
 */
function evaluate({ expression, checksumRequired, index }) {
    const mChecksum = expression.match(String.raw `(${RE.reChecksum})$`);
    if (mChecksum === null && checksumRequired === true)
        throw new Error(`Error: descriptor ${expression} has not checksum`);
    //evaluatedExpression: a bare desc without checksum and particularized for a certain
    //index (if desc was a range descriptor)
    let evaluatedExpression = expression;
    if (mChecksum !== null) {
        const checksum = mChecksum[0].substring(1); //remove the leading #
        evaluatedExpression = expression.substring(0, expression.length - mChecksum[0].length);
        if (checksum !== (0, checksum_1.DescriptorChecksum)(evaluatedExpression)) {
            throw new Error(`Error: invalid descriptor checksum for ${expression}`);
        }
    }
    if (index !== undefined) {
        const mWildcard = evaluatedExpression.match(/\*/g);
        if (mWildcard && mWildcard.length > 0) {
            //From  https://github.com/bitcoin/bitcoin/blob/master/doc/descriptors.md
            //To prevent a combinatorial explosion of the search space, if more than
            //one of the multi() key arguments is a BIP32 wildcard path ending in /* or
            //*', the multi() expression only matches multisig scripts with the ith
            //child key from each wildcard path in lockstep, rather than scripts with
            //any combination of child keys from each wildcard path.
            //We extend this reasoning for musig for all cases
            evaluatedExpression = evaluatedExpression.replaceAll('*', index.toString());
        }
        else
            throw new Error(`Error: index passed for non-ranged descriptor: ${expression}`);
    }
    return evaluatedExpression;
}
/**
 * Builds the functions needed to operate with descriptors using an external elliptic curve (ecc) library.
 * @param {Object} ecc - an object containing elliptic curve operations, such as [tiny-secp256k1](https://github.com/bitcoinjs/tiny-secp256k1) or [@bitcoinerlab/secp256k1](https://github.com/bitcoinerlab/secp256k1).
 */
function DescriptorsFactory(ecc) {
    var _Descriptor_instances, _Descriptor_payment, _Descriptor_preimages, _Descriptor_signersPubKeys, _Descriptor_miniscript, _Descriptor_witnessScript, _Descriptor_redeemScript, _Descriptor_isSegwit, _Descriptor_expandedExpression, _Descriptor_expandedMiniscript, _Descriptor_expansionMap, _Descriptor_network, _Descriptor_getTimeConstraints, _Descriptor_assertPsbtInput;
    const BIP32 = (0, bip32_1.BIP32Factory)(ecc);
    const ECPair = (0, ecpair_1.ECPairFactory)(ecc);
    const signatureValidator = (pubkey, msghash, signature) => ECPair.fromPublicKey(pubkey).verify(msghash, signature);
    /**
     * Takes a string key expression (xpub, xprv, pubkey or wif) and parses it
     */
    const parseKeyExpression = ({ keyExpression, isSegwit, network = bitcoinjs_lib_1.networks.bitcoin }) => {
        return (0, keyExpressions_1.parseKeyExpression)({
            keyExpression,
            network,
            ...(typeof isSegwit === 'boolean' ? { isSegwit } : {}),
            ECPair,
            BIP32
        });
    };
    /**
     * Takes a descriptor (expression) and expands it to its corresponding Bitcoin script and other relevant details.
     *
     * @throws {Error} Throws an error if the descriptor cannot be parsed or does not conform to the expected format.
     */
    const expand = ({ expression, index, checksumRequired = false, network = bitcoinjs_lib_1.networks.bitcoin, allowMiniscriptInP2SH = false }) => {
        let expandedExpression;
        let miniscript;
        let expansionMap;
        let isSegwit;
        let expandedMiniscript;
        let payment;
        let witnessScript;
        let redeemScript;
        const isRanged = expression.indexOf('*') !== -1;
        if (index !== undefined)
            if (!Number.isInteger(index) || index < 0)
                throw new Error(`Error: invalid index ${index}`);
        //Verify and remove checksum (if exists) and
        //particularize range descriptor for index (if desc is range descriptor)
        const canonicalExpression = evaluate({
            expression,
            ...(index !== undefined ? { index } : {}),
            checksumRequired
        });
        const isCanonicalRanged = canonicalExpression.indexOf('*') !== -1;
        //addr(ADDR)
        if (canonicalExpression.match(RE.reAddrAnchored)) {
            if (isRanged)
                throw new Error(`Error: addr() cannot be ranged`);
            const matchedAddress = canonicalExpression.match(RE.reAddrAnchored)?.[1]; //[1]-> whatever is found addr(->HERE<-)
            if (!matchedAddress)
                throw new Error(`Error: could not get an address in ${expression}`);
            let output;
            try {
                output = bitcoinjs_lib_1.address.toOutputScript(matchedAddress, network);
            }
            catch (e) {
                throw new Error(`Error: invalid address ${matchedAddress}`);
            }
            try {
                payment = p2pkh({ output, network });
            }
            catch (e) { }
            try {
                payment = p2sh({ output, network });
            }
            catch (e) { }
            try {
                payment = p2wpkh({ output, network });
            }
            catch (e) { }
            try {
                payment = p2wsh({ output, network });
            }
            catch (e) { }
            try {
                payment = p2tr({ output, network });
            }
            catch (e) { }
            if (!payment) {
                throw new Error(`Error: invalid address ${matchedAddress}`);
            }
        }
        //pk(KEY)
        else if (canonicalExpression.match(RE.rePkAnchored)) {
            isSegwit = false;
            const keyExpression = canonicalExpression.match(RE.reKeyExp)?.[0];
            if (!keyExpression)
                throw new Error(`Error: keyExpression could not me extracted`);
            if (canonicalExpression !== `pk(${keyExpression})`)
                throw new Error(`Error: invalid expression ${expression}`);
            expandedExpression = 'pk(@0)';
            const pKE = parseKeyExpression({ keyExpression, network, isSegwit });
            expansionMap = { '@0': pKE };
            if (!isCanonicalRanged) {
                const pubkey = pKE.pubkey;
                //Note there exists no address for p2pk, but we can still use the script
                if (!pubkey)
                    throw new Error(`Error: could not extract a pubkey from ${expression}`);
                payment = p2pk({ pubkey, network });
            }
        }
        //pkh(KEY) - legacy
        else if (canonicalExpression.match(RE.rePkhAnchored)) {
            isSegwit = false;
            const keyExpression = canonicalExpression.match(RE.reKeyExp)?.[0];
            if (!keyExpression)
                throw new Error(`Error: keyExpression could not me extracted`);
            if (canonicalExpression !== `pkh(${keyExpression})`)
                throw new Error(`Error: invalid expression ${expression}`);
            expandedExpression = 'pkh(@0)';
            const pKE = parseKeyExpression({ keyExpression, network, isSegwit });
            expansionMap = { '@0': pKE };
            if (!isCanonicalRanged) {
                const pubkey = pKE.pubkey;
                if (!pubkey)
                    throw new Error(`Error: could not extract a pubkey from ${expression}`);
                payment = p2pkh({ pubkey, network });
            }
        }
        //sh(wpkh(KEY)) - nested segwit
        else if (canonicalExpression.match(RE.reShWpkhAnchored)) {
            isSegwit = true;
            const keyExpression = canonicalExpression.match(RE.reKeyExp)?.[0];
            if (!keyExpression)
                throw new Error(`Error: keyExpression could not me extracted`);
            if (canonicalExpression !== `sh(wpkh(${keyExpression}))`)
                throw new Error(`Error: invalid expression ${expression}`);
            expandedExpression = 'sh(wpkh(@0))';
            const pKE = parseKeyExpression({ keyExpression, network, isSegwit });
            expansionMap = { '@0': pKE };
            if (!isCanonicalRanged) {
                const pubkey = pKE.pubkey;
                if (!pubkey)
                    throw new Error(`Error: could not extract a pubkey from ${expression}`);
                payment = p2sh({ redeem: p2wpkh({ pubkey, network }), network });
                redeemScript = payment.redeem?.output;
                if (!redeemScript)
                    throw new Error(`Error: could not calculate redeemScript for ${expression}`);
            }
        }
        //wpkh(KEY) - native segwit
        else if (canonicalExpression.match(RE.reWpkhAnchored)) {
            isSegwit = true;
            const keyExpression = canonicalExpression.match(RE.reKeyExp)?.[0];
            if (!keyExpression)
                throw new Error(`Error: keyExpression could not me extracted`);
            if (canonicalExpression !== `wpkh(${keyExpression})`)
                throw new Error(`Error: invalid expression ${expression}`);
            expandedExpression = 'wpkh(@0)';
            const pKE = parseKeyExpression({ keyExpression, network, isSegwit });
            expansionMap = { '@0': pKE };
            if (!isCanonicalRanged) {
                const pubkey = pKE.pubkey;
                if (!pubkey)
                    throw new Error(`Error: could not extract a pubkey from ${expression}`);
                payment = p2wpkh({ pubkey, network });
            }
        }
        //sh(wsh(miniscript))
        else if (canonicalExpression.match(RE.reShWshMiniscriptAnchored)) {
            isSegwit = true;
            miniscript = canonicalExpression.match(RE.reShWshMiniscriptAnchored)?.[1]; //[1]-> whatever is found sh(wsh(->HERE<-))
            if (!miniscript)
                throw new Error(`Error: could not get miniscript in ${expression}`);
            ({ expandedMiniscript, expansionMap } = expandMiniscript({
                miniscript,
                isSegwit,
                network
            }));
            expandedExpression = `sh(wsh(${expandedMiniscript}))`;
            if (!isCanonicalRanged) {
                const script = (0, miniscript_1.miniscript2Script)({ expandedMiniscript, expansionMap });
                witnessScript = script;
                if (script.byteLength > MAX_STANDARD_P2WSH_SCRIPT_SIZE) {
                    throw new Error(`Error: script is too large, ${script.byteLength} bytes is larger than ${MAX_STANDARD_P2WSH_SCRIPT_SIZE} bytes`);
                }
                const nonPushOnlyOps = countNonPushOnlyOPs(script);
                if (nonPushOnlyOps > MAX_OPS_PER_SCRIPT) {
                    throw new Error(`Error: too many non-push ops, ${nonPushOnlyOps} non-push ops is larger than ${MAX_OPS_PER_SCRIPT}`);
                }
                payment = p2sh({
                    redeem: p2wsh({ redeem: { output: script, network }, network }),
                    network
                });
                redeemScript = payment.redeem?.output;
                if (!redeemScript)
                    throw new Error(`Error: could not calculate redeemScript for ${expression}`);
            }
        }
        //sh(miniscript)
        else if (canonicalExpression.match(RE.reShMiniscriptAnchored)) {
            //isSegwit false because we know it's a P2SH of a miniscript and not a
            //P2SH that embeds a witness payment.
            isSegwit = false;
            miniscript = canonicalExpression.match(RE.reShMiniscriptAnchored)?.[1]; //[1]-> whatever is found sh(->HERE<-)
            if (!miniscript)
                throw new Error(`Error: could not get miniscript in ${expression}`);
            if (allowMiniscriptInP2SH === false &&
                //These top-level expressions within sh are allowed within sh.
                //They can be parsed with miniscript2Script, but first we must make sure
                //that other expressions are not accepted (unless forced with allowMiniscriptInP2SH).
                miniscript.search(/^(pk\(|pkh\(|wpkh\(|combo\(|multi\(|sortedmulti\(|multi_a\(|sortedmulti_a\()/) !== 0) {
                throw new Error(`Error: Miniscript expressions can only be used in wsh`);
            }
            ({ expandedMiniscript, expansionMap } = expandMiniscript({
                miniscript,
                isSegwit,
                network
            }));
            expandedExpression = `sh(${expandedMiniscript})`;
            if (!isCanonicalRanged) {
                const script = (0, miniscript_1.miniscript2Script)({ expandedMiniscript, expansionMap });
                redeemScript = script;
                if (script.byteLength > MAX_SCRIPT_ELEMENT_SIZE) {
                    throw new Error(`Error: P2SH script is too large, ${script.byteLength} bytes is larger than ${MAX_SCRIPT_ELEMENT_SIZE} bytes`);
                }
                const nonPushOnlyOps = countNonPushOnlyOPs(script);
                if (nonPushOnlyOps > MAX_OPS_PER_SCRIPT) {
                    throw new Error(`Error: too many non-push ops, ${nonPushOnlyOps} non-push ops is larger than ${MAX_OPS_PER_SCRIPT}`);
                }
                payment = p2sh({ redeem: { output: script, network }, network });
            }
        }
        //wsh(miniscript)
        else if (canonicalExpression.match(RE.reWshMiniscriptAnchored)) {
            isSegwit = true;
            miniscript = canonicalExpression.match(RE.reWshMiniscriptAnchored)?.[1]; //[1]-> whatever is found wsh(->HERE<-)
            if (!miniscript)
                throw new Error(`Error: could not get miniscript in ${expression}`);
            ({ expandedMiniscript, expansionMap } = expandMiniscript({
                miniscript,
                isSegwit,
                network
            }));
            expandedExpression = `wsh(${expandedMiniscript})`;
            if (!isCanonicalRanged) {
                const script = (0, miniscript_1.miniscript2Script)({ expandedMiniscript, expansionMap });
                witnessScript = script;
                if (script.byteLength > MAX_STANDARD_P2WSH_SCRIPT_SIZE) {
                    throw new Error(`Error: script is too large, ${script.byteLength} bytes is larger than ${MAX_STANDARD_P2WSH_SCRIPT_SIZE} bytes`);
                }
                const nonPushOnlyOps = countNonPushOnlyOPs(script);
                if (nonPushOnlyOps > MAX_OPS_PER_SCRIPT) {
                    throw new Error(`Error: too many non-push ops, ${nonPushOnlyOps} non-push ops is larger than ${MAX_OPS_PER_SCRIPT}`);
                }
                payment = p2wsh({ redeem: { output: script, network }, network });
            }
        }
        else {
            throw new Error(`Error: Could not parse descriptor ${expression}`);
        }
        return {
            ...(payment !== undefined ? { payment } : {}),
            ...(expandedExpression !== undefined ? { expandedExpression } : {}),
            ...(miniscript !== undefined ? { miniscript } : {}),
            ...(expansionMap !== undefined ? { expansionMap } : {}),
            ...(isSegwit !== undefined ? { isSegwit } : {}),
            ...(expandedMiniscript !== undefined ? { expandedMiniscript } : {}),
            ...(redeemScript !== undefined ? { redeemScript } : {}),
            ...(witnessScript !== undefined ? { witnessScript } : {}),
            isRanged,
            canonicalExpression
        };
    };
    /**
     * Expand a miniscript to a generalized form using variables instead of key
     * expressions. Variables will be of this form: @0, @1, ...
     * This is done so that it can be compiled with compileMiniscript and
     * satisfied with satisfier.
     * Also compute pubkeys from descriptors to use them later.
     */
    function expandMiniscript({ miniscript, isSegwit, network = bitcoinjs_lib_1.networks.bitcoin }) {
        return (0, miniscript_1.expandMiniscript)({
            miniscript,
            isSegwit,
            network,
            BIP32,
            ECPair
        });
    }
    class Descriptor {
        /**
         * @param options
         * @throws {Error} - when descriptor is invalid
         */
        constructor({ expression, index, checksumRequired = false, allowMiniscriptInP2SH = false, network = bitcoinjs_lib_1.networks.bitcoin, preimages = [], signersPubKeys }) {
            _Descriptor_instances.add(this);
            _Descriptor_payment.set(this, void 0);
            _Descriptor_preimages.set(this, []);
            _Descriptor_signersPubKeys.set(this, void 0);
            _Descriptor_miniscript.set(this, void 0);
            _Descriptor_witnessScript.set(this, void 0);
            _Descriptor_redeemScript.set(this, void 0);
            //isSegwit true if witnesses are needed to the spend coins sent to this descriptor.
            //may be unset because we may get addr(P2SH) which we don't know if they have segwit.
            _Descriptor_isSegwit.set(this, void 0);
            _Descriptor_expandedExpression.set(this, void 0);
            _Descriptor_expandedMiniscript.set(this, void 0);
            _Descriptor_expansionMap.set(this, void 0);
            _Descriptor_network.set(this, void 0);
            __classPrivateFieldSet(this, _Descriptor_network, network, "f");
            __classPrivateFieldSet(this, _Descriptor_preimages, preimages, "f");
            if (typeof expression !== 'string')
                throw new Error(`Error: invalid descriptor type`);
            const expandedResult = expand({
                expression,
                ...(index !== undefined ? { index } : {}),
                checksumRequired,
                network,
                allowMiniscriptInP2SH
            });
            if (expandedResult.isRanged && index === undefined)
                throw new Error(`Error: index was not provided for ranged descriptor`);
            if (!expandedResult.payment)
                throw new Error(`Error: could not extract a payment from ${expression}`);
            __classPrivateFieldSet(this, _Descriptor_payment, expandedResult.payment, "f");
            if (expandedResult.expandedExpression !== undefined)
                __classPrivateFieldSet(this, _Descriptor_expandedExpression, expandedResult.expandedExpression, "f");
            if (expandedResult.miniscript !== undefined)
                __classPrivateFieldSet(this, _Descriptor_miniscript, expandedResult.miniscript, "f");
            if (expandedResult.expansionMap !== undefined)
                __classPrivateFieldSet(this, _Descriptor_expansionMap, expandedResult.expansionMap, "f");
            if (expandedResult.isSegwit !== undefined)
                __classPrivateFieldSet(this, _Descriptor_isSegwit, expandedResult.isSegwit, "f");
            if (expandedResult.expandedMiniscript !== undefined)
                __classPrivateFieldSet(this, _Descriptor_expandedMiniscript, expandedResult.expandedMiniscript, "f");
            if (expandedResult.redeemScript !== undefined)
                __classPrivateFieldSet(this, _Descriptor_redeemScript, expandedResult.redeemScript, "f");
            if (expandedResult.witnessScript !== undefined)
                __classPrivateFieldSet(this, _Descriptor_witnessScript, expandedResult.witnessScript, "f");
            if (signersPubKeys) {
                __classPrivateFieldSet(this, _Descriptor_signersPubKeys, signersPubKeys, "f");
            }
            else {
                if (__classPrivateFieldGet(this, _Descriptor_expansionMap, "f")) {
                    __classPrivateFieldSet(this, _Descriptor_signersPubKeys, Object.values(__classPrivateFieldGet(this, _Descriptor_expansionMap, "f")).map(keyInfo => {
                        const pubkey = keyInfo.pubkey;
                        if (!pubkey)
                            throw new Error(`Error: could not extract a pubkey from ${expression}`);
                        return pubkey;
                    }), "f");
                }
                else {
                    //We should only miss expansionMap in addr() expressions:
                    if (!expandedResult.canonicalExpression.match(RE.reAddrAnchored)) {
                        throw new Error(`Error: expansionMap not available for expression ${expression} that is not an address`);
                    }
                    __classPrivateFieldSet(this, _Descriptor_signersPubKeys, [this.getScriptPubKey()], "f");
                }
            }
        }
        getPayment() {
            return __classPrivateFieldGet(this, _Descriptor_payment, "f");
        }
        /**
         * Returns the Bitcoin Address
         */
        getAddress() {
            if (!__classPrivateFieldGet(this, _Descriptor_payment, "f").address)
                throw new Error(`Error: could extract an address from the payment`);
            return __classPrivateFieldGet(this, _Descriptor_payment, "f").address;
        }
        getScriptPubKey() {
            if (!__classPrivateFieldGet(this, _Descriptor_payment, "f").output)
                throw new Error(`Error: could extract output.script from the payment`);
            return __classPrivateFieldGet(this, _Descriptor_payment, "f").output;
        }
        /**
         * Returns the compiled script satisfaction
         * @param {PartialSig[]} signatures An array of signatures using this format: `interface PartialSig { pubkey: Buffer; signature: Buffer; }`
         * @returns {Buffer}
         */
        getScriptSatisfaction(signatures) {
            const miniscript = __classPrivateFieldGet(this, _Descriptor_miniscript, "f");
            const expandedMiniscript = __classPrivateFieldGet(this, _Descriptor_expandedMiniscript, "f");
            const expansionMap = __classPrivateFieldGet(this, _Descriptor_expansionMap, "f");
            if (miniscript === undefined ||
                expandedMiniscript === undefined ||
                expansionMap === undefined)
                throw new Error(`Error: cannot get satisfaction from not expanded miniscript ${miniscript}`);
            //Note that we pass the nLockTime and nSequence that is deduced
            //using preimages and signersPubKeys.
            //satisfyMiniscript will make sure
            //that the actual solution given, using real signatures, still meets the
            //same nLockTime and nSequence constraints
            const scriptSatisfaction = (0, miniscript_1.satisfyMiniscript)({
                expandedMiniscript,
                expansionMap,
                signatures,
                preimages: __classPrivateFieldGet(this, _Descriptor_preimages, "f"),
                //Here we pass the TimeConstraints obtained using signersPubKeys to
                //verify that the solutions found using the final signatures have not
                //changed
                timeConstraints: {
                    nLockTime: this.getLockTime(),
                    nSequence: this.getSequence()
                }
            }).scriptSatisfaction;
            if (!scriptSatisfaction)
                throw new Error(`Error: could not produce a valid satisfaction`);
            return scriptSatisfaction;
        }
        getSequence() {
            return __classPrivateFieldGet(this, _Descriptor_instances, "m", _Descriptor_getTimeConstraints).call(this)?.nSequence;
        }
        getLockTime() {
            return __classPrivateFieldGet(this, _Descriptor_instances, "m", _Descriptor_getTimeConstraints).call(this)?.nLockTime;
        }
        getWitnessScript() {
            return __classPrivateFieldGet(this, _Descriptor_witnessScript, "f");
        }
        getRedeemScript() {
            return __classPrivateFieldGet(this, _Descriptor_redeemScript, "f");
        }
        getNetwork() {
            return __classPrivateFieldGet(this, _Descriptor_network, "f");
        }
        isSegwit() {
            return __classPrivateFieldGet(this, _Descriptor_isSegwit, "f");
        }
        /**
         * Updates a Psbt where the descriptor describes an utxo.
         * The txHex (nonWitnessUtxo) and vout of the utxo must be passed.
         *
         * updatePsbt adds an input to the psbt and updates the tx locktime if needed.
         * It also adds a new input to the Psbt based on txHex
         * It returns the number of the input that is added.
         * psbt and vout are mandatory. Also pass txHex.
         *
         * The following is not recommended but, alternatively, ONLY for Segwit inputs,
         * you can pass txId and value, instead of txHex.
         * If you do so, it is your responsibility to make sure that `value` is
         * correct to avoid possible fee vulnerability attacks:
         * https://github.com/bitcoinjs/bitcoinjs-lib/issues/1625
         * Note that HW wallets require the full txHex also for Segwit anyways:
         * https://blog.trezor.io/details-of-firmware-updates-for-trezor-one-version-1-9-1-and-trezor-model-t-version-2-3-1-1eba8f60f2dd
         *
         * In doubt, simply pass txHex (and you can skip passing txId and value) and
         * you shall be fine.
         */
        updatePsbt({ psbt, txHex, txId, value, vout //vector output index
         }) {
            if (txHex === undefined) {
                console.warn(`Warning: missing txHex may allow fee attacks`);
            }
            const isSegwit = this.isSegwit();
            if (isSegwit === undefined) {
                //This should only happen when using addr() expressions
                throw new Error(`Error: could not determine whether this is a segwit descriptor`);
            }
            return (0, psbt_1.updatePsbt)({
                psbt,
                vout,
                ...(txHex !== undefined ? { txHex } : {}),
                ...(txId !== undefined ? { txId } : {}),
                ...(value !== undefined ? { value } : {}),
                sequence: this.getSequence(),
                locktime: this.getLockTime(),
                keysInfo: __classPrivateFieldGet(this, _Descriptor_expansionMap, "f") ? Object.values(__classPrivateFieldGet(this, _Descriptor_expansionMap, "f")) : [],
                scriptPubKey: this.getScriptPubKey(),
                isSegwit,
                witnessScript: this.getWitnessScript(),
                redeemScript: this.getRedeemScript()
            });
        }
        finalizePsbtInput({ index, psbt, validate = true }) {
            if (validate &&
                !psbt.validateSignaturesOfInput(index, signatureValidator)) {
                throw new Error(`Error: invalid signatures on input ${index}`);
            }
            //An index must be passed since finding the index in the psbt cannot be
            //done:
            //Imagine the case where you received money twice to
            //the same miniscript-based address. You would have the same scriptPubKey,
            //same sequences, ... The descriptor does not store the hash of the previous
            //transaction since it is a general Descriptor object. Indices must be kept
            //out of the scope of this class and then passed.
            const signatures = psbt.data.inputs[index]?.partialSig;
            if (!signatures)
                throw new Error(`Error: cannot finalize without signatures`);
            __classPrivateFieldGet(this, _Descriptor_instances, "m", _Descriptor_assertPsbtInput).call(this, { index, psbt });
            if (!__classPrivateFieldGet(this, _Descriptor_miniscript, "f")) {
                //Use standard finalizers
                psbt.finalizeInput(index);
            }
            else {
                const scriptSatisfaction = this.getScriptSatisfaction(signatures);
                psbt.finalizeInput(index, (0, psbt_1.finalScriptsFuncFactory)(scriptSatisfaction, __classPrivateFieldGet(this, _Descriptor_network, "f")));
            }
        }
        expand() {
            return {
                ...(__classPrivateFieldGet(this, _Descriptor_expandedExpression, "f") !== undefined
                    ? { expandedExpression: __classPrivateFieldGet(this, _Descriptor_expandedExpression, "f") }
                    : {}),
                ...(__classPrivateFieldGet(this, _Descriptor_miniscript, "f") !== undefined
                    ? { miniscript: __classPrivateFieldGet(this, _Descriptor_miniscript, "f") }
                    : {}),
                ...(__classPrivateFieldGet(this, _Descriptor_expandedMiniscript, "f") !== undefined
                    ? { expandedMiniscript: __classPrivateFieldGet(this, _Descriptor_expandedMiniscript, "f") }
                    : {}),
                ...(__classPrivateFieldGet(this, _Descriptor_expansionMap, "f") !== undefined
                    ? { expansionMap: __classPrivateFieldGet(this, _Descriptor_expansionMap, "f") }
                    : {})
            };
        }
    }
    _Descriptor_payment = new WeakMap(), _Descriptor_preimages = new WeakMap(), _Descriptor_signersPubKeys = new WeakMap(), _Descriptor_miniscript = new WeakMap(), _Descriptor_witnessScript = new WeakMap(), _Descriptor_redeemScript = new WeakMap(), _Descriptor_isSegwit = new WeakMap(), _Descriptor_expandedExpression = new WeakMap(), _Descriptor_expandedMiniscript = new WeakMap(), _Descriptor_expansionMap = new WeakMap(), _Descriptor_network = new WeakMap(), _Descriptor_instances = new WeakSet(), _Descriptor_getTimeConstraints = function _Descriptor_getTimeConstraints() {
        const miniscript = __classPrivateFieldGet(this, _Descriptor_miniscript, "f");
        const preimages = __classPrivateFieldGet(this, _Descriptor_preimages, "f");
        const expandedMiniscript = __classPrivateFieldGet(this, _Descriptor_expandedMiniscript, "f");
        const expansionMap = __classPrivateFieldGet(this, _Descriptor_expansionMap, "f");
        const signersPubKeys = __classPrivateFieldGet(this, _Descriptor_signersPubKeys, "f");
        //Create a method. solvePreimages to solve them.
        if (miniscript) {
            if (expandedMiniscript === undefined || expansionMap === undefined)
                throw new Error(`Error: cannot get time constraints from not expanded miniscript ${miniscript}`);
            //We create some fakeSignatures since we may not have them yet.
            //We only want to retrieve the nLockTime and nSequence of the satisfaction and
            //signatures don't matter
            const fakeSignatures = signersPubKeys.map(pubkey => ({
                pubkey,
                signature: Buffer.alloc(64, 0)
            }));
            const { nLockTime, nSequence } = (0, miniscript_1.satisfyMiniscript)({
                expandedMiniscript,
                expansionMap,
                signatures: fakeSignatures,
                preimages
            });
            return { nLockTime, nSequence };
        }
        else
            return undefined;
    }, _Descriptor_assertPsbtInput = function _Descriptor_assertPsbtInput({ psbt, index }) {
        const input = psbt.data.inputs[index];
        const txInput = psbt.txInputs[index];
        if (!input || !txInput)
            throw new Error(`Error: invalid input or txInput`);
        const { sequence: inputSequence, index: vout } = txInput;
        let scriptPubKey;
        if (input.witnessUtxo)
            scriptPubKey = input.witnessUtxo.script;
        else {
            if (!input.nonWitnessUtxo)
                throw new Error(`Error: input should have either witnessUtxo or nonWitnessUtxo`);
            const tx = bitcoinjs_lib_1.Transaction.fromBuffer(input.nonWitnessUtxo);
            const out = tx.outs[vout];
            if (!out)
                throw new Error(`Error: utxo should exist`);
            scriptPubKey = out.script;
        }
        const locktime = this.getLockTime() || 0;
        let sequence = this.getSequence();
        if (sequence === undefined && locktime !== 0)
            sequence = 0xfffffffe;
        if (sequence === undefined && locktime === 0)
            sequence = 0xffffffff;
        if (Buffer.compare(scriptPubKey, this.getScriptPubKey()) !== 0 ||
            sequence !== inputSequence ||
            locktime !== psbt.locktime ||
            this.getWitnessScript() !== input.witnessScript ||
            this.getRedeemScript() !== input.redeemScript) {
            throw new Error(`Error: cannot finalize psbt index ${index} since it does not correspond to this descriptor`);
        }
    };
    return { Descriptor, parseKeyExpression, expand, ECPair, BIP32 };
}
exports.DescriptorsFactory = DescriptorsFactory;
