package ch.rmy.android.http_shortcuts.variables

import ch.rmy.android.http_shortcuts.data.domains.variables.VariableKeyOrId
import ch.rmy.android.http_shortcuts.data.enums.HttpMethod
import ch.rmy.android.http_shortcuts.data.models.GlobalVariable
import ch.rmy.android.http_shortcuts.data.models.Shortcut
import ch.rmy.android.http_shortcuts.variables.types.VariableTypeFactory
import ch.rmy.android.testutils.DefaultModels
import io.mockk.coEvery
import io.mockk.every
import io.mockk.impl.annotations.RelaxedMockK
import io.mockk.junit5.MockKExtension
import io.mockk.mockk
import kotlin.test.BeforeTest
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.test.runTest
import org.junit.jupiter.api.extension.ExtendWith

@ExperimentalCoroutinesApi
@ExtendWith(MockKExtension::class)
class VariableResolverTest {

    @RelaxedMockK
    private lateinit var variableTypeFactory: VariableTypeFactory

    private val resolutionOrder = mutableListOf<String>()

    @BeforeTest
    fun setUp() {
        resolutionOrder.clear()
        every { variableTypeFactory.getType(any()) } answers {
            mockk {
                coEvery { resolve(any(), any()) } answers {
                    val variable = firstArg<GlobalVariable>()
                    resolutionOrder.add(variable.id)
                    variable.value.orEmpty()
                }
            }
        }
    }

    @Test
    fun `test variable resolution of static variables`() = runTest {
        val variableManager = VariableManager(
            listOf(
                variable(id = "1234", key = "myVariable", value = "Hello World"),
            ),
        )
        VariableResolver(variableTypeFactory)
            .resolve(
                variableManager = variableManager,
                variableKeysOrIds = VariableResolver.findResolvableVariableIdentifiersExcludingScripting(
                    shortcutWithContent("{{1234}}"),
                    headers = emptyList(),
                    parameters = emptyList(),
                ),
                dialogHandle = mockk(),
            )

        val values = variableManager.getVariableValues()
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("1234")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("myVariable")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("1234")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("myVariable")],
        )
    }

    @Test
    fun `test variable resolution of static variables referencing other static variables`() = runTest {
        val variableManager = VariableManager(
            listOf(
                variable(id = "1234", key = "myVariable1", value = "Hello {{5678}}"),
                variable(id = "5678", key = "myVariable2", value = "World"),
            ),
        )
        VariableResolver(variableTypeFactory)
            .resolve(
                variableManager = variableManager,
                variableKeysOrIds = VariableResolver.findResolvableVariableIdentifiersExcludingScripting(
                    shortcutWithContent("{{1234}}"),
                    headers = emptyList(),
                    parameters = emptyList(),
                ),
                dialogHandle = mockk(),
            )

        val values = variableManager.getVariableValues()
        assertEquals(
            "World",
            values[VariableKeyOrId("5678")],
        )
        assertEquals(
            "World",
            values[VariableKeyOrId("myVariable2")],
        )
        assertEquals(
            "World",
            values[VariableKeyOrId("5678")],
        )
        assertEquals(
            "World",
            values[VariableKeyOrId("myVariable2")],
        )

        assertEquals(
            "Hello World",
            values[VariableKeyOrId("1234")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("myVariable1")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("1234")],
        )
        assertEquals(
            "Hello World",
            values[VariableKeyOrId("myVariable1")],
        )
    }

    @Test
    fun `test variable resolution of static variable references in JS code`() {
        val shortcut = shortcutWithJSContent(
            content = """
            const foo = getVariable(/*[variable]*/"1234"/*[/variable]*/);
            getVariable("my_variable");
            """.trimIndent(),
        )
        val globalVariableIds = VariableResolver.findResolvableVariableIdentifiersIncludingScripting(
            shortcut,
            headers = emptyList(),
            parameters = emptyList(),
        )

        assertEquals(
            setOf(VariableKeyOrId("1234"), VariableKeyOrId("my_variable")),
            globalVariableIds,
        )
    }

    @Test
    fun `test variable resolution order`() = runTest {
        val variableManager = VariableManager(
            listOf(
                variable(id = "123", key = "myVariable1", value = "Hello {{789}}"),
                variable(id = "456", key = "myVariable2", value = "!!!"),
                variable(id = "789", key = "myVariable3", value = "World"),
            ),
        )
        VariableResolver(variableTypeFactory)
            .resolve(
                variableManager = variableManager,
                variableKeysOrIds = setOf(VariableKeyOrId("123"), VariableKeyOrId("456")),
                dialogHandle = mockk(),
            )

        assertEquals(
            listOf("123", "789", "456"),
            resolutionOrder,
        )
        assertEquals(
            ResolvedVariableValues(
                globalVariableValues = mapOf(
                    "123" to "Hello World",
                    "456" to "!!!",
                    "789" to "World",
                ),
                localVariablesValues = emptyMap(),
                globalVariableKeysToIds = mapOf(
                    "myVariable1" to "123",
                    "myVariable2" to "456",
                    "myVariable3" to "789",
                ),
            ),
            variableManager.getVariableValues(),
        )
    }

    @Test
    fun `test multi-level recursion variable`() = runTest {
        val variableManager = VariableManager(
            listOf(
                variable(id = "123", key = "myVariable1", value = "Hello {{456}}"),
                variable(id = "456", key = "myVariable2", value = "World{{789}}"),
                variable(id = "789", key = "myVariable3", value = "!!!"),
            ),
        )
        VariableResolver(variableTypeFactory)
            .resolve(
                variableManager = variableManager,
                variableKeysOrIds = setOf(VariableKeyOrId("123")),
                dialogHandle = mockk(),
            )

        assertEquals(
            mapOf(
                "123" to "Hello World!!!",
                "456" to "World!!!",
                "789" to "!!!",
            ),
            variableManager.getVariableValues().globalVariableValues,
        )
    }

    @Test
    fun `test self-referential variable`() = runTest {
        val variableManager = VariableManager(
            listOf(
                variable(id = "123", key = "myVariable1", value = "Hello {{123}}"),
            ),
        )
        VariableResolver(variableTypeFactory)
            .resolve(
                variableManager = variableManager,
                variableKeysOrIds = setOf(VariableKeyOrId("123")),
                dialogHandle = mockk(),
            )

        assertEquals(
            mapOf(
                "123" to "Hello Hello Hello {{123}}",
            ),
            variableManager.getVariableValues().globalVariableValues,
        )
    }

    companion object {

        private fun variable(id: String, key: String, value: String): GlobalVariable =
            DefaultModels.variable.copy(
                id = id,
                key = key,
                value = value,
            )

        private fun shortcutWithContent(content: String): Shortcut =
            DefaultModels.shortcut.copy(
                method = HttpMethod.POST,
                bodyContent = content,
            )

        private fun shortcutWithJSContent(content: String): Shortcut =
            DefaultModels.shortcut.copy(
                method = HttpMethod.POST,
                codeOnSuccess = content,
            )
    }
}
