package ch.rmy.android.http_shortcuts.scripting.actions.types

import android.content.Context
import android.net.Uri
import android.webkit.MimeTypeMap
import androidx.documentfile.provider.DocumentFile
import ch.rmy.android.framework.extensions.logInfo
import ch.rmy.android.framework.extensions.takeUnlessEmpty
import ch.rmy.android.http_shortcuts.data.domains.working_directories.WorkingDirectoryRepository
import ch.rmy.android.http_shortcuts.exceptions.ActionException
import ch.rmy.android.http_shortcuts.scripting.ExecutionContext
import ch.rmy.android.http_shortcuts.utils.WorkingDirectoryUtil
import ch.rmy.android.scripting.JsObject
import dagger.hilt.android.qualifiers.ApplicationContext
import java.io.File
import java.nio.charset.Charset
import java.nio.charset.IllegalCharsetNameException
import java.nio.charset.UnsupportedCharsetException
import javax.inject.Inject

class GetDirectoryAction
@Inject
constructor(
    @ApplicationContext
    private val context: Context,
    private val workingDirectoryRepository: WorkingDirectoryRepository,
    private val workingDirectoryUtil: WorkingDirectoryUtil,
) : Action<GetDirectoryAction.Params> {
    override suspend fun Params.execute(executionContext: ExecutionContext): JsObject {
        val directoryHandle = getDirectoryHandle(directoryNameOrId, executionContext)
        val contentResolver = context.contentResolver
        return executionContext.scriptingEngine.buildJsObject {
            function("readFile") { args ->
                logInfo("directory.readFile() called")
                val filePath = args.getString(0)!!
                val encoding = args.getString(1)
                val fileUri = directoryHandle.getFileUriForReading(filePath)
                val charset = encoding?.let {
                    try {
                        Charset.forName(it)
                    } catch (_: IllegalCharsetNameException) {
                        executionContext.throwException(
                            ActionException {
                                "Invalid charset: $it"
                            },
                        )
                    } catch (_: UnsupportedCharsetException) {
                        executionContext.throwException(
                            ActionException {
                                "Unsupported charset: $it"
                            },
                        )
                    }
                } ?: Charsets.UTF_8
                contentResolver.openInputStream(fileUri)!!
                    .use {
                        it.reader(charset).readText()
                    }
            }
            function("writeFile") { args ->
                logInfo("directory.writeFile() called")
                val filePath = args.getString(0)!!
                val content = args.getByteArray(1) ?: return@function null
                val fileUri = directoryHandle.getFileUriForWriting(filePath)
                contentResolver.openOutputStream(fileUri, "wt")!!
                    .use { out ->
                        out.write(content)
                    }
            }
            function("appendFile") { args ->
                logInfo("directory.appendFile() called")
                val filePath = args.getString(0)!!
                val content = args.getByteArray(1) ?: return@function null
                val fileUri = directoryHandle.getFileUriForWriting(filePath)
                contentResolver.openOutputStream(fileUri, "wa")!!
                    .use { out ->
                        out.write(content)
                    }
            }
        }
    }

    private suspend fun getDirectoryHandle(directoryNameOrId: String, executionContext: ExecutionContext): DirectoryHandle {
        if (directoryNameOrId == TEMPORARY_DIRECTORY_ID) {
            return getTemporaryDirectoryHandle(executionContext)
        }

        val workingDirectory = try {
            workingDirectoryRepository.getWorkingDirectoryByNameOrId(directoryNameOrId)
        } catch (_: NoSuchElementException) {
            throw ActionException {
                "Directory \"${directoryNameOrId}\" not found"
            }
        }
        workingDirectoryRepository.touchWorkingDirectory(workingDirectory.id)
        val directory = workingDirectoryUtil.getDocumentFile(workingDirectory)
        if (directory == null || !directory.isDirectory) {
            throw ActionException {
                "Directory \"${workingDirectory.name}\" is not mounted"
            }
        }

        return object : DirectoryHandle {
            override fun getFileUriForReading(filePath: String): Uri =
                directory.findFileFromPath(filePath)
                    ?.uri
                    ?: executionContext.throwException(
                        ActionException {
                            "File \"$filePath\" not found in directory \"${workingDirectory.name}\""
                        },
                    )

            override fun getFileUriForWriting(filePath: String): Uri =
                directory.findOrCreateFileFromPath(filePath)
                    ?.uri
                    ?: executionContext.throwException(
                        ActionException {
                            "File \"$filePath\" not found in directory \"${workingDirectory.name}\""
                        },
                    )
        }
    }

    private fun getTemporaryDirectoryHandle(executionContext: ExecutionContext): DirectoryHandle =
        object : DirectoryHandle {
            override fun getFileUriForReading(filePath: String): Uri =
                executionContext.fileUploadResult?.getFiles()
                    ?.find { it.id == filePath || it.fileName == it.id }
                    ?.data
                    ?: executionContext.throwException(
                        ActionException {
                            "No file with name or id \"$filePath\" found"
                        },
                    )

            override fun getFileUriForWriting(filePath: String): Uri =
                executionContext.throwException(
                    ActionException {
                        "Selected files are readonly"
                    },
                )
        }

    private fun DocumentFile.findFileFromPath(filePath: String): DocumentFile? {
        var fileHandle: DocumentFile = this
        filePath.split('/').forEach { fileName ->
            fileHandle = fileHandle.findFile(fileName)
                ?: return null
        }
        return fileHandle
    }

    private fun DocumentFile.findOrCreateFileFromPath(filePath: String): DocumentFile? {
        var fileHandle: DocumentFile = this
        val parts = filePath.split('/')
        parts.forEachIndexed { index, fileName ->
            if (fileName == "." || fileName == "..") {
                return null
            }
            fileHandle = fileHandle.findFile(fileName)
                ?: (
                    if (index != parts.lastIndex) {
                        fileHandle.createDirectory(fileName)
                    } else {
                        fileHandle.createFile(
                            determineMimeType(fileName),
                            fileName,
                        )
                    }
                    )
                ?: return null
        }
        return fileHandle
    }

    private fun determineMimeType(fileName: String): String =
        File(fileName).extension.takeUnlessEmpty()?.let { extension ->
            MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension)
        }
            ?: "text/plain"

    interface DirectoryHandle {
        fun getFileUriForReading(filePath: String): Uri

        fun getFileUriForWriting(filePath: String): Uri
    }

    data class Params(
        val directoryNameOrId: String,
    )

    companion object {
        private const val TEMPORARY_DIRECTORY_ID = ""
    }
}
