/*
 * Copyright (C) 2025. Nyabsi <nyabsi@sovellus.cc>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cc.sovellus.vrcaa.ui.components.misc

import android.util.Log
import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.VectorConverter
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectHorizontalDragGestures
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.foundation.gestures.detectVerticalDragGestures
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.graphicsLayer
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.layout.onSizeChanged
import com.bumptech.glide.integration.compose.ExperimentalGlideComposeApi
import com.bumptech.glide.integration.compose.GlideImage
import kotlinx.coroutines.launch

@OptIn(ExperimentalGlideComposeApi::class)
@Composable
fun ZoomableImage(
    imageUrl: String,
    modifier: Modifier = Modifier,
    onClose: () -> Unit
) {
    val scope = rememberCoroutineScope()
    val scale = remember { Animatable(1.0f) }
    val offset = remember { Animatable(Offset.Zero, Offset.VectorConverter) }
    val containerSize = remember { mutableStateOf(androidx.compose.ui.geometry.Size.Zero) }

    val transformModifier = modifier
        .pointerInput(Unit) {
            detectTapGestures(
                onDoubleTap = {
                    scope.launch {
                        scale.animateTo(1.0f)
                        offset.animateTo(Offset.Zero)
                    }
                }
            )
        }
        .pointerInput(Unit) {
            detectTransformGestures { _, pan, zoom, _ ->
                scope.launch {
                    val newScale = (scale.value * zoom).coerceIn(1f, 2f)
                    scale.snapTo(newScale)

                    val maxX = ((containerSize.value.width * (newScale - 1)) / 2).coerceAtLeast(0f)
                    val maxY = maxX / 2

                    val newOffset = offset.value + pan
                    val clampedOffset = Offset(
                        newOffset.x.coerceIn(-maxX, maxX),
                        newOffset.y.coerceIn(-maxY, maxY)
                    )

                    offset.snapTo(clampedOffset)
                }
            }
        }
        .pointerInput(Unit) {
            detectVerticalDragGestures (
                onVerticalDrag = { _, amount ->
                    if (amount >= 100.0f)
                        onClose()
                }
            )
        }
        .onSizeChanged {
            containerSize.value = androidx.compose.ui.geometry.Size(it.width.toFloat(), it.height.toFloat())
        }
        .graphicsLayer(
            scaleX = scale.value,
            scaleY = scale.value,
            translationX = offset.value.x,
            translationY = offset.value.y
        )

    GlideImage(
        model = imageUrl,
        contentDescription = null,
        contentScale = ContentScale.None,
        modifier = transformModifier
            .fillMaxSize()
            .aspectRatio(16.0f / 10.0f)
            .background(Color.Transparent)
    )
}
