/*
 * Copyright (C) 2025. Nyabsi <nyabsi@sovellus.cc>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cc.sovellus.vrcaa.ui.components.layout

import android.graphics.Bitmap
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Android
import androidx.compose.material.icons.filled.Computer
import androidx.compose.material.icons.filled.PhoneIphone
import androidx.compose.material.icons.filled.Web
import androidx.compose.material3.Badge
import androidx.compose.material3.Icon
import androidx.compose.material3.ListItem
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import cc.sovellus.vrcaa.R
import cc.sovellus.vrcaa.api.vrchat.http.models.Friend
import cc.sovellus.vrcaa.helper.LocationHelper
import cc.sovellus.vrcaa.helper.StatusHelper
import com.bumptech.glide.integration.compose.ExperimentalGlideComposeApi
import com.bumptech.glide.integration.compose.GlideImage
import com.bumptech.glide.integration.compose.placeholder
import com.bumptech.glide.request.RequestOptions

@OptIn(ExperimentalGlideComposeApi::class)
@Composable
fun FriendItem(friend: Friend, callback: () -> Unit) {
    ListItem(
        headlineContent = {
            Text(friend.displayName)
        },
        overlineContent = {
            Text(friend.statusDescription.ifEmpty {
                if (friend.platform.isEmpty()) { StatusHelper.Status.Offline.toString() } else { StatusHelper.getStatusFromString(friend.status).toString() }
            }, maxLines = 1)
        },
        supportingContent = {
            Text(text =
                (if (friend.platform == "web") {
                    "Active on website."
                }
                else if (friend.platform.isEmpty()) {
                    "offline"
                }
                else {
                    LocationHelper.getReadableLocation(friend.location)
                }),
                maxLines = 1,
                overflow = TextOverflow.Ellipsis
            )
        },
        leadingContent = {
            Column {
                Badge(
                    containerColor = if (friend.platform.isEmpty()) { StatusHelper.Status.Offline.toColor() } else { StatusHelper.getStatusFromString(friend.status).toColor() },
                    modifier = Modifier
                        .size(64.dp)
                        .align(Alignment.CenterHorizontally)
                ) {
                    GlideImage(
                        model = friend.userIcon.ifEmpty { friend.profilePicOverride.ifEmpty { friend.currentAvatarImageUrl } },
                        contentDescription = null,
                        modifier = Modifier
                            .size(56.dp)
                            .clip(RoundedCornerShape(50)),
                        contentScale = ContentScale.Crop,
                        alignment = Alignment.Center,
                        loading = placeholder(R.drawable.image_placeholder),
                        failure = placeholder(R.drawable.image_placeholder),
                        requestBuilderTransform = { requestBuilder ->
                            requestBuilder
                                .apply(
                                    RequestOptions()
                                        .override(64, 64)
                                        .dontAnimate()
                                        .encodeFormat(Bitmap.CompressFormat.JPEG)
                                        .encodeQuality(70)
                                )
                        }
                    )
                }
            }
        },
        trailingContent = {
            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                verticalArrangement = Arrangement.Center,
                modifier = Modifier.size(20.dp)
            ) {
                when (friend.platform) {
                    "standalonewindows" -> {
                        Icon(Icons.Default.Computer, null, modifier = Modifier.size(20.dp).align(Alignment.CenterHorizontally))
                    }
                    "android" -> {
                        Icon(Icons.Default.Android, null, modifier = Modifier.size(20.dp).align(Alignment.CenterHorizontally))
                    }
                    "ios" -> {
                        Icon(Icons.Default.PhoneIphone, null, modifier = Modifier.size(20.dp).align(Alignment.CenterHorizontally))
                    }
                    "web" -> {
                        Icon(Icons.Default.Web, null, modifier = Modifier.size(20.dp).align(Alignment.CenterHorizontally))
                    }
                }
            }
        },
        modifier = Modifier.clickable(
            onClick = {
                callback()
            }
        )
    )
}