/*
 * Copyright (C) 2025. Nyabsi <nyabsi@sovellus.cc>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cc.sovellus.vrcaa.ui.components.dialog

import android.annotation.SuppressLint
import android.net.Uri
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.ExitTransition
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideOutHorizontally
import androidx.compose.animation.slideOutVertically
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Download
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import cc.sovellus.vrcaa.App
import cc.sovellus.vrcaa.R
import cc.sovellus.vrcaa.ui.components.misc.ZoomableImage
import com.bumptech.glide.Glide
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

private fun saveGlideImageToPickedUri(imageUrl: String, pickedUri: Uri) {
    CoroutineScope(Dispatchers.IO).launch {
        try {
            val file = Glide.with(App.getContext())
                .asFile()
                .load(imageUrl)
                .submit()
                .get()

            App.getContext().contentResolver.openOutputStream(pickedUri)?.use { outputStream ->
                file.inputStream().copyTo(outputStream)
            }

            withContext(Dispatchers.Main) {
                Toast.makeText(
                    App.getContext(),
                    App.getContext().getString(R.string.image_preview_toast_picture_saved),
                    Toast.LENGTH_SHORT
                ).show()
            }
        } catch (e: Exception) {
            e.printStackTrace()
            withContext(Dispatchers.Main) {
                Toast.makeText(
                    App.getContext(),
                    App.getContext().getString(R.string.image_preview_toast_picture_save_failed),
                    Toast.LENGTH_SHORT
                ).show()
            }
        }
    }
}

@SuppressLint("UnusedMaterial3ScaffoldPaddingParameter")
@OptIn(ExperimentalMaterial3Api::class, ExperimentalAnimationApi::class)
@Composable
fun ImagePreviewDialog(
    name: String,
    url: String,
    onDismiss: () -> Unit
) {
    var pendingImageUrl by remember { mutableStateOf<String?>(null) }
    val launcher = rememberLauncherForActivityResult(ActivityResultContracts.CreateDocument("image/jpeg")) { uri ->
        uri?.let {
            pendingImageUrl?.let { imageUrl ->
                saveGlideImageToPickedUri(imageUrl, it)
            }
            pendingImageUrl = null
        }
    }

    var isVisible by remember { mutableStateOf(true) }
    var slideExit by remember { mutableStateOf(false) }

    LaunchedEffect(isVisible) {
        if (!isVisible) {
            delay(250)
            onDismiss()
        }
    }

    Dialog(
        onDismissRequest = { isVisible = false },
        properties = DialogProperties(
            usePlatformDefaultWidth = false,
            decorFitsSystemWindows = false
        )
    ) {
        AnimatedVisibility(
            visible = isVisible,
            exit = if (slideExit)
                slideOutVertically(targetOffsetY = { it / 2 }) + fadeOut()
            else
                slideOutHorizontally(targetOffsetX = { it }) + fadeOut()
        ) {
            Scaffold(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(0.dp),
                containerColor = Color.Transparent,
                contentWindowInsets = WindowInsets(0),
                topBar = {
                    TopAppBar(
                        colors = TopAppBarDefaults.topAppBarColors(containerColor = Color.Transparent),
                        navigationIcon = {
                            IconButton(onClick = { isVisible = false }) {
                                Icon(
                                    imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = null,
                                    tint = Color.White
                                )
                            }
                        },
                        actions = {
                            IconButton(onClick = {
                                pendingImageUrl = url
                                launcher.launch("${name}.png")
                            }) {
                                Icon(
                                    imageVector = Icons.Default.Download,
                                    contentDescription = null,
                                    tint = Color.White
                                )
                            }
                        },
                        title = {
                            Text(
                                text = stringResource(R.string.image_preview_dialog_title),
                                color = Color.White
                            )
                        }
                    )
                }
            ) { innerPadding ->
                Box(
                    modifier = Modifier
                        .fillMaxSize()
                        .padding(top = innerPadding.calculateTopPadding())
                ) {
                    ZoomableImage(
                        imageUrl = url,
                        modifier = Modifier.fillMaxSize()
                    ) {
                        slideExit = true
                        isVisible = false
                    }
                }
            }
        }
    }
}
