import 'package:meta/meta.dart';

import '../sentry.dart';

@internal
class PropagationContext {
  /// Either represents the incoming `traceId` or the `traceId` generated by the current SDK, if there was no incoming trace.
  SentryId traceId = SentryId.newId();

  /// The dynamic sampling context.
  SentryBaggage? baggage;

  bool? _sampled;

  /// Indicates whether the current trace is sampled or not.
  ///
  /// This flag follows the lifecycle of a trace:
  /// * It starts as `null` (undecided).
  /// * The **first** transaction that receives a sampling decision (root
  ///   transaction) sets the flag to the decided value. Subsequent
  ///   transactions for the same trace MUST NOT change the value.
  /// * When a new trace is started (i.e. when a new `traceId` is generated),
  ///   the flag is reset back to `null`.
  ///
  /// The flag is propagated via the `sentry-trace` header so that downstream
  /// services can honour the original sampling decision.
  bool? get sampled => _sampled;

  /// Applies the sampling decision exactly once per trace.
  void applySamplingDecision(bool sampled) {
    _sampled ??= sampled;
  }

  /// Random number generated for sampling decisions.
  ///
  /// This value must be generated **once per trace** and reused across all
  /// child spans and transactions that belong to the same trace. It is reset
  /// whenever a new trace is started.
  double? sampleRand;

  /// Starts a brand-new trace (new ID, new sampling value & sampled state).
  void resetTrace() {
    traceId = SentryId.newId();
    sampleRand = null;
    _sampled = null;
  }

  /// Baggage header to attach to http headers.
  SentryBaggageHeader? toBaggageHeader() =>
      baggage != null ? SentryBaggageHeader.fromBaggage(baggage!) : null;

  /// Sentry trace header to attach to http headers.
  SentryTraceHeader toSentryTrace() => generateSentryTraceHeader(
        traceId: traceId,
        sampled: sampled,
      );
}
