package com.opengps.locationsharing

import androidx.compose.runtime.Composable
import androidx.compose.ui.window.ComposeUIViewController
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.room.Room
import androidx.sqlite.driver.bundled.BundledSQLiteDriver
import kotlinx.cinterop.ExperimentalForeignApi
import platform.Contacts.CNContact
import platform.ContactsUI.CNContactPickerDelegateProtocol
import platform.ContactsUI.CNContactPickerViewController
import platform.CoreBluetooth.CBCentralManager
import platform.CoreBluetooth.CBCentralManagerDelegateProtocol
import platform.CoreBluetooth.CBPeripheral
import platform.Foundation.NSCachesDirectory
import platform.Foundation.NSDocumentDirectory
import platform.Foundation.NSFileManager
import platform.Foundation.NSLibraryDirectory
import platform.Foundation.NSNumber
import platform.Foundation.NSSearchPathForDirectoriesInDomains
import platform.Foundation.NSURL
import platform.Foundation.NSUserDomainMask
import platform.UIKit.UIApplication
import platform.UIKit.UIDevice
import platform.UIKit.UINavigationController
import platform.UIKit.UIPasteboard
import platform.UIKit.UITabBarController
import platform.UIKit.UIViewController
import platform.UserNotifications.UNAuthorizationOptionAlert
import platform.UserNotifications.UNMutableNotificationContent
import platform.UserNotifications.UNNotification
import platform.UserNotifications.UNNotificationPresentationOptionBanner
import platform.UserNotifications.UNNotificationPresentationOptions
import platform.UserNotifications.UNNotificationRequest
import platform.UserNotifications.UNUserNotificationCenter
import platform.UserNotifications.UNUserNotificationCenterDelegateProtocol
import platform.darwin.NSObject
import kotlin.random.Random
import kotlin.random.nextULong

class IOSPlatform: Platform() {

    @OptIn(ExperimentalForeignApi::class)
    override val dataStore: DataStore<Preferences> = createDataStore(
    producePath = {
        val documentDirectory: NSURL? = NSFileManager.defaultManager.URLForDirectory(
            directory = NSDocumentDirectory,
            inDomain = NSUserDomainMask,
            appropriateForURL = null,
            create = false,
            error = null,
        )
        requireNotNull(documentDirectory).path + "/$dataStoreFileName"
    }
    )
    override val database = Room.databaseBuilder<AppDatabase>(
        name = documentDirectory() + "/my_room.db",
    ).setDriver(BundledSQLiteDriver()).addMigrations(*migrations).build()

    private val contactPicker = CNContactPickerViewController()

    @Composable
    override fun requestPickContact(callback: (String, String?) -> Unit): () -> Unit {
        return {
            contactPicker.delegate = object : NSObject(), CNContactPickerDelegateProtocol {
                override fun contactPickerDidCancel(picker: CNContactPickerViewController) {
                    contactPicker.delegate = null
                    picker.dismissViewControllerAnimated(true, null)
                }

                override fun contactPicker(
                    picker: CNContactPickerViewController,
                    didSelectContact: CNContact,
                ) {
                    //val id = didSelectContact.identifier
                    val name = "${didSelectContact.givenName} ${didSelectContact.familyName}".trim()

                    //TODO: add image
                    callback(name, null)
                    contactPicker.delegate = null
                    picker.dismissViewControllerAnimated(true, null)

                }
            }

            UIViewController.topMostViewController()
                ?.presentViewController(contactPicker, true, null)
        }
    }

    val cbmanager = CBCentralManager()
    override fun startScanBluetoothDevices(setRSSI: (String, Int) -> Unit): () -> Unit {
        cbmanager.delegate = object : NSObject(), CBCentralManagerDelegateProtocol {
            override fun centralManager(
                central: CBCentralManager,
                didDiscoverPeripheral: CBPeripheral,
                advertisementData: Map<Any?, *>,
                RSSI: NSNumber
            ) {
                val address = didDiscoverPeripheral.identifier.UUIDString
                val name = didDiscoverPeripheral.name?: return
                if(!nearBluetoothDevices.any { it.name == name })
                    nearBluetoothDevices.add(BluetoothDevice(Random.nextULong(), name))
                setRSSI(address, RSSI.intValue)
            }

            override fun centralManagerDidUpdateState(central: CBCentralManager) {}
        }
        cbmanager.scanForPeripheralsWithServices(null, null)
        return {cbmanager.stopScan()}
    }

    override fun runBackgroundService() {
        BackgroundService()
        UIDevice.currentDevice.batteryMonitoringEnabled = true
    }

    override fun createNotification(title: String, body: String, channelId: String) {
        val content = UNMutableNotificationContent()
        content.setTitle(title)
        content.setBody(body)
        val uuidString = Random.nextLong().toString()
        val request = UNNotificationRequest.requestWithIdentifier(uuidString, content, null)
        val notificationCenter = UNUserNotificationCenter.currentNotificationCenter()
        notificationCenter.delegate = object : NSObject(), UNUserNotificationCenterDelegateProtocol {
            override fun userNotificationCenter(
                center: UNUserNotificationCenter,
                willPresentNotification: UNNotification,
                withCompletionHandler: (UNNotificationPresentationOptions) -> Unit
            ) {
                withCompletionHandler(UNNotificationPresentationOptionBanner)
            }
        }
        notificationCenter.requestAuthorizationWithOptions(UNAuthorizationOptionAlert) { _, _ ->
            notificationCenter.addNotificationRequest(request, null)
        }
    }

    override fun copyToClipboard(text: String) {
        val pasteboard = UIPasteboard.generalPasteboard()
        pasteboard.string = text
    }

    override val batteryLevel: Float
        get() = UIDevice.currentDevice.batteryLevel * 100

    override val name: String = "IOS"
}

@OptIn(ExperimentalForeignApi::class)
private fun documentDirectory(): String {
    val documentDirectory = NSFileManager.defaultManager.URLForDirectory(
        directory = NSDocumentDirectory,
        inDomain = NSUserDomainMask,
        appropriateForURL = null,
        create = false,
        error = null,
    )
    return requireNotNull(documentDirectory?.path)
}

private fun UIViewController.Companion.topMostViewController(): UIViewController? {
    return findTopMostViewController(UIApplication.sharedApplication.keyWindow?.rootViewController)
}

private fun findTopMostViewController(rootViewController: UIViewController?): UIViewController? {
    if (rootViewController?.presentedViewController == null) {
        return rootViewController
    }

    if (rootViewController.presentedViewController is UINavigationController) {
        val navigationController =
            rootViewController.presentedViewController as UINavigationController
        return navigationController.visibleViewController ?: navigationController
    }

    if (rootViewController.presentedViewController is UITabBarController) {
        val tabBarController = rootViewController.presentedViewController as UITabBarController
        return tabBarController.selectedViewController ?: tabBarController
    }

    return null
}

actual var platformInternal: Platform? = IOSPlatform()

fun MainViewController(): UIViewController =
    ComposeUIViewController {
        Main()
    }
