package cc.calliope.mini.ui.dialog.scripts;

import android.app.Dialog;
import android.content.Context;
import android.content.Intent;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.ParcelFileDescriptor;
import android.os.storage.StorageManager;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.TextView;

import com.google.android.material.bottomsheet.BottomSheetBehavior;
import com.google.android.material.bottomsheet.BottomSheetDialog;
import com.google.android.material.bottomsheet.BottomSheetDialogFragment;

import org.apache.commons.io.FilenameUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.appcompat.widget.PopupMenu;
import androidx.core.content.FileProvider;
import androidx.documentfile.provider.DocumentFile;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import cc.calliope.mini.core.state.State;
import cc.calliope.mini.core.service.FlashingService;
import cc.calliope.mini.utils.file.FileWrapper;
import cc.calliope.mini.R;
import cc.calliope.mini.ui.activity.FlashingActivity;
import cc.calliope.mini.ui.dialog.DialogUtils;
import cc.calliope.mini.core.state.ApplicationStateHandler;
import cc.calliope.mini.utils.Constants;
import cc.calliope.mini.databinding.FragmentScriptsBinding;
import cc.calliope.mini.ui.model.EditorType;
import cc.calliope.mini.utils.settings.Settings;
import cc.calliope.mini.utils.Utils;

import static android.app.Activity.RESULT_OK;
import static cc.calliope.mini.core.state.Notification.ERROR;
import static cc.calliope.mini.core.state.Notification.INFO;

public class ScriptsFragment extends BottomSheetDialogFragment {
    private static final String TAG = "Scripts";
    private static final String FILE_EXTENSION = ".hex";

    private FragmentScriptsBinding binding;
    private FragmentActivity activity;
    private ScriptsRecyclerAdapter scriptsRecyclerAdapter;
    private FrameLayout bottomSheet;
    private int state = BottomSheetBehavior.STATE_COLLAPSED;
    private String sourceFilePath;

    private final BottomSheetBehavior.BottomSheetCallback bottomSheetCallback =
            new BottomSheetBehavior.BottomSheetCallback() {
                @Override
                public void onStateChanged(@NonNull View bottomSheet, int newState) {
                    state = newState;
                }

                @Override
                public void onSlide(@NonNull View bottomSheet, float slideOffset) {
                }
            };

    @NonNull
    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        BottomSheetDialog dialog = (BottomSheetDialog) super.onCreateDialog(savedInstanceState);
        dialog.setOnShowListener(di -> {
            BottomSheetDialog d = (BottomSheetDialog) di;
            bottomSheet = d.findViewById(com.google.android.material.R.id.design_bottom_sheet);
            if (bottomSheet != null) {
                BottomSheetBehavior.from(bottomSheet).addBottomSheetCallback(bottomSheetCallback);
                bottomSheet.setBackgroundColor(android.graphics.Color.TRANSPARENT);
            }
        });
        return dialog;
    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater,
                             ViewGroup container, Bundle savedInstanceState) {
        binding = FragmentScriptsBinding.inflate(inflater, container, false);
        activity = requireActivity();
        return binding.getRoot();
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        BottomSheetBehavior.from(bottomSheet).removeBottomSheetCallback(bottomSheetCallback);
        binding = null;
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        ArrayList<FileWrapper> filesList = new ArrayList<>();
        for (EditorType editor : EditorType.values()) {
            filesList.addAll(getFiles(editor));
        }
        TextView infoTextView = binding.infoTextView;
        RecyclerView recyclerView = binding.scriptsRecyclerView;

        if (filesList.isEmpty()) {
            infoTextView.setVisibility(View.VISIBLE);
            recyclerView.setVisibility(View.INVISIBLE);
        } else {
            infoTextView.setVisibility(View.INVISIBLE);
            recyclerView.setVisibility(View.VISIBLE);
            recyclerView.setItemAnimator(new DefaultItemAnimator());
            recyclerView.setLayoutManager(new LinearLayoutManager(getContext()));
            scriptsRecyclerAdapter = new ScriptsRecyclerAdapter(filesList);
            scriptsRecyclerAdapter.setOnItemClickListener(this::openDfuActivity);
            scriptsRecyclerAdapter.setOnItemLongClickListener(this::openPopupMenu);
            recyclerView.setAdapter(scriptsRecyclerAdapter);
        }
    }

    private ArrayList<FileWrapper> getFiles(EditorType editor) {
        File[] filesArray = new File(activity.getFilesDir().toString() + File.separator + editor).listFiles();
        ArrayList<FileWrapper> filesList = new ArrayList<>();
        if (filesArray != null) {
            for (File file : filesArray) {
                String name = file.getName();
                if (name.contains(FILE_EXTENSION)) {
                    filesList.add(new FileWrapper(file, editor));
                }
            }
        }
        return filesList;
    }

    private void openDfuActivity(FileWrapper file) {
        if (!Utils.isBluetoothEnabled()) {
            ApplicationStateHandler.updateNotification(ERROR, getString(R.string.error_snackbar_bluetooth_disabled));
            dismiss();
            return;
        }
        if (ApplicationStateHandler.getDeviceAvailabilityLiveData().getValue() == null
                || !ApplicationStateHandler.getDeviceAvailabilityLiveData().getValue()) {
            ApplicationStateHandler.updateNotification(ERROR, R.string.error_no_connected);
            return;
        }
        if (!Settings.isBackgroundFlashingEnable(activity)) {
            final Intent intent = new Intent(activity, FlashingActivity.class);
            intent.putExtra(Constants.EXTRA_FILE_PATH, file.getAbsolutePath());
            startActivity(intent);
        }
        Intent serviceIntent = new Intent(activity, FlashingService.class);
        serviceIntent.putExtra(Constants.EXTRA_FILE_PATH, file.getAbsolutePath());
        activity.startService(serviceIntent);
        dismiss();
    }

    private void openPopupMenu(View view, FileWrapper file) {
        PopupMenu popup = new PopupMenu(view.getContext(), view);
        popup.setOnMenuItemClickListener(item -> {
            int id = item.getItemId();
            if (id == R.id.copy) {
                copyFile(file);
                return true;
            } else if (id == R.id.share) {
                shareFile(file);
                return true;
            } else if (id == R.id.rename) {
                renameFile(file);
                return true;
            } else if (id == R.id.remove) {
                removeFile(file);
                return true;
            }
            return false;
        });
        boolean miniConnected = isMiniConnected();
        int menuResource = miniConnected ? R.menu.scripts_popup_menu_ex : R.menu.scripts_popup_menu;
        popup.inflate(menuResource);
        popup.show();
    }

    private void renameFile(FileWrapper file) {
        String title = getResources().getString(R.string.title_dialog_rename);
        String input = FilenameUtils.removeExtension(file.getName());
        DialogUtils.showEditDialog(activity, title, input, output -> {
            File dir = new File(FilenameUtils.getFullPath(file.getAbsolutePath()));
            if (dir.exists()) {
                FileWrapper dest = new FileWrapper(new File(dir, output + FILE_EXTENSION), file.editor());
                if (file.exists()) {
                    if (!dest.exists() && file.renameTo(dest.file())) {
                        scriptsRecyclerAdapter.change(file, dest);
                    } else {
                        activity.runOnUiThread(() -> {
                            ApplicationStateHandler.updateNotification(ERROR, R.string.error_snackbar_name_exists);
                            ApplicationStateHandler.updateState(State.STATE_ERROR);
                        });
                    }
                }
            }
        });
    }

    private void removeFile(FileWrapper file) {
        String title = getResources().getString(R.string.title_dialog_delete);
        String message = String.format(getString(R.string.info_dialog_delete), FilenameUtils.removeExtension(file.getName()));
        DialogUtils.showWarningDialog(activity, title, message, () -> {
            if (file.delete()) {
                scriptsRecyclerAdapter.remove(file);
            }
        });
    }

    private void shareFile(FileWrapper file) {
        if (file.exists()) {
            Uri uri = FileProvider.getUriForFile(activity, "cc.calliope.file_provider", file.file());
            Intent intent = new Intent(Intent.ACTION_SEND);
            intent.setType("text/plain");
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
            intent.putExtra(Intent.EXTRA_STREAM, uri);
            intent.putExtra(Intent.EXTRA_SUBJECT, getString(R.string.subject_dialog_share));
            intent.putExtra(Intent.EXTRA_TEXT, getString(R.string.text_dialog_share));
            startActivity(Intent.createChooser(intent, getString(R.string.title_dialog_share)));
        }
    }

    public void copyFile(FileWrapper file) {
        sourceFilePath = file.getAbsolutePath();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            openDocumentTreeNewApi();
        } else {
            openDocumentTree();
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.Q)
    private void openDocumentTreeNewApi() {
        StorageManager storageManager = (StorageManager) activity.getSystemService(Context.STORAGE_SERVICE);
        Intent intent = storageManager.getPrimaryStorageVolume().createOpenDocumentTreeIntent();
        String targetDirectory = "MINI";
        Uri uri = intent.getParcelableExtra("android.provider.extra.INITIAL_URI");
        String scheme = uri.toString();
        scheme = scheme.replace("/root/", "/document/");
        scheme += "%3A" + targetDirectory;
        uri = Uri.parse(scheme);
        intent.putExtra("android.provider.extra.INITIAL_URI", uri);
        treeUriResultLauncher.launch(intent);
    }

    private void openDocumentTree() {
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT_TREE);
        treeUriResultLauncher.launch(intent);
    }

    ActivityResultLauncher<Intent> treeUriResultLauncher = registerForActivityResult(
            new ActivityResultContracts.StartActivityForResult(), result -> {
                if (result.getResultCode() == RESULT_OK && result.getData() != null) {
                    Uri treeUri = result.getData().getData();
                    try {
                        activity.getContentResolver().takePersistableUriPermission(
                                treeUri,
                                Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION
                        );
                        writeFile(treeUri);
                    } catch (SecurityException e) {
                        Log.e(TAG, "Persistable URI permission failed", e);
                        activity.runOnUiThread(() -> {
                            ApplicationStateHandler.updateNotification(ERROR, R.string.usb_copy_access_denied);
                            ApplicationStateHandler.updateState(State.STATE_ERROR);
                        });
                    }
                }
            }
    );

    public void writeFile(Uri uri) {
        new Thread(() -> {
            try {
                DocumentFile directory = DocumentFile.fromTreeUri(activity, uri);
                DocumentFile file = directory.createFile("application/octet-stream", "firmware.hex");

                FileInputStream inputStream = new FileInputStream(sourceFilePath);
                long sourceFileSize = new File(sourceFilePath).length();

                ParcelFileDescriptor pfd = activity.getContentResolver().openFileDescriptor(file.getUri(), "w");
                FileOutputStream outputStream = new FileOutputStream(pfd.getFileDescriptor());

                activity.runOnUiThread(() -> {
                    ApplicationStateHandler.updateNotification(INFO, R.string.usb_copy_started);
                    ApplicationStateHandler.updateState(State.STATE_BUSY);
                });

                byte[] buffer = new byte[8192];
                int bytesRead;
                long totalBytesWritten = 0;

                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                    totalBytesWritten += bytesRead;
                }

                outputStream.flush();
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    try {
                        outputStream.getFD().sync();
                    } catch (Exception ignored) {
                    }
                }

                inputStream.close();
                outputStream.close();

                activity.runOnUiThread(() -> {
                    ApplicationStateHandler.updateNotification(INFO, R.string.usb_copy_finished);
                    ApplicationStateHandler.updateState(State.STATE_IDLE);
                });
            } catch (IOException e) {
                Log.e(TAG, "File copy error", e);
                activity.runOnUiThread(() -> {
                    ApplicationStateHandler.updateNotification(ERROR, R.string.usb_copy_failed);
                    ApplicationStateHandler.updateState(State.STATE_ERROR);
                });
            } catch (Exception e) {
                Log.e(TAG, "Unexpected error", e);
                activity.runOnUiThread(() -> {
                    ApplicationStateHandler.updateNotification(ERROR, R.string.usb_copy_unexpected_error);
                    ApplicationStateHandler.updateState(State.STATE_ERROR);
                });
            }
        }).start();
    }

    private boolean isMiniConnected() {
        UsbManager manager = (UsbManager) activity.getSystemService(Context.USB_SERVICE);
        HashMap<String, UsbDevice> deviceList = manager.getDeviceList();
        if (deviceList.isEmpty()) return false;
        for (UsbDevice device : deviceList.values()) {
            String productName = device.getProductName();
            if (productName != null &&
                    (productName.contains("Calliope") ||
                            productName.contains("mini") ||
                            productName.contains("micro:bit") ||
                            productName.contains("Microbit"))) {
                return true;
            }
        }
        return false;
    }
}