package cash.p.terminal.wallet.useCases

import cash.p.terminal.wallet.Account
import cash.p.terminal.wallet.AccountType
import cash.p.terminal.wallet.IAccountManager
import cash.p.terminal.wallet.IAdapterManager
import cash.p.terminal.wallet.IReceiveAdapter
import cash.p.terminal.wallet.IWalletManager
import cash.p.terminal.wallet.OneTimeReceiveAdapter
import cash.p.terminal.wallet.Token
import cash.p.terminal.wallet.Wallet
import cash.p.terminal.wallet.WalletFactory
import cash.p.terminal.wallet.entities.TokenQuery
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first

class WalletUseCase(
    private val walletManager: IWalletManager,
    private val accountManager: IAccountManager,
    private val adapterManager: IAdapterManager,
    private val getHardwarePublicKeyForWalletUseCase: GetHardwarePublicKeyForWalletUseCase,
    private val scanToAddUseCase: ScanToAddUseCase,
    private val walletFactory: WalletFactory
) {
    fun getWallet(token: Token): Wallet? = walletManager.activeWallets.find {
        it.token == token
    }

    fun getWallet(coinUid: String, blockchainType: String): Wallet? =
        walletManager.activeWallets.find {
            it.token.coin.uid == coinUid && it.token.blockchainType.uid == blockchainType
        }

    suspend fun createWallets(tokensToAdd: Set<Token>): Boolean {
        val account = accountManager.activeAccount ?: return false
        return if (account.type is AccountType.HardwareCard) {
            createWalletsForHardwareWallet(
                account = account,
                tokensToAdd = tokensToAdd
            )
        } else {
            walletManager.saveSuspended(tokensToAdd.mapNotNull {
                walletFactory.create(
                    it,
                    account,
                    null
                )
            })
            true
        }
    }

    private suspend fun createWalletsForHardwareWallet(
        account: Account,
        tokensToAdd: Set<Token>
    ): Boolean {
        var hardwarePublicKeys =
            tokensToAdd.mapNotNull { token ->
                getHardwarePublicKeyForWalletUseCase(
                    account = account,
                    blockchainType = token.blockchainType,
                    tokenType = token.type
                )?.let { hardwarePublicKey ->
                    token to hardwarePublicKey
                }
            }
        if (tokensToAdd.size == hardwarePublicKeys.size) {
            // We already had all hardware public keys
            walletManager.save(hardwarePublicKeys.mapNotNull { (token, hardwarePublicKey) ->
                walletFactory.create(token, account, hardwarePublicKey)
            })
            return true
        }
        val queryList = tokensToAdd.map {
            TokenQuery(
                blockchainType = it.blockchainType,
                tokenType = it.type
            )
        }
        val cardId = (account.type as AccountType.HardwareCard).cardId
        val allKeysCreated = scanToAddUseCase.addTokensByScan(
            blockchainsToDerive = queryList,
            cardId = cardId,
            accountId = account.id
        )
        hardwarePublicKeys =
            tokensToAdd.mapNotNull { token ->
                getHardwarePublicKeyForWalletUseCase(
                    account = account,
                    blockchainType = token.blockchainType,
                    tokenType = token.type
                )?.let { hardwarePublicKey ->
                    token to hardwarePublicKey
                }
            }
        walletManager.save(hardwarePublicKeys.mapNotNull { (token, hardwarePublicKey) ->
            walletFactory.create(token, account, hardwarePublicKey)
        })
        return allKeysCreated
    }

    suspend fun createWalletIfNotExists(token: Token): Wallet? =
        getWallet(token) ?: if (createWallets(setOf(token))) getWallet(token) else null

    suspend fun awaitWallets(tokens: Set<Token>) {
        if (tokens.all { getWallet(it) != null }) return

        walletManager.activeWalletsFlow
            .filter { wallets ->
                tokens.all { token ->
                    wallets.any { it.token == token }
                }
            }
            .first()
    }

    fun getReceiveAddress(token: Token): String {
        val adapter = getReceiveAdapter(token)
        requireNotNull(adapter) { "WalletUseCase: adapter for $token is not found" }

        return adapter.receiveAddress
    }

    fun getReceiveAdapter(token: Token): IReceiveAdapter? {
        val wallet = getWallet(token)
        requireNotNull(wallet) { "WalletUseCase: wallet for $token is not found" }
        return adapterManager.getReceiveAdapterForWallet(wallet)
    }

    suspend fun getOneTimeReceiveAddress(token: Token): String {
        val adapter = getReceiveAdapter(token)
        requireNotNull(adapter) { "WalletUseCase: adapter for $token is not found" }

        return (adapter as? OneTimeReceiveAdapter)?.generateOneTimeAddress()
            ?: adapter.receiveAddress
    }
}
