package cash.p.terminal.modules.send.zcash

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.viewModelScope
import cash.p.terminal.R
import io.horizontalsystems.core.logger.AppLogger
import cash.p.terminal.core.HSCaution
import cash.p.terminal.core.ISendZcashAdapter
import cash.p.terminal.core.LocalizedException
import cash.p.terminal.core.managers.PendingTransactionRegistrar
import cash.p.terminal.core.providers.AppConfigProvider
import cash.p.terminal.entities.Address
import cash.p.terminal.entities.PendingTransactionDraft
import cash.p.terminal.modules.amount.SendAmountService
import cash.p.terminal.modules.contacts.ContactsRepository
import cash.p.terminal.modules.send.SendConfirmationData
import cash.p.terminal.modules.send.SendResult
import cash.p.terminal.modules.xrate.XRateService
import cash.p.terminal.strings.helpers.TranslatableString
import cash.p.terminal.wallet.Wallet
import cash.z.ecc.android.sdk.ext.collectWith
import io.grpc.StatusRuntimeException
import io.horizontalsystems.core.ViewModelUiState
import io.horizontalsystems.core.toHexReversed
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.math.BigDecimal
import java.net.UnknownHostException

class SendZCashViewModel(
    private val adapter: ISendZcashAdapter,
    val wallet: Wallet,
    xRateService: XRateService,
    private val amountService: SendAmountService,
    private val addressService: SendZCashAddressService,
    private val memoService: SendZCashMemoService,
    private val contactsRepo: ContactsRepository,
    private val address: Address?,
    private val showAddressInput: Boolean,
    private val pendingRegistrar: PendingTransactionRegistrar
) : ViewModelUiState<SendZCashUiState>() {
    val blockchainType = wallet.token.blockchainType
    val coinMaxAllowedDecimals = wallet.token.decimals
    val fiatMaxAllowedDecimals = AppConfigProvider.fiatDecimal
    val memoMaxLength by memoService::memoMaxLength

    private val fee = adapter.fee
    private var amountState = amountService.stateFlow.value
    private var addressState = addressService.stateFlow.value
    private var memoState = memoService.stateFlow.value
    private var pendingTxId: String? = null

    var coinRate by mutableStateOf(xRateService.getRate(wallet.coin.uid))
        private set
    var sendResult by mutableStateOf<SendResult?>(null)
        private set

    private val logger = AppLogger("Send-${wallet.coin.code}")

    init {
        xRateService.getRateFlow(wallet.coin.uid).collectWith(viewModelScope) {
            coinRate = it
        }
        amountService.stateFlow.collectWith(viewModelScope) {
            handleUpdatedAmountState(it)
        }
        addressService.stateFlow.collectWith(viewModelScope) {
            handleUpdatedAddressState(it)
        }
        memoService.stateFlow.collectWith(viewModelScope) {
            handleUpdatedMemoState(it)
        }
        viewModelScope.launch {
            addressService.setAddress(address)
        }
    }

    companion object {
        fun createCaution(error: Throwable) = when (error) {
            is UnknownHostException -> HSCaution(TranslatableString.ResString(R.string.Hud_Text_NoInternet))
            is LocalizedException -> HSCaution(TranslatableString.ResString(error.errorTextRes))
            else -> HSCaution(TranslatableString.PlainString(error.message ?: ""))
        }
    }

    override fun createState() = SendZCashUiState(
        fee = fee.value,
        availableBalance = amountState.availableBalance,
        addressError = addressState.addressError,
        amountCaution = amountState.amountCaution,
        memoIsAllowed = memoState.memoIsAllowed,
        canBeSend = amountState.canBeSend && addressState.canBeSend,
        showAddressInput = showAddressInput,
        address = addressState.address
    )

    fun onEnterAmount(amount: BigDecimal?) {
        amountService.setAmount(amount)
    }

    fun onEnterAddress(address: Address?) {
        viewModelScope.launch {
            addressService.setAddress(address)
        }
    }

    fun onEnterMemo(memo: String) {
        memoService.setMemo(memo)
    }

    private fun handleUpdatedAmountState(amountState: SendAmountService.State) {
        this.amountState = amountState

        emitState()
    }

    private fun handleUpdatedAddressState(addressState: SendZCashAddressService.State) {
        this.addressState = addressState

        addressState.address?.hex?.let {
            memoService.setAddress(it)
        }

        emitState()
    }

    private fun handleUpdatedMemoState(memoState: SendZCashMemoService.State) {
        this.memoState = memoState

        emitState()
    }

    fun getConfirmationData(): SendConfirmationData {
        val address = addressState.address!!
        val contact = contactsRepo.getContactsFiltered(
            blockchainType,
            addressQuery = address.hex
        ).firstOrNull()
        return SendConfirmationData(
            amount = amountState.amount!!,
            fee = fee.value,
            address = address,
            contact = contact,
            coin = wallet.coin,
            feeCoin = wallet.coin,
            memo = memoState.memo
        )
    }

    fun onClickSend() {
        viewModelScope.launch {
            send()
        }
    }

    private suspend fun send() = withContext(Dispatchers.IO) {
        val logger = logger.getScopedUnique()
        logger.info("click send button")

        try {
            sendResult = SendResult.Sending
            // 1. Create pending transaction draft BEFORE sending
            val draft = PendingTransactionDraft(
                wallet = wallet,
                token = wallet.token,
                amount = amountState.amount!!,
                fee = fee.value,
                fromAddress = "",  // ZCash doesn't require from address
                toAddress = addressState.address!!.hex,
                memo = memoState.memo,
                txHash = null  // ZCash doesn't return hash immediately
            )

            // 2. Register pending transaction
            pendingTxId = pendingRegistrar.register(draft)

            // 3. Broadcast transaction
            val txId = adapter.send(
                amountState.amount!!,
                addressState.address!!.hex,
                memoState.memo,
                logger
            )
            pendingTxId?.let {
                pendingRegistrar.updateTxId(it, txId.byteArray.toHexReversed())
            }

            sendResult = SendResult.Sent(txId.byteArray.toHexReversed())
            logger.info("success")
        } catch (e: StatusRuntimeException) {
            // Delete pending transaction on error
            pendingTxId?.let {
                pendingRegistrar.deleteFailed(it)
            }
            logger.warning("failed", e)
            sendResult =
                SendResult.Failed(HSCaution(TranslatableString.ResString(R.string.transaction_error_need_to_check)))
        } catch (e: Throwable) {
            // Delete pending transaction on error
            pendingTxId?.let {
                pendingRegistrar.deleteFailed(it)
            }
            logger.warning("failed", e)
            sendResult = SendResult.Failed(createCaution(e))
        }
    }
}

data class SendZCashUiState(
    val fee: BigDecimal,
    val availableBalance: BigDecimal,
    val addressError: Throwable?,
    val amountCaution: HSCaution?,
    val memoIsAllowed: Boolean,
    val canBeSend: Boolean,
    val showAddressInput: Boolean,
    val address: Address?,
)
