package cash.p.terminal.modules.manageaccount

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Icon
import androidx.compose.material.ripple
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.platform.LocalClipboardManager
import androidx.compose.ui.platform.LocalView
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import androidx.navigation.compose.rememberNavController
import cash.p.terminal.MainGraphDirections
import cash.p.terminal.R
import cash.p.terminal.core.authorizedAction
import cash.p.terminal.core.managers.FaqManager
import cash.p.terminal.core.premiumAction
import cash.p.terminal.modules.balance.HeaderNote
import cash.p.terminal.modules.balance.ui.NoteError
import cash.p.terminal.modules.balance.ui.NoteWarning
import cash.p.terminal.modules.manageaccount.ManageAccountModule.BackupItem
import cash.p.terminal.modules.manageaccount.ManageAccountModule.KeyAction
import cash.p.terminal.modules.manageaccount.recoveryphrase.RecoveryPhraseFragment
import cash.p.terminal.modules.manageaccounts.ManageAccountsModule
import cash.p.terminal.modules.resettofactorysettings.ResetToFactorySettingsFragment
import cash.p.terminal.modules.resettofactorysettings.ResetToFactorySettingsFragment.Input
import cash.p.terminal.modules.restoreaccount.RestoreAccountFragment
import cash.p.terminal.modules.settings.main.HsSettingCell
import cash.p.terminal.navigation.slideFromBottom
import cash.p.terminal.navigation.slideFromBottomForResult
import cash.p.terminal.navigation.slideFromRight
import cash.p.terminal.strings.helpers.TranslatableString
import cash.p.terminal.ui.compose.components.FormsInput
import cash.p.terminal.ui_compose.components.AppBar
import cash.p.terminal.ui_compose.components.ButtonSecondaryDefault
import cash.p.terminal.ui_compose.components.CellUniversalLawrenceSection
import cash.p.terminal.ui_compose.components.HSpacer
import cash.p.terminal.ui_compose.components.HeaderText
import cash.p.terminal.ui_compose.components.HsBackButton
import cash.p.terminal.ui_compose.components.HsImage
import cash.p.terminal.ui_compose.components.HudHelper
import cash.p.terminal.ui_compose.components.InfoText
import cash.p.terminal.ui_compose.components.MenuItem
import cash.p.terminal.ui_compose.components.RowUniversal
import cash.p.terminal.ui_compose.components.VSpacer
import cash.p.terminal.ui_compose.components.body_jacob
import cash.p.terminal.ui_compose.components.body_leah
import cash.p.terminal.ui_compose.components.body_lucian
import cash.p.terminal.ui_compose.theme.ComposeAppTheme
import cash.p.terminal.wallet.Account
import cash.p.terminal.wallet.AccountOrigin
import cash.p.terminal.wallet.AccountType

@Composable
internal fun ManageAccountScreen(
    navController: NavController,
    viewState: ManageAccountModule.ViewState,
    account: Account,
    onCloseClicked: () -> Unit,
    onSaveClicked: () -> Unit,
    onNameChanged: (String) -> Unit,
    onActionClick: (KeyAction) -> Unit,
) {
    if (viewState.closeScreen) {
        navController.popBackStack()
        onCloseClicked()
    }

    Column(modifier = Modifier.background(color = ComposeAppTheme.colors.tyler)) {
        AppBar(
            title = viewState.title,
            navigationIcon = {
                HsBackButton(onClick = { navController.popBackStack() })
            },
            menuItems = listOf(
                MenuItem(
                    title = TranslatableString.ResString(R.string.ManageAccount_Save),
                    onClick = onSaveClicked,
                    enabled = viewState.canSave
                )
            )
        )

        Column(
            modifier = Modifier.verticalScroll(rememberScrollState())
        ) {
            HeaderText(stringResource(id = R.string.ManageAccount_Name))

            FormsInput(
                modifier = Modifier.padding(horizontal = 16.dp),
                initial = viewState.title,
                hint = "",
                onValueChange = onNameChanged
            )

            viewState.signedHashes?.let {
                VSpacer(32.dp)
                CellUniversalLawrenceSection(
                    listOf {
                        HsSettingCell(
                            title = R.string.signed,
                            icon = R.drawable.ic_info_20,
                            value = stringResource(
                                R.string.details_row_subtitle_signed_hashes_format,
                                it
                            ),
                        )
                    }
                )
            }

            when (viewState.headerNote) {
                HeaderNote.NonStandardAccount -> {
                    NoteError(
                        modifier = Modifier.padding(
                            start = 16.dp,
                            end = 16.dp,
                            top = 32.dp
                        ),
                        text = stringResource(R.string.AccountRecovery_MigrationRequired),
                        onClick = {
                            FaqManager.showFaqPage(
                                navController,
                                FaqManager.faqPathMigrationRequired
                            )
                        }
                    )
                }

                HeaderNote.NonRecommendedAccount -> {
                    NoteWarning(
                        modifier = Modifier.padding(
                            start = 16.dp,
                            end = 16.dp,
                            top = 32.dp
                        ),
                        text = stringResource(R.string.AccountRecovery_MigrationRecommended),
                        onClick = {
                            FaqManager.showFaqPage(
                                navController,
                                FaqManager.faqPathMigrationRecommended
                            )
                        },
                        onClose = null
                    )
                }

                HeaderNote.None -> Unit
            }

            KeyActions(
                viewState = viewState,
                account = account,
                onActionClick = onActionClick,
                navController = navController
            )

            if (viewState.backupActions.isNotEmpty()) {
                BackupActions(
                    viewState.backupActions,
                    account,
                    navController
                )
            }

            if (viewState.canBeDuplicated) {
                VSpacer(32.dp)
                CellUniversalLawrenceSection(
                    listOf {
                        AccountActionWithInfoItem(
                            title = stringResource(id = R.string.duplicate_wallet),
                            icon = painterResource(id = R.drawable.ic_copy_24px),
                            onInfoClick = {
                                navController.slideFromBottom(
                                    R.id.duplicateWalletInfoDialog,
                                    account
                                )
                            },
                            onClick = {
                                navController.authorizedAction {
                                    navController.slideFromRight(
                                        R.id.restoreAccountFragment,
                                        ManageAccountsModule.Input(
                                            popOffOnSuccess = R.id.manageAccountsFragment,
                                            popOffInclusive = true,
                                            defaultRoute = RestoreAccountFragment.ROUTE_DUPLICATE,
                                            accountId = account.id
                                        )
                                    )
                                }
                            }
                        )
                    })
            }

            VSpacer(32.dp)
            CellUniversalLawrenceSection(
                listOf {
                    RedActionItem(
                        title = stringResource(id = R.string.ManageAccount_Unlink),
                        icon = painterResource(id = R.drawable.ic_delete_20)
                    ) {
                        navController.slideFromBottom(
                            R.id.unlinkConfirmationDialog,
                            account
                        )
                    }
                })
            VSpacer(32.dp)
        }
    }
}

@Composable
private fun KeyActions(
    viewState: ManageAccountModule.ViewState,
    account: Account,
    onActionClick: (KeyAction) -> Unit,
    navController: NavController
) {
    val actionItems = mutableListOf<@Composable () -> Unit>()

    viewState.keyActions.forEach { keyAction ->
        when (keyAction) {
            KeyAction.RecoveryPhrase -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.RecoveryPhrase_Title),
                        icon = painterResource(id = R.drawable.icon_paper_contract_20)
                    ) {
                        navController.authorizedAction {
                            navController.slideFromRight(
                                MainGraphDirections.actionGlobalToRecoveryPhraseFragment(
                                    RecoveryPhraseFragment.Input(
                                        account = account,
                                        recoveryPhraseType = RecoveryPhraseFragment.RecoveryPhraseType.Mnemonic
                                    )
                                )
                            )
                        }
                    }
                }
            }

            KeyAction.RecoveryPhraseMonero -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.RecoveryPhrase_monero_Title),
                        icon = painterResource(id = R.drawable.icon_paper_contract_20),
                        iconTint = ComposeAppTheme.colors.jacob
                    ) {
                        navController.authorizedAction {
                            navController.premiumAction {
                                navController.slideFromRight(
                                    MainGraphDirections.actionGlobalToRecoveryPhraseFragment(
                                        RecoveryPhraseFragment.Input(
                                            account = account,
                                            recoveryPhraseType = RecoveryPhraseFragment.RecoveryPhraseType.Monero
                                        )
                                    )
                                )
                            }
                        }
                    }
                }
            }

            KeyAction.PrivateKeys -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.PrivateKeys_Title),
                        icon = painterResource(id = R.drawable.ic_key_20)
                    ) {
                        navController.slideFromRight(
                            R.id.privateKeysFragment,
                            account
                        )
                    }
                }
            }

            KeyAction.PublicKeys -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.PublicKeys_Title),
                        icon = painterResource(id = R.drawable.icon_binocule_20)
                    ) {
                        navController.slideFromRight(
                            R.id.publicKeysFragment,
                            account
                        )
                    }
                }
            }

            KeyAction.ResetToFactorySettings -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.reset_to_factory_settings),
                        icon = painterResource(id = R.drawable.ic_delete_20)
                    ) {
                        navController.slideFromBottomForResult<ResetToFactorySettingsFragment.Result>(
                            resId = R.id.resetToFactorySettingsFragment,
                            input = Input(account)
                        ) {
                            if (it.success) {
                                onActionClick(keyAction)
                            }
                        }
                    }
                }
            }

            KeyAction.AccessCodeRecovery -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.card_settings_access_code_recovery_title),
                        icon = painterResource(id = R.drawable.icon_unlocked_48)
                    ) {
                        onActionClick(keyAction)
                    }
                }
            }

            KeyAction.ChangeAccessCode -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.change_access_code),
                        icon = painterResource(id = R.drawable.ic_key_20)
                    ) {
                        onActionClick(keyAction)
                    }
                }
            }

            KeyAction.ForgotAccessCode -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.forgot_access_code),
                        icon = painterResource(id = R.drawable.icon_warning_2_20)
                    ) {
                        onActionClick(keyAction)
                    }
                }
            }

            KeyAction.ViewKey -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.view_key),
                        icon = painterResource(id = R.drawable.icon_binocule_20)
                    ) {
                        onActionClick(keyAction)
                    }
                }
            }

            KeyAction.SpendKey -> {
                actionItems.add {
                    AccountActionItem(
                        title = stringResource(id = R.string.spend_key),
                        icon = painterResource(id = R.drawable.icon_paper_contract_20)
                    ) {
                        onActionClick(keyAction)
                    }
                }
            }
        }
    }

    if (actionItems.isNotEmpty()) {
        VSpacer(32.dp)
        CellUniversalLawrenceSection(actionItems)
    }
}

@Composable
private fun BackupActions(
    backupActions: List<BackupItem>,
    account: Account,
    navController: NavController
) {
    val actionItems = mutableListOf<@Composable () -> Unit>()
    val infoItems = mutableListOf<@Composable () -> Unit>()

    backupActions.forEach { action ->
        when (action) {
            is BackupItem.ManualBackup -> {
                actionItems.add {
                    YellowActionItem(
                        title = stringResource(id = R.string.ManageAccount_RecoveryPhraseBackup),
                        icon = painterResource(id = R.drawable.ic_edit_24),
                        attention = action.showAttention,
                        completed = action.completed
                    ) {
                        navController.authorizedAction {
                            navController.slideFromBottom(
                                R.id.backupKeyFragment,
                                account
                            )
                        }
                    }
                }
            }

            is BackupItem.LocalBackup -> {
                actionItems.add {
                    YellowActionItem(
                        title = stringResource(id = R.string.ManageAccount_LocalBackup),
                        icon = painterResource(id = R.drawable.ic_file_24),
                        attention = action.showAttention
                    ) {
                        navController.authorizedAction {
                            navController.slideFromBottom(R.id.backupLocalFragment, account)
                        }
                    }
                }
            }

            is BackupItem.InfoText -> {
                infoItems.add {
                    InfoText(text = stringResource(action.textRes))
                }
            }
        }
    }
    if (actionItems.isNotEmpty()) {
        VSpacer(32.dp)
        CellUniversalLawrenceSection(actionItems)
    }
    infoItems.forEach {
        it.invoke()
    }

}

@Composable
private fun RedActionItem(
    title: String,
    icon: Painter,
    onClick: () -> Unit
) {

    RowUniversal(
        onClick = onClick
    ) {
        Icon(
            modifier = Modifier
                .padding(horizontal = 16.dp)
                .size(24.dp),
            painter = icon,
            contentDescription = null,
            tint = ComposeAppTheme.colors.lucian
        )

        body_lucian(
            text = title,
        )
    }
}

@Composable
private fun AccountActionItem(
    title: String,
    icon: Painter? = null,
    iconTint: Color? = null,
    coinIconUrl: String? = null,
    coinIconPlaceholder: Int? = null,
    attention: Boolean = false,
    badge: String? = null,
    onClick: (() -> Unit)? = null
) {
    RowUniversal(
        onClick = onClick
    ) {
        icon?.let {
            Icon(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .size(24.dp),
                painter = icon,
                contentDescription = null,
                tint = iconTint ?: ComposeAppTheme.colors.grey
            )
        }

        if (coinIconUrl != null) {
            HsImage(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .size(20.dp),
                url = coinIconUrl,
                placeholder = coinIconPlaceholder
            )
        }

        body_leah(
            modifier = Modifier.weight(1f),
            text = title,
        )

        if (attention) {
            Icon(
                modifier = Modifier.padding(horizontal = 16.dp),
                painter = painterResource(id = R.drawable.ic_attention_20),
                contentDescription = null,
                tint = ComposeAppTheme.colors.lucian
            )
            Spacer(modifier = Modifier.width(6.dp))
        }

        badge?.let {
            val view = LocalView.current
            val clipboardManager = LocalClipboardManager.current

            ButtonSecondaryDefault(
                modifier = Modifier.padding(horizontal = 16.dp),
                title = it,
                onClick = {
                    clipboardManager.setText(AnnotatedString(it))
                    HudHelper.showSuccessMessage(view, R.string.Hud_Text_Copied)
                }
            )
        }

        onClick?.let {
            Icon(
                painter = painterResource(id = R.drawable.ic_arrow_right),
                contentDescription = null,
                tint = ComposeAppTheme.colors.grey
            )
            HSpacer(16.dp)
        }
    }
}

@Composable
private fun AccountActionWithInfoItem(
    title: String,
    onInfoClick: () -> Unit,
    icon: Painter? = null,
    iconTint: Color? = null,
    onClick: (() -> Unit)? = null
) {
    RowUniversal(
        onClick = onClick
    ) {
        icon?.let {
            Icon(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .size(24.dp),
                painter = icon,
                contentDescription = null,
                tint = iconTint ?: ComposeAppTheme.colors.grey
            )
        }

        body_leah(
            modifier = Modifier.weight(1f),
            text = title,
        )

        Icon(
            modifier = Modifier
                .padding(horizontal = 16.dp)
                .clickable(
                    onClick = onInfoClick,
                    interactionSource = remember { MutableInteractionSource() },
                    indication = ripple(bounded = false)
                ),
            painter = painterResource(id = R.drawable.ic_info_20),
            contentDescription = null,
            tint = ComposeAppTheme.colors.grey,
        )

        onClick?.let {
            Icon(
                painter = painterResource(id = R.drawable.ic_arrow_right),
                contentDescription = null,
                tint = ComposeAppTheme.colors.grey
            )
            HSpacer(16.dp)
        }
    }
}

@Composable
private fun YellowActionItem(
    title: String,
    icon: Painter? = null,
    attention: Boolean = false,
    completed: Boolean = false,
    onClick: (() -> Unit)? = null
) {

    RowUniversal(
        onClick = onClick
    ) {
        icon?.let {
            Icon(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .size(24.dp),
                painter = icon,
                contentDescription = null,
                tint = ComposeAppTheme.colors.jacob
            )
        }

        body_jacob(
            modifier = Modifier.weight(1f),
            text = title,
        )

        if (attention) {
            Icon(
                modifier = Modifier.padding(horizontal = 16.dp),
                painter = painterResource(id = R.drawable.ic_attention_20),
                contentDescription = null,
                tint = ComposeAppTheme.colors.lucian
            )
            HSpacer(6.dp)
        } else if (completed) {
            Icon(
                modifier = Modifier.padding(horizontal = 16.dp),
                painter = painterResource(id = R.drawable.ic_checkmark_20),
                contentDescription = null,
                tint = ComposeAppTheme.colors.remus
            )
            HSpacer(6.dp)
        }
    }
}

@Preview(showBackground = true)
@Composable
private fun ManageAccountScreenPreview() {
    ComposeAppTheme {
        ManageAccountScreen(
            navController = rememberNavController(),
            viewState = ManageAccountModule.ViewState(
                title = "Account name",
                newName = "Account name",
                canSave = true,
                closeScreen = false,
                headerNote = HeaderNote.None,
                keyActions = listOf(
                    KeyAction.AccessCodeRecovery,
                    KeyAction.ChangeAccessCode,
                    KeyAction.ResetToFactorySettings,
                    KeyAction.ForgotAccessCode,
                ),
                backupActions = emptyList(),
                signedHashes = 2,
                canBeDuplicated = true
            ),
            account = Account(
                id = "id",
                name = "name",
                type = AccountType.HardwareCard(
                    cardId = "",
                    backupCardsCount = 0,
                    walletPublicKey = "",
                    signedHashes = 2
                ),
                origin = AccountOrigin.Created,
                level = 0
            ),
            onCloseClicked = {},
            onSaveClicked = {},
            onNameChanged = {},
            onActionClick = {}
        )
    }
}