package cash.p.terminal.core.managers

import cash.p.terminal.entities.PendingTransactionEntity
import cash.p.terminal.entities.TransactionValue
import cash.p.terminal.entities.transactionrecords.PendingTransactionRecord
import cash.p.terminal.entities.transactionrecords.TransactionRecord
import cash.p.terminal.entities.transactionrecords.ton.TonTransactionRecord
import java.math.BigDecimal
import kotlin.math.abs

data class MatchScore(
    val isMatch: Boolean,
    val confidence: Double
)

class PendingTransactionMatcher(
    private val repository: PendingTransactionRepository
) {

    suspend fun matchAndResolve(allRealTxs: List<TransactionRecord>) {
        // Get all pending transactions from all wallets
        val firstRealTransaction = allRealTxs.firstOrNull { it !is PendingTransactionRecord } ?: return
        val pending = repository.getPendingForWallet(firstRealTransaction.source.account.id)

        allRealTxs.forEach { real ->
            if (real is PendingTransactionRecord) {
                return@forEach
            }

            val matched = findBestMatch(real, pending)
            if (matched != null) {
                repository.deleteById(matched.id)
            }
        }
    }

    private fun findBestMatch(
        real: TransactionRecord,
        pending: List<PendingTransactionEntity>
    ): PendingTransactionEntity? {
        return pending
            .map { it to calculateMatchScore(it, real) }
            .filter { it.second.isMatch }
            .maxByOrNull { it.second.confidence }
            ?.first
    }

    private fun calculateMatchScoreInternal(
        txHash: String,
        timestampPending: Long,
        blockchainTypeUid: String,
        amountAtomic: String,
        toAddress: String,
        real: TransactionRecord
    ): MatchScore {
        // Priority 1: Hash Match (confidence = 1.0)
        if (txHash.isNotEmpty() && txHash == real.transactionHash) {
            return MatchScore(isMatch = true, confidence = 1.0)
        }

        // Priority 2: Fuzzy Match (confidence = 0.9)
        val blockchainMatches = blockchainTypeUid == real.blockchainType.uid
        val amountMatches = compareAmounts(amountAtomic, real)
        val realTo = real.to?.firstOrNull()

        if (blockchainMatches && amountMatches) {
            val addressMatches = realTo != null && toAddress.isNotEmpty() &&
                    toAddress.equals(realTo, ignoreCase = true)
            if (addressMatches) {
                return MatchScore(isMatch = true, confidence = 0.9)
            } else if (compareTimestamps(timestampPending, real.timestamp)) {
                return MatchScore(isMatch = true, confidence = 0.8)
            }
        }

        return MatchScore(isMatch = false, confidence = 0.0)
    }

    fun calculateMatchScore(
        pending: PendingTransactionRecord,
        real: TransactionRecord
    ): MatchScore = calculateMatchScoreInternal(
        txHash = pending.transactionHash,
        timestampPending = pending.timestamp,
        blockchainTypeUid = pending.blockchainType.uid,
        amountAtomic = pending.amount.movePointRight(pending.token.decimals).toBigInteger()
            .toString(),
        toAddress = pending.to?.firstOrNull() ?: "",
        real = real
    )

    private fun calculateMatchScore(
        pending: PendingTransactionEntity,
        real: TransactionRecord
    ): MatchScore = calculateMatchScoreInternal(
        txHash = pending.txHash ?: "",
        timestampPending = pending.createdAt/1000,
        blockchainTypeUid = pending.blockchainTypeUid,
        amountAtomic = pending.amountAtomic,
        toAddress = pending.toAddress,
        real = real
    )

    private fun compareAmounts(
        pendingAmountAtomic: String,
        real: TransactionRecord
    ): Boolean {
        return try {
            val pendingAmount = BigDecimal(pendingAmountAtomic)
                .movePointLeft(getRealDecimal(real))
                .abs()

            val realAmount = getRealAmount(real)?.abs() ?: return false

            // Allow 0.1% difference for floating point errors
            val difference = (pendingAmount - realAmount).abs()
            val threshold = pendingAmount.multiply(BigDecimal("0.001"))

            difference <= threshold
        } catch (e: Exception) {
            false
        }
    }

    private fun getRealDecimal(real: TransactionRecord): Int {
        val value = if(real is TonTransactionRecord) {
            real.actions.firstOrNull { it.type is TonTransactionRecord.Action.Type.Swap }
                ?.let { action ->
                    (action.type as? TonTransactionRecord.Action.Type.Swap)?.valueIn?.decimals
                }
                ?: real.actions.firstOrNull { it.type is TonTransactionRecord.Action.Type.Send }
                    ?.let { action ->
                        (action.type as? TonTransactionRecord.Action.Type.Send)?.value?.decimals
                    }
        } else {
            null
        }
        return value ?: real.token.decimals
    }

    private fun compareTimestamps(
        timestampPending: Long,
        timestampReal: Long
    ): Boolean {
        val differenceSeconds = abs(timestampPending - timestampReal)
        return differenceSeconds <= 10 // 10 seconds tolerance
    }

    private fun getRealAmount(
        real: TransactionRecord
    ): BigDecimal? {
        val value = if(real is TonTransactionRecord) {
            real.actions.firstOrNull { it.type is TonTransactionRecord.Action.Type.Swap }
                ?.let { action ->
                    (action.type as? TonTransactionRecord.Action.Type.Swap)?.valueIn?.decimalValue
                }
                ?: real.actions.firstOrNull { it.type is TonTransactionRecord.Action.Type.Send }
                    ?.let { action ->
                        (action.type as? TonTransactionRecord.Action.Type.Send)?.value?.decimalValue
                    }
        } else {
            null
        }
        if (value == null && real.mainValue != null) {
            real.mainValue?.let {
                when (it) {
                    is TransactionValue.CoinValue -> it.value
                    else -> null
                }
            }
        }
        return value
    }
}
