package cash.p.terminal.wallet

import cash.p.terminal.wallet.entities.Coin
import cash.p.terminal.wallet.entities.EnabledWallet
import cash.p.terminal.wallet.entities.TokenQuery
import cash.p.terminal.wallet.useCases.GetHardwarePublicKeyForWalletUseCase
import cash.p.terminal.wallet.WalletFactory
import io.horizontalsystems.core.entities.BlockchainType
import kotlinx.coroutines.runBlocking

class WalletStorage(
    private val marketKit: MarketKitWrapper,
    private val storage: IEnabledWalletStorage,
    private val getHardwarePublicKeyForWalletUseCase: GetHardwarePublicKeyForWalletUseCase,
    private val walletFactory: WalletFactory
) : IWalletStorage {

    private val map: HashMap<Wallet, Long> = HashMap()

    override fun wallets(account: Account): List<Wallet> {
        val enabledWallets = storage.enabledWallets(account.id)
        map.clear()

        val queries = enabledWallets.mapNotNull {
            TokenQuery.fromId(it.tokenQueryId)?.let {
                if (it.blockchainType is BlockchainType.Unsupported) {
                    return@mapNotNull null
                } else {
                    it
                }
            }
        }
        val tokens = marketKit.tokens(queries)

        val blockchainUids = queries.map { it.blockchainType.uid }
        val blockchains = marketKit.blockchains(blockchainUids)

        return enabledWallets.mapNotNull { enabledWallet ->
            val tokenQuery = TokenQuery.fromId(enabledWallet.tokenQueryId) ?: return@mapNotNull null

            tokens.find { it.tokenQuery == tokenQuery }?.let { token ->
                val hardwarePublicKey = runBlocking {
                    getHardwarePublicKeyForWalletUseCase(
                        account = account,
                        blockchainType = token.blockchainType,
                        tokenType = token.type
                    )
                }
                return@mapNotNull walletFactory.create(
                    token = token,
                    account = account,
                    hardwarePublicKey = hardwarePublicKey
                )?.apply { map[this] = enabledWallet.id }
            }

            if (enabledWallet.coinName != null && enabledWallet.coinCode != null && enabledWallet.coinDecimals != null) {
                val coinUid = tokenQuery.customCoinUid
                val blockchain = blockchains.firstOrNull { it.uid == tokenQuery.blockchainType.uid }
                    ?: return@mapNotNull null

                val token = Token(
                    coin = Coin(
                        uid = coinUid,
                        name = enabledWallet.coinName,
                        code = enabledWallet.coinCode,
                        image = enabledWallet.coinImage
                    ),
                    blockchain = blockchain,
                    type = tokenQuery.tokenType,
                    decimals = enabledWallet.coinDecimals
                )

                val hardwarePublicKey = runBlocking {
                    getHardwarePublicKeyForWalletUseCase(
                        account = account,
                        blockchainType = token.blockchainType,
                        tokenType = token.type
                    )
                }

                walletFactory.create(
                    token = token,
                    account = account,
                    hardwarePublicKey = hardwarePublicKey
                )?.apply {
                    map[this] = enabledWallet.id
                }
            } else {
                null
            }
        }
    }

    override fun save(wallets: List<Wallet>) {
        if (wallets.isEmpty()) return

        val accountId = wallets.first().account.id
        val existingTokenIds = storage.enabledWallets(accountId)
            .mapTo(mutableSetOf()) { it.tokenQueryId }

        val walletsToAdd = wallets
            .filter { wallet ->
                wallet.token.tokenQuery.id !in existingTokenIds
            }
            .distinctBy { it.token.tokenQuery.id }

        if (walletsToAdd.isEmpty()) return

        val enabledWallets = walletsToAdd.mapIndexed { index, wallet ->
            enabledWallet(wallet, index)
        }

        storage.save(enabledWallets).forEachIndexed { index, id ->
            map[walletsToAdd[index]] = id
        }
    }

    override fun delete(wallets: List<Wallet>) {
        storage.delete(wallets.mapNotNull { map[it] })
    }

    override fun handle(newEnabledWallets: List<EnabledWallet>) {
        storage.save(newEnabledWallets)
    }

    override fun clear() {
        storage.deleteAll()
    }

    private fun enabledWallet(wallet: Wallet, index: Int? = null): EnabledWallet {
        return EnabledWallet(
            tokenQueryId = wallet.token.tokenQuery.id,
            accountId = wallet.account.id,
            walletOrder = index,
            coinName = wallet.coin.name,
            coinCode = wallet.coin.code,
            coinDecimals = wallet.decimal,
            coinImage = wallet.coin.image
        )
    }
}
