package cash.p.terminal.modules.walletconnect.request

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import cash.p.terminal.modules.sendevmtransaction.SectionViewItem
import cash.p.terminal.modules.walletconnect.WCDelegate
import cash.p.terminal.strings.helpers.TranslatableString
import com.reown.walletkit.client.Wallet
import io.horizontalsystems.core.ViewModelUiState
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

class WCRequestViewModel(
    private val sessionRequest: Wallet.Model.SessionRequest,
    private val wcAction: AbstractWCAction
) : ViewModelUiState<WCRequestUiState>() {

    private var approveInProgress = false
    private var finish: Boolean = false
    private var error: Throwable? = null

    private var actionState = wcAction.stateFlow.value

    init {
        viewModelScope.launch {
            wcAction.stateFlow.collect {
                actionState = it

                emitState()
            }
        }

        wcAction.start(viewModelScope)
    }

    override fun createState() = WCRequestUiState(
        title = wcAction.getTitle(),
        finish = finish,
        runnable = !approveInProgress && actionState.runnable,
        approveButtonTitle = wcAction.getApproveButtonTitle(),
        contentItems = actionState.items
    )

    fun approve() = viewModelScope.launch(Dispatchers.Default) {
        error = null
        approveInProgress = true
        emitState()

        val actionResult = wcAction.performAction()

        WCDelegate.respondPendingRequest(
            sessionRequest.request.id,
            sessionRequest.topic,
            actionResult,
            onSuccessResult = {
                approveInProgress = false
                finish = true
                emitState()
            },
            onErrorResult = {
                approveInProgress = true
                error = it
                emitState()
            }
        )
    }

    fun reject() = viewModelScope.launch(Dispatchers.Default) {
        error = null

        WCDelegate.rejectRequest(
            sessionRequest.topic,
            sessionRequest.request.id,
            onSuccessResult = {
                finish = true
                emitState()
            },
            onErrorResult = {
                error = it
                emitState()
            }
        )

        emitState()
    }

    class Factory(
        private val sessionRequest: Wallet.Model.SessionRequest,
        private val wcAction: AbstractWCAction,
    ) : ViewModelProvider.Factory {
        @Suppress("UNCHECKED_CAST")
        override fun <T : ViewModel> create(modelClass: Class<T>): T {
            return WCRequestViewModel(sessionRequest, wcAction) as T
        }
    }
}

data class WCRequestUiState(
    val title: TranslatableString,
    val finish: Boolean,
    val runnable: Boolean,
    val approveButtonTitle: TranslatableString,
    val contentItems: List<SectionViewItem>
)
