package cash.p.terminal.entities.transactionrecords.ton

import cash.p.terminal.core.adapters.TonAdapter
import cash.p.terminal.entities.TransactionValue
import cash.p.terminal.entities.transactionrecords.TransactionRecord
import cash.p.terminal.entities.transactionrecords.TransactionRecordType
import cash.p.terminal.modules.transactions.TransactionStatus
import cash.p.terminal.wallet.Token
import cash.p.terminal.wallet.transaction.TransactionSource
import io.horizontalsystems.tonkit.models.Event
import kotlin.math.absoluteValue

class TonTransactionRecord(
    source: TransactionSource,
    event: Event,
    token: Token,
    val actions: List<Action>
) : TransactionRecord(
    uid = event.id,
    transactionHash = event.id,
    transactionIndex = 0,
    blockHeight = null,
    confirmationsThreshold = 1,
    timestamp = event.timestamp,
    failed = false,
    spam = event.scam,
    source = source,
    transactionRecordType = TransactionRecordType.TON,
    token = token,
) {

    override val to: List<String>?
        get() = actions.firstOrNull { it.type is Action.Type.Send }?.let { listOf((it.type as Action.Type.Send).to) } ?:
            actions.firstOrNull { it.type is Action.Type.Swap }?.let { listOf((it.type as Action.Type.Swap).routerAddress) }

    override val from: String?
        get() = actions.firstOrNull { it.type is Action.Type.Receive }?.let { (it.type as Action.Type.Receive).from }

    override val sentToSelf: Boolean
        get() = actions.firstOrNull { it.type is Action.Type.Send }?.let { (it.type as Action.Type.Send).sentToSelf } ?: false

    override val memo: String?
        get() = actions.firstNotNullOfOrNull {
            when (val type = it.type) {
                is Action.Type.Send -> type.comment
                is Action.Type.Receive -> type.comment
                else -> null
            }
        }
    val lt = event.lt
    val inProgress = event.inProgress
    val fee = TransactionValue.CoinValue(token, TonAdapter.Companion.getAmount(event.extra.absoluteValue))

    override fun status(lastBlockHeight: Int?) = when {
        inProgress -> TransactionStatus.Pending
        else -> TransactionStatus.Completed
    }

    override val mainValue: TransactionValue?
        get() = actions.singleOrNull()?.let { action ->
            when (val type = action.type) {
                is Action.Type.Receive -> type.value
                is Action.Type.Send -> type.value
                is Action.Type.Burn -> type.value
                is Action.Type.Mint -> type.value
                is Action.Type.ContractCall -> type.value
                is Action.Type.ContractDeploy,
                is Action.Type.Swap,
                is Action.Type.Unsupported -> null
            }
        }

    data class Action(
        val type: Type,
        val status: TransactionStatus
    ) {
        sealed class Type {
            data class Send(
                val value: TransactionValue,
                val to: String,
                val sentToSelf: Boolean,
                val comment: String?,
            ) : Type()

            data class Receive(
                val value: TransactionValue,
                val from: String,
                val to: String?,
                val comment: String?,
            ) : Type()

            data class Burn(val value: TransactionValue) : Type()

            data class Mint(val value: TransactionValue, val to: String?) : Type()

            data class Swap(
                val routerName: String?,
                val routerAddress: String,
                val valueIn: TransactionValue,
                val valueOut: TransactionValue
            ) : Type()

            data class ContractDeploy(val interfaces: List<String>) : Type()

            data class ContractCall(
                val address: String,
                val value: TransactionValue,
                val operation: String
            ) : Type()

            data class Unsupported(val type: String) : Type()
        }
    }
}