package ca.chancehorizon.paseo.background

import android.app.*
import android.appwidget.AppWidgetManager
import android.content.*
import android.hardware.Sensor
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import android.net.Uri
import android.os.Binder
import android.os.Build
import android.os.Environment
import android.os.IBinder
import android.speech.tts.TextToSpeech
import android.speech.tts.Voice
import android.util.Log
import android.widget.RemoteViews
import android.widget.Toast
import androidx.activity.result.ActivityResult
import androidx.core.app.NotificationCompat
import androidx.core.net.toUri
import ca.chancehorizon.paseo.*
import java.io.File
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream
import java.text.NumberFormat
import java.text.SimpleDateFormat
import java.util.*
import kotlin.math.max



class StepCounterService : Service(), SensorEventListener, TextToSpeech.OnInitListener {

    private val SERVICEID = 1001

    var running = false
    var hasStepCounter = true
    var sensorManager: SensorManager? = null
    var startSteps = 0
    var currentSteps = 0
    var endSteps = 0
    private var targetSteps = 10000 // default values for target steps (overridden later from shared preferences)
    var latestDay = 0
    var latestHour = 0

    lateinit var paseoDBHelper : PaseoDBHelper

    private var tts: TextToSpeech? = null
    private var ttsAvailable = false


    internal var mBinder: IBinder = LocalBinder()

    // set up things for resetting steps (to zero (most of the time) at midnight
    var myPendingIntent: PendingIntent? = null
    var midnightAlarmManager: AlarmManager? = null
    var myBroadcastReceiver: BroadcastReceiver? = null



    inner class LocalBinder : Binder() {
        val serverInstance: StepCounterService
            get() = this@StepCounterService
    }



    override fun onBind(intent: Intent): IBinder {
        return mBinder
    }



    override fun onCreate() {
        super.onCreate()

        sensorManager = getSystemService(Context.SENSOR_SERVICE) as SensorManager

        running = true

        val stepsSensor = sensorManager?.getDefaultSensor(Sensor.TYPE_STEP_COUNTER)

        if (stepsSensor != null) {
            sensorManager?.registerListener(this, stepsSensor, SensorManager.SENSOR_DELAY_UI, SensorManager.SENSOR_DELAY_UI)
        }

        // point to the Paseo database that stores all the daily steps data
        paseoDBHelper = PaseoDBHelper(this)

        try {
            tts = TextToSpeech(this, this)
        }
        catch (e: Exception) {
            tts = null
        }

        // set the user's target:
        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)
        targetSteps = paseoPrefs!!.getFloat("prefDailyStepsTarget", 10000F).toInt()

        // set up time to reset steps - immediately (10 seconds) after midnight
        val midnightAlarmCalendar: Calendar = Calendar.getInstance()
        midnightAlarmCalendar.set(Calendar.HOUR_OF_DAY, 0)
        midnightAlarmCalendar.set(Calendar.MINUTE, 0)
        midnightAlarmCalendar.set(Calendar.SECOND, 10)

        val midnightAlarmTime = midnightAlarmCalendar.getTimeInMillis()

        // set up the alarm to reset steps after midnight
        registerMyAlarmBroadcast()

        // set alarm to repeat every day (as in every 24 hours, which should be every day immediately after midnight)
        midnightAlarmManager?.setRepeating(AlarmManager.RTC, midnightAlarmTime, AlarmManager.INTERVAL_DAY,
            myPendingIntent!!
        )
    }



    override fun onInit(status: Int) {

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        // set up the text to speech voice
        if (status == TextToSpeech.SUCCESS) {
            val result = tts?.setLanguage(Locale.US)
            if (result == TextToSpeech.LANG_MISSING_DATA ||
                    result == TextToSpeech.LANG_NOT_SUPPORTED) {
                Log.e("TTS", "Language not supported")
            }

            ttsAvailable = true
        } else {
            Log.e("TTS", "Initialization failed")
            ttsAvailable = false
        }

        // update shared preferences to not show first run dialog again
        val edit: SharedPreferences.Editor = paseoPrefs!!.edit()
        edit.putBoolean("prefTTSAvailable", ttsAvailable)
        edit.apply()

        // make sure that Paseo's widget and notification have up to date steps shown
        updatePaseoWidget()
        updatePaseoNotification()
    }



    // set up the alarm to reset steps after midnight (shown in widget and notification)
    //
    // this is done so that the number of steps shown on a new day when no steps have yet been taken (sensed)
    //  is reset to zero, rather than showing yesterday's step total (the number of steps shown is only updated when steps are sensed)
    //
    // also create a backup of the database (if selected by user in settings)
    private fun registerMyAlarmBroadcast() {

        // This is the call back function(BroadcastReceiver) which will be called when
        //  the paseo alarm time is reached (to reset the widget and notification every day at midnight and auto backup the database).
        myBroadcastReceiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context?, intent: Intent?) {
                updatePaseoWidget()
                updatePaseoNotification()

                autoBackupDB()
            }
        }

        registerReceiver(myBroadcastReceiver, IntentFilter("ca.chancehorizon.paseo"))
        myPendingIntent = PendingIntent.getBroadcast(this.getApplicationContext(), 0, Intent("ca.chancehorizon.paseo"), PendingIntent.FLAG_IMMUTABLE)
        midnightAlarmManager = this.getSystemService(ALARM_SERVICE) as AlarmManager
    }



    // make a backup
    private fun autoBackupDB() {

        // set the user's target:
        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)
        val autoBackupsOn = paseoPrefs!!.getBoolean("prefAutoBackup", false)

        if (autoBackupsOn) {
            // flag to check that there is not already a backup saved today
            //  (sometimes Android is a bit cheeky and executes the daily alarm when Paseo is restarted)
            var alreadyBackedToday = false

            val dbFile: File = this.getDatabasePath("paseoDB.db")

            if (dbFile.exists()) {
                // set default name of exported file
                val theDate = SimpleDateFormat("yyyyMMdd-HHmmss", Locale.getDefault()).format(Date())

                // delete all files in the folder that are older than seven days
                // list all files in folder
                val backupFolder = createBackupFolder()
                val files = backupFolder.listFiles()

                // make sure there are files in folder first
                if (files != null) {
                    // determine today's date (to make sure there is only one backup saved each day,
                    //  sometimes Android is cheeky and executes the daily alarm on each launch of Paseo)
                    val todayDayOfYear = Calendar.getInstance()
                    todayDayOfYear.set(Calendar.HOUR_OF_DAY, 0)

                    // determine date of seven days ago (to delete files older than one week)
                    val weekAgoDate = Calendar.getInstance()
                    weekAgoDate.add(Calendar.DAY_OF_YEAR, -7)

                    // loop through all the files in the folder
                    for (theFileNum in files.indices) {
                        // only check for actual paseo backup files (based on filename)
                        if (files[theFileNum].exists() &&
                                files[theFileNum].name.substring(0, 5) == "paseo" &&
                                files[theFileNum].name.substringAfterLast(".") == "db") {
                            // get the date of the file
                            val lastModified = Date(files[theFileNum].lastModified())

                            // if the file is older than a week then delete it
                            if (lastModified.before(weekAgoDate.time)) {
                                files[theFileNum].delete()
                            }
                            if (lastModified.after(todayDayOfYear.time)) {
                                alreadyBackedToday = true
                            }
                        }
                    }
                }

                // save a new backup file (if none already saved today)
                if (!alreadyBackedToday) {
                    val backupFile = File(backupFolder.toString(), "paseoDB$theDate-a.db") // File Name
                    try {
                        dbFile.copyTo(backupFile, overwrite = true)
                    }
                    catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }
        }
    }



    private fun createBackupFolder() : File {
        val appName = getApplicationName(this).filterNot { it.isWhitespace() }
        var theFolder = File(Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS), appName)

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        // retrieve the user set backup folder from the shared preferences
        val theBackupFolderName = paseoPrefs!!.getString("theBackupFolder", "")
        val theBackupFolder = File((theBackupFolderName), "");

        // check if user selected folder is set and actually exists - if yes, then use it
        if (theBackupFolderName != "") {
            if (theBackupFolder.exists()) {
                theFolder = theBackupFolder
            }
        }
        // otherwise, default to the downloads folder
        else if (!theFolder.exists()) {
            theFolder.mkdir()
        }

        return theFolder
    }



    // rather than hardcode the name of the application, retrieve it from the application itself.
    //  this prevents potential error in the future if the name of the application is changed but the hardcoded names in the
    //  code are left unchanged (removal of effort to search through all code to implement the change)
    //  One might argue that this is ineffective owing to the application name being used in comments and variable name
    //  This is a shortcoming, and indeed only a partial solution to the problem, but still, better than no solution
    fun getApplicationName(context: Context): String {
        val applicationInfo = context.applicationInfo
        val stringId = applicationInfo.labelRes
        return if (stringId == 0) applicationInfo.nonLocalizedLabel.toString() else context.getString(stringId)
    }



    private fun UnregisterAlarmBroadcast() {
        midnightAlarmManager?.cancel(myPendingIntent!!)
        baseContext.unregisterReceiver(myBroadcastReceiver)
    }



    override fun onDestroy() {

        // turn off step counter service
        stopForeground(true)

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        val restartService = paseoPrefs!!.getBoolean("prefRestartService", true)

        // turn off auto start service
        if (restartService) {
            val broadcastIntent = Intent()
            broadcastIntent.action = "restartservice"
            broadcastIntent.setClass(this, Restarter::class.java)
            this.sendBroadcast(broadcastIntent)
        }

        // Shutdown TTS
        if (tts != null) {
            tts!!.stop()
            tts!!.shutdown()
        }

        sensorManager?.unregisterListener(this)

        UnregisterAlarmBroadcast()

        super.onDestroy()
    }



    override fun onTaskRemoved(rootIntent: Intent?) {
        super.onTaskRemoved(rootIntent)
        stopSelf()
    }



    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        startForegroundServiceWithNotification()

        super.onStartCommand(intent, flags, startId)

        return START_STICKY
    }



    private fun startForegroundServiceWithNotification() {
        val resultIntent = Intent(this, MainActivity::class.java)
        val pendingIntent = PendingIntent.getActivity(this, 0, resultIntent, PendingIntent.FLAG_IMMUTABLE)

        val builder: NotificationCompat.Builder
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channelName = getString(R.string.notification_channel_step_count)
            //val importance = NotificationManager.IMPORTANCE_LOW
            val importance = NotificationManager.IMPORTANCE_NONE
            val channelId = "PASEO_CHANNEL_ID"
            val channel = NotificationChannel(channelId, channelName, importance)
            channel.setShowBadge(true)

            builder = NotificationCompat.Builder(this, channelId)

            val notificatioManager = getSystemService(NotificationManager::class.java)
            notificatioManager.createNotificationChannel(channel)
        } else {
            builder = NotificationCompat.Builder(this)
        }

        val icon = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) R.mipmap.ic_launcher_nb else R.mipmap.ic_notification

        val dateFormat = SimpleDateFormat("yyyyMMdd", Locale.getDefault()) // looks like "20101225"
        val today = dateFormat.format(Date()).toInt()

        // if currentSteps > startSteps then
        //  do not need to create a new record
        val theSteps = paseoDBHelper.getDaysSteps(today)

        with(builder) {
            builder.setContentTitle("" + NumberFormat.getIntegerInstance().format(theSteps) + getString(R.string.steps_today))
            builder.setStyle(NotificationCompat.BigTextStyle().bigText(getString(R.string.target_notif) +
                    NumberFormat.getIntegerInstance().format(targetSteps) + ". " +
                    NumberFormat.getIntegerInstance().format(max(targetSteps - theSteps, 0)) +
                    getString(R.string.to_go)))
            setWhen(System.currentTimeMillis())
            setSmallIcon(icon)
            setContentIntent(pendingIntent)
            priority = NotificationCompat.PRIORITY_LOW

            setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
        }

        val notification = builder.build()

        // make sure app opens with user taps on notification
        notification.flags = notification.flags or NotificationCompat.FLAG_AUTO_CANCEL

        updatePaseoWidget()

        updatePaseoNotification()


        startForeground(SERVICEID, notification)
    }



    // needed for step counting (even though it is empty)
    override fun onAccuracyChanged(p0: Sensor?, p1: Int)
    {
    }



    // this function is triggered whenever there is an event on the device's step counter sensor
    //  (steps have been detected)
    override fun onSensorChanged(event: SensorEvent)
    {
        if (running)
        {
            var dateFormat = SimpleDateFormat("yyyyMMdd", Locale.getDefault()) // looks like "19891225"
            val today = dateFormat.format(Date()).toInt()
            dateFormat = SimpleDateFormat("HH", Locale.getDefault())
            val currentHour = dateFormat.format(Date()).toInt()

            if (hasStepCounter) {
                // read the step count value from the devices step counter sensor
                currentSteps = event.values[0].toInt()
            }

            // get the latest step information from the database
            if (paseoDBHelper.readRowCount() > 0) {
                latestDay = paseoDBHelper.readLastStepsDate()
                latestHour = paseoDBHelper.readLastStepsTime()
                startSteps = paseoDBHelper.readLastStartSteps()
                endSteps = paseoDBHelper.readLastEndSteps()
            }

            // hour is one more than last hour recorded -> add new hour record to database
            if(today == latestDay && currentHour == latestHour + 1 && currentSteps >= startSteps) {
                addSteps(today, currentHour, endSteps, currentSteps)
            }
            // add a new hour record (may be for current day or for a new day)
            //  also add a new record if the current steps is less than the most recent start steps (happens when phone has been rebooted)
            else if (today != latestDay || currentHour != latestHour || currentSteps < startSteps) {
                addSteps(today, currentHour, currentSteps, currentSteps)
            }
            else {
                //  set endSteps to current steps (update the end steps for current hour)
                addSteps(today, currentHour, 0, currentSteps, true)
            }

            // retrieve today's step total
            val theSteps = paseoDBHelper.getDaysSteps(today)

            updatePaseoWidget()

            updatePaseoNotification()

            // check if the user has a mini goal running and update all of the values needed
            checkMiniGoal()

            // send message to application activity so that it can react to new steps being sensed
            val local = Intent()

            local.action = "ca.chancehorizon.paseo.action"
            local.putExtra("data", theSteps)
            this.sendBroadcast(local)
        }
    }



    // update the number of steps shown in Paseo's widget
    private fun updatePaseoWidget() {
        // retrieve today's step total to show in the notification
        val dateFormat = SimpleDateFormat("yyyyMMdd", Locale.getDefault()) // looks like "19891225"
        val today = dateFormat.format(Date()).toInt()
        val theSteps = paseoDBHelper.getDaysSteps(today)

        // update the step count in the widget
        val context = this
        val appWidgetManager = AppWidgetManager.getInstance(context)
        val remoteViews = RemoteViews(context.packageName, R.layout.paseo_steps_widget)
        val remoteViews2 = RemoteViews(context.packageName, R.layout.paseo_steps_widget_black)
        val paseoWidget = ComponentName(context, PaseoStepsWidget::class.java)
        val paseoWidgetBlack = ComponentName(context, PaseoStepsWidgetBlack::class.java)
        remoteViews.setTextViewText(R.id.stepwidget_text, "" + NumberFormat.getIntegerInstance().format(theSteps))
        remoteViews2.setTextViewText(R.id.stepwidget_text, "" + NumberFormat.getIntegerInstance().format(theSteps))

        // Open App on Widget Click
        remoteViews.setOnClickPendingIntent(R.id.stepwidget_text,
                PendingIntent.getActivity(context, 0, Intent(context, MainActivity::class.java), PendingIntent.FLAG_IMMUTABLE))
        remoteViews2.setOnClickPendingIntent(R.id.stepwidget_text,
                PendingIntent.getActivity(context, 0, Intent(context, MainActivity::class.java), PendingIntent.FLAG_IMMUTABLE))

        appWidgetManager.updateAppWidget(paseoWidget, remoteViews)
        appWidgetManager.updateAppWidget(paseoWidgetBlack, remoteViews2)
    }



    // update the number of steps shown in Paseo's notification
    private fun updatePaseoNotification() {

        val resultIntent = Intent(this, MainActivity::class.java)
        val pendingIntent = PendingIntent.getActivity(this, 0, resultIntent, PendingIntent.FLAG_IMMUTABLE)

        // retrieve today's step total to show in the notification
        val dateFormat = SimpleDateFormat("yyyyMMdd", Locale.getDefault()) // looks like "20101225"
        val today = dateFormat.format(Date()).toInt()
        val theSteps = paseoDBHelper.getDaysSteps(today)

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        val notifyPercentage = paseoPrefs!!.getBoolean("prefNotificationPercent", false)

        var notificationMessage = "" + NumberFormat.getIntegerInstance().format(theSteps) + " steps today. "

        if (theSteps == 0) {
            notificationMessage = getString(R.string.NoStepsYetToday)
        }

        var notificationSubMessage: String

        if (notifyPercentage) {
            notificationSubMessage = "Target: " +
                    NumberFormat.getIntegerInstance().format(targetSteps) + " . " +
                    NumberFormat.getIntegerInstance().format(theSteps.toDouble() / targetSteps.toDouble() * 100.0) +
                    "% "
        }
        else {
            notificationSubMessage = "Target: " +
                    NumberFormat.getIntegerInstance().format(targetSteps) + " . "
            if (theSteps > targetSteps) {
                notificationSubMessage = notificationSubMessage + " Exceeded by " +
                        NumberFormat.getIntegerInstance().format(max(theSteps - targetSteps, 0))
            }
            else {
                notificationSubMessage = notificationSubMessage +
                        NumberFormat.getIntegerInstance().format(max(targetSteps - theSteps, 0)) + " to go! "
            }
        }

        // update the notification with new step count
        val builder: NotificationCompat.Builder
        val channelId = "PASEO_CHANNEL_ID"

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val notificationManager = getSystemService(NotificationManager::class.java)
            val icon = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) R.mipmap.ic_launcher_nb else R.mipmap.ic_notification

            builder = NotificationCompat.Builder(this, channelId)
            builder.setContentTitle(notificationMessage)
            builder.setStyle(NotificationCompat.BigTextStyle().bigText(notificationSubMessage))
            builder.setSmallIcon(icon)
            builder.setContentIntent(pendingIntent)
            builder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
            val notification = builder.build()

            // make sure app opens with user taps on notification
            notification.flags = notification.flags or NotificationCompat.FLAG_AUTO_CANCEL

            try {
                notificationManager.notify(SERVICEID, builder.build())
            } catch (e: Exception) {
                print(e)
            }
        }
    }



    // update mini goal fragement and speak alerts when goal or step interval achieved
    fun checkMiniGoal() {

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        val isGoalActive = paseoPrefs!!.getBoolean("prefMiniGoalActive", false)
        val useDaySteps = paseoPrefs.getBoolean("prefDaySetpsGoal", false)
        val continueAnnouncing = paseoPrefs.getBoolean("prefContinueAnnounce", false)

        // continue to announce mini goal progress if the goal is active (not yet achieved)
        //  or the user has chosen to continue announcements beyond the goal being met
        if (isGoalActive || continueAnnouncing) {

            // get the mini goal steps amount
            val miniGoalSteps = paseoPrefs.getInt("prefMiniGoalSteps", 20)

            // get the mini goal interval for text to speech announcements
            val miniGoalAlertInterval = paseoPrefs.getInt("prefMiniGoalAlertInterval", 0)

            // get the number of steps at which the next announcement will be spoken
            var miniGoalNextAlert = paseoPrefs.getInt("prefMiniGoalNextAlert", 0)

            // load the number of steps in this day at which the mini goal was started
            val miniGoalStartSteps = paseoPrefs.getInt("prefMiniGoalStartSteps", 0)

            val stepCount : Int

            // load the current number on the devices step counter sensor
            val miniGoalEndSteps = paseoDBHelper.readLastEndSteps()

            // display goal step count
            // default to the steps starting at zero (or use the day's set count, if user has set that option)
            if (!useDaySteps) {
                stepCount = miniGoalEndSteps - miniGoalStartSteps
            }
            // or get the current day's steps
            else {
                stepCount = paseoDBHelper.getDaysSteps(SimpleDateFormat("yyyyMMdd", Locale.getDefault()).format(Date()).toInt())

                // start the alert steps at the beginning number of steps for the current day
                if (miniGoalNextAlert < stepCount - miniGoalAlertInterval && miniGoalAlertInterval > 0) {
                    miniGoalNextAlert = ((stepCount + miniGoalAlertInterval)/miniGoalAlertInterval - 1) * miniGoalAlertInterval
                }
            }

            // check if mini goal has been achieved and congratulate the user if it has
            if (stepCount >= miniGoalSteps && isGoalActive) {
                // update shared preferences to flag that there is no longer a mini goal running
                val edit: SharedPreferences.Editor = paseoPrefs.edit()
                edit.putBoolean("prefMiniGoalActive", false)
                edit.apply()

                speakOut(getString(R.string.congratulations) + " $miniGoalSteps " + getString(R.string.steps) + "!")

                // even though the goal has been achieved, update the next alert steps when the user
                //  has chosen to continue announcements beyond the goal
                if (continueAnnouncing) {
                    miniGoalNextAlert = miniGoalNextAlert + miniGoalAlertInterval
                    edit.putInt("prefMiniGoalNextAlert", miniGoalNextAlert)
                    edit.apply()
                }
            }

            // mini goal not yet achieved (or user has chosen announcing to continue), announce mini goal progress at user selected interval
            else if ((stepCount >= miniGoalNextAlert) && miniGoalAlertInterval > 0) {
                // update shared preferences to save the next alert steps
                val edit: SharedPreferences.Editor = paseoPrefs.edit()

                speakOut("$miniGoalNextAlert " + getString(R.string.steps) + "!")

                // set the next step count for an announcement
                miniGoalNextAlert = miniGoalNextAlert + miniGoalAlertInterval
                edit.putInt("prefMiniGoalNextAlert", miniGoalNextAlert)
                edit.apply()
            }

            // When announcements continue after the goals has been met and the next day has started
            //  turn off announcements
            else if (stepCount < miniGoalNextAlert - miniGoalAlertInterval) {
                // update shared preferences to save the next alert steps
                val edit: SharedPreferences.Editor = paseoPrefs.edit()
                // turn off future announcements
                edit.putBoolean("prefContinueAnnounce", false)
                edit.apply()
            }
        }
    }



    // insert or update a steps record in the Paseo database
    fun addSteps(date: Int = 0, time: Int = 0, startSteps: Int = 0, endSteps: Int = 0, update: Boolean = false)
    {
        // update the endsteps for the current hour
        if (update)
        {
            var result = paseoDBHelper.updateEndSteps(StepsModel(0, date = date, hour = time,
                    startSteps = startSteps, endSteps = endSteps))
        }
        else
        {
            var result = paseoDBHelper.insertSteps(StepsModel(0, date = date, hour = time,
                    startSteps = startSteps, endSteps = endSteps))
        }

        latestDay = date
    }



    // use text to speech to "speak" some text
    fun speakOut(theText : String) {

        val paseoPrefs = this.getSharedPreferences("ca.chancehorizon.paseo_preferences", 0)

        val ttsPitch = paseoPrefs!!.getFloat("prefVoicePitch", 100F)
        val ttsRate = paseoPrefs.getFloat("prefVoiceRate", 100F)

        // set the voice to use to speak with
        val ttsVoice = paseoPrefs.getString("prefVoiceLanguage", "en_US - en-US-language")
        val ttsLocale1 = ttsVoice!!.substring(0, 2)
        val ttsLocale2 = ttsVoice.substring(3)
        val voiceobj = Voice(ttsVoice, Locale(ttsLocale1, ttsLocale2), 1, 1, false, null)
        tts?.voice = voiceobj

        tts?.setPitch(ttsPitch / 100)
        tts?.setSpeechRate(ttsRate / 100)

        var ttsResult = tts?.speak(theText, TextToSpeech.QUEUE_FLUSH, null, "")

        if(ttsResult == -1) {
            tts = TextToSpeech(this, this)
            ttsResult = tts?.speak(theText, TextToSpeech.QUEUE_FLUSH, null, "")
        }
    }



    private fun exportDBFile(result: ActivityResult) {
        val intent = result.data

        // get the steps database
        val dbFile = File("/data/data/ca.chancehorizon.paseo/databases/paseoDB.db")
        if (dbFile.exists()) {
            File(intent.toString()).delete()
            try {
                try {
                    val uri: Uri = intent?.getData()!!
                    val outputStream: OutputStream? = this.getContentResolver()?.openOutputStream(uri)
                    val inputStream: InputStream? = this.getContentResolver()?.openInputStream(dbFile.toUri())

                    val buffer = ByteArray(1024)
                    var read: Int

                    while (inputStream?.read(buffer).also { read = it!! } !== -1) {
                        outputStream?.write(buffer, 0, read)
                    }
                    inputStream?.close()
                    outputStream?.flush()
                    outputStream?.close()

                    Toast.makeText(this, "Paseo database exported successfully", Toast.LENGTH_SHORT).show()
                } catch (e: IOException) {
                    Toast.makeText(this, "Export failed", Toast.LENGTH_SHORT).show()
                }
            } catch (e: Error) {
                Toast.makeText(this, e.message, Toast.LENGTH_LONG).show()
            }
        }
    }



    companion object {
        private const val STEPDETECTED = 1
        private const val NOSTEPDETECTED = 0
        private var LASTDETECTION = NOSTEPDETECTED
    }
}
