/*
 * Copyright © All Contributors. See LICENSE and AUTHORS in the root directory for details.
 */

package at.bitfire.davdroid.resource

/**
 * This is an interface between the Syncer/SyncManager and a collection in the local storage.
 *
 * It defines operations that are used during sync for all sync data types.
 */
interface LocalCollection<out T: LocalResource> {

    /** a tag that uniquely identifies the collection (DAVx5-wide) */
    val tag: String

    /** ID of the collection in the database (corresponds to [at.bitfire.davdroid.db.Collection.id]) */
    val dbCollectionId: Long?

    /** collection title (used for user notifications etc.) **/
    val title: String

    var lastSyncState: SyncState?

    /**
     * Whether the collection should be treated as read-only on sync.
     * Stops uploading dirty events (Server side changes are still downloaded).
     */
    val readOnly: Boolean

    /**
     * Finds local resources of this collection which have been marked as *deleted* by the user
     * or an app acting on their behalf.
     *
     * @return list of resources marked as *deleted*
     */
    fun findDeleted(): List<T>

    /**
     * Finds local resources of this collection which have been marked as *dirty*, i.e. resources
     * which have been modified by the user or an app acting on their behalf.
     *
     * @return list of resources marked as *dirty*
     */
    fun findDirty(): List<T>

    /**
     * Finds a local resource of this collection with a given file name. (File names are assigned
     * by the sync adapter.)
     *
     * @param name file name to look for
     * @return resource with the given name, or null if none
     */
    fun findByName(name: String): T?

    /**
     * Updates the flags value for entries which are not dirty.
     *
     * @param flags    value of flags to set (for instance, [LocalResource.FLAG_REMOTELY_PRESENT]])
     *
     * @return         number of marked entries
     */
    fun markNotDirty(flags: Int): Int

    /**
     * Removes entries which are not dirty with a given flag combination.
     *
     * @param flags    exact flags value to remove entries with (for instance, if this is [LocalResource.FLAG_REMOTELY_PRESENT]],
     *                 all entries with exactly this flag will be removed)
     *
     * @return         number of removed entries
     */
    fun removeNotDirtyMarked(flags: Int): Int


    /**
     * Forgets the ETags of all members so that they will be reloaded from the server during sync.
     */
    fun forgetETags()

}