/*
 * Copyright (c) 2014 - 2025 askanimus@gmail.com
 *
 * This File is part of "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2).
 * "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2) is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2).  If not, see <http://www.gnu.org/licenses/>.
 * Diese Datei ist Teil von "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2).
 * "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2) ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder (nach Ihrer Wahl) jeder neueren
 * veröffentlichten Version, weiter verteilen und/oder modifizieren.
 * "Arbeitszeitkonto" (askanimus.arbeitszeiterfassung2) wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <https://www.gnu.org/licenses/>.
 */

package askanimus.arbeitszeiterfassung2.feiertage;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ExpandableListView;
import android.widget.ImageView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import java.util.ArrayList;
import java.util.Locale;

import askanimus.arbeitszeiterfassung2.Datum;
import askanimus.arbeitszeiterfassung2.Ics.IIcs;
import askanimus.arbeitszeiterfassung2.R;
import askanimus.arbeitszeiterfassung2.setup.ASettings;
import askanimus.arbeitszeiterfassung2.setup.ISettings;

public class FeiertageFragment
        extends
        Fragment
        implements
        ISettings,
        IIcs,
        FeiertageListAdapter.FeiertageCallbacks
{
    private static final int OPEN_ICS_FILE = 2;

    private ExpandableListView lFeiertage;
    private FeiertageListAdapter mAdapter;

    // Der Fragmentcontext
    private Context mContext;
    int openGroup = -1;


    /*
     * Neue Instanz anlegen
     */
    public static FeiertageFragment newInstance(long datum) {
        FeiertageFragment fragment = new FeiertageFragment();
        Bundle args = new Bundle();
        args.putLong(ARG_DATUM, datum);
        fragment.setArguments(args);

        return fragment;
    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        super.onCreateView(inflater, container, savedInstanceState);
        mContext = getContext();
        return inflater.inflate(R.layout.fragment_feiertage, container, false);
    }

    @Override
    public void onResume() {
        super.onResume();
        ASettings.init(mContext, this::resume);
    }

    void resume() {
        Datum mDatum;
        Bundle mArgs = getArguments();
        if (mArgs != null) {
            mDatum = new Datum(mArgs.getLong(ARG_DATUM), ASettings.aktJob.getWochenbeginn());
        } else {
            mDatum = new Datum(ASettings.aktDatum);
        }

        // View befüllen
        View mInhalt = getView();
        if (mInhalt != null) {
            // Standartelemente finden
            lFeiertage = mInhalt.findViewById(R.id.F_liste);
            ImageView bAddFeiertag = mInhalt.findViewById(R.id.F_add);
            ImageView bImportFeiertag = mInhalt.findViewById(R.id.F_import);

            bAddFeiertag.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    int jahr;
                    jahr = openGroup < 0 ?
                            ASettings.feiertage.getPosJahr(mDatum.getJahr()) :
                            mAdapter.getGroup(openGroup).getJahr();
                    Feiertag ft = new Feiertag(jahr, 6, 15, getString(R.string.feiertag_bergfest));
                    onFeiertagEdit(ft);
                }
            });

            bImportFeiertag.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
                    intent.addCategory(Intent.CATEGORY_OPENABLE);
                    intent.setType(DATEI_TYP_ICS);

                    startActivityForResult(intent, OPEN_ICS_FILE);
                }
            });

            // Feiertagsliste einrichten
            mAdapter = new FeiertageListAdapter(mContext, ASettings.feiertage, mDatum.getJahr(), this);
            lFeiertage.setAdapter(mAdapter);

            // handler für das öffnen der Kindelemente, nur eins soll zur gleichen Zeit offen sein
            lFeiertage.setOnGroupExpandListener(new ExpandableListView.OnGroupExpandListener() {

                @Override
                public void onGroupExpand(int groupPosition) {
                    if (groupPosition != openGroup)
                        lFeiertage.collapseGroup(openGroup);
                    openGroup = groupPosition;
                }
            });
            if(ASettings.feiertage.getAnzahlJahre() > 0) {
                lFeiertage.expandGroup(ASettings.feiertage.getPosJahr(mDatum.getJahr()), true);
            }
        }
    }


    @Override
    public void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == OPEN_ICS_FILE && resultCode == Activity.RESULT_OK) {
            Uri uri;
            if (data != null) {
                uri = data.getData();
                if (uri != null) {
                    ArrayList<Feiertag> fl = ASettings.feiertage.FeiertageIcsImport(mContext, uri);
                    if( fl != null){
                        int anzahl = ASettings.feiertage.FeiertageListeImport(fl);
                        Toast.makeText(
                                mContext,
                                String.format(
                                        Locale.getDefault(),
                                        "%d %s",
                                        anzahl,
                                        getString(R.string.toast_import_erfolg)
                                ),
                                Toast.LENGTH_LONG
                        ).show();
                    } else {
                        Toast.makeText(mContext, getString(R.string.toast_import_miserfolg), Toast.LENGTH_LONG).show();
                    }
                }
            }
        }
    }

    @Override
    public void onFeiertagEdit(Feiertag editFeiertag) {
        FragmentManager fManager;
        fManager = getParentFragmentManager();

        // erweiterter Eingabedialog
        FragmentDialogEditFeiertag mDialog = new FragmentDialogEditFeiertag(
                editFeiertag,
                fManager,
                mAdapter
        );
        mDialog.show(fManager, "EditFeiertagDialog");
    }

    @Override
    public void onFeiertagKopie(int posJahr, int posTag) {
        Feiertag fQeulle = ASettings.feiertage.getFeiertag(posJahr, posTag);
        Feiertag fNeu = new Feiertag(
                fQeulle.getJahr()+1,
                fQeulle.getDatumStart().getMonat(),
                fQeulle.getDatumStart().getTagimMonat(),
                fQeulle.getName()
                );

        if (ASettings.feiertage.add(fNeu)){
            mAdapter.notifyDataSetChanged();
            fNeu.save();
            Toast.makeText(
                    mContext,
                    getString(R.string.copy_ok, getString(R.string.feiertage)),
                    Toast.LENGTH_LONG
            ).show();
        } else {
            Toast.makeText(
                    mContext,
                    getString(
                            R.string.copy_fail,
                            getString(R.string.feiertage),
                            getString(R.string.feiertag_existiert)
                    ),
                    Toast.LENGTH_LONG
            ).show();
        }
    }

    @Override
    public void onFeiertagDelete(int posJahr, int posTag) {
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
        String name = ASettings.feiertage.getFeiertag(posJahr, posTag).getName();
        int jahr = ASettings.feiertage.getJahr(posJahr);
        builder.setMessage(getString(
                R.string.dialog_delete_frage, name))
                .setIcon(R.mipmap.ic_launcher_round)
                .setTitle(R.string.delete)
                .setPositiveButton(R.string.ja, (dialog, id) -> {
                    if(ASettings.feiertage.loescheFeiertag(posJahr, posTag)){
                        mAdapter.notifyDataSetChanged();
                        Toast.makeText(
                                mContext,
                                String.format("%s %d %s",name, jahr, getString(R.string.delete_toast_erfolg)),
                                Toast.LENGTH_LONG
                        ).show();
                    }
                })
                .setNegativeButton(R.string.nein, (dialog, id) -> {
                    // nichts tun
                });
        // den Dialog erzeugen und anzeigen
        builder.create().show();
    }

    @Override
    public void onJahrDelete(int posJahr) {
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
        String jahr = String.valueOf(ASettings.feiertage.getJahr(posJahr));
        builder.setMessage(getString(R.string.dialog_delete_frage, jahr))
                .setIcon(R.mipmap.ic_launcher_round)
                .setTitle(R.string.delete)
                .setPositiveButton(R.string.ja, (dialog, id) -> {
                    if(ASettings.feiertage.loescheJahr(posJahr)){
                        mAdapter.notifyDataSetChanged();
                        Toast.makeText(
                                mContext,
                                String.format("%s %s",jahr ,getString(R.string.delete_toast_erfolg)),
                                Toast.LENGTH_LONG
                        ).show();
                    }
                })
                .setNegativeButton(R.string.nein, (dialog, id) -> {
                    // nichts tun
                });
        // den Dialog erzeugen und anzeigen
        builder.create().show();

    }
}
