import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'package:wispar/generated_l10n/app_localizations.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:wispar/theme_provider.dart';
import 'package:wispar/locale_provider.dart';

class DisplaySettingsScreen extends StatefulWidget {
  const DisplaySettingsScreen({super.key});

  @override
  DisplaySettingsScreenState createState() => DisplaySettingsScreenState();
}

class DisplaySettingsScreenState extends State<DisplaySettingsScreen> {
  int _pdfThemeOption = 0;
  int _pdfOrientationOption = 0;

  bool _showPublicationCount = false;

  final Map<String, String> _languageLabels = {
    'en': 'English',
    'fr': 'Français',
    'es': 'Español',
    'nb': 'Norsk bokmål',
    'ta': 'தமிழ்',
    'nl': 'Nederlands',
    'fa': 'فارسی',
    'tr': 'Türkçe',
    'ru': 'Русский',
    'ja': '日本語',
    'id': 'Bahasa Indonesia',
    'pt': 'Português',
    'de': 'Deutsch',
    'zh': '中文',
  };

  final List<Locale> _supportedLocales = [
    Locale('en'),
    Locale('fr'),
    Locale('es'),
    Locale('nb'),
    Locale('ta'),
    Locale('nl'),
    Locale('fa'),
    Locale('tr'),
    Locale('ru'),
    Locale('ja'),
    Locale('id'),
    Locale('pt'),
    Locale('de'),
    Locale('zh'),
  ];

  @override
  void initState() {
    super.initState();
    _loadPdfThemeOption();
    _loadPdfOrientationOption();
    _loadShowPublicationCount();
  }

  void _loadPdfThemeOption() async {
    final prefs = await SharedPreferences.getInstance();
    setState(() {
      _pdfThemeOption = prefs.getInt('pdfThemeOption') ?? 0;
    });
  }

  void _savePdfThemeOption(int value) async {
    final prefs = await SharedPreferences.getInstance();
    prefs.setInt('pdfThemeOption', value);
  }

  void _loadPdfOrientationOption() async {
    final prefs = await SharedPreferences.getInstance();
    setState(() {
      _pdfOrientationOption = prefs.getInt('pdfOrientationOption') ?? 0;
    });
  }

  void _savePdfOrientationOption(int value) async {
    final prefs = await SharedPreferences.getInstance();
    prefs.setInt('pdfOrientationOption', value);
  }

  void _loadShowPublicationCount() async {
    final prefs = await SharedPreferences.getInstance();
    setState(() {
      _showPublicationCount = prefs.getBool('showPublicationCount') ?? false;
    });
  }

  void _saveShowPublicationCount(bool value) async {
    final prefs = await SharedPreferences.getInstance();
    prefs.setBool('showPublicationCount', value);
  }

  @override
  Widget build(BuildContext context) {
    final items = [
      {
        "icon": Icons.brightness_4_outlined,
        "label": AppLocalizations.of(context)!.appearance,
        "subtitle": _getThemeSubtitle(
          context,
          Provider.of<ThemeProvider>(context).themeMode,
        ),
        "onTap": () => _showThemeDialog(context),
      },
      {
        "icon": Icons.picture_as_pdf_outlined,
        "label": AppLocalizations.of(context)!.pdfTheme,
        "subtitle": _getPdfThemeSubtitle(_pdfThemeOption),
        "onTap": () => _showPdfThemeDialog(context),
      },
      {
        "icon": Icons.screen_rotation_alt_rounded,
        "label": AppLocalizations.of(context)!.pdfReadingOrientation,
        "subtitle": _getPdfOrientationSubtitle(_pdfOrientationOption),
        "onTap": () => _showPdfOrientationDialog(context),
      },
      {
        "icon": Icons.format_list_numbered,
        "label": AppLocalizations.of(context)!.showPublicationCount,
        "subtitle": _showPublicationCount
            ? AppLocalizations.of(context)!.enabled
            : AppLocalizations.of(context)!.disabled,
        "onTap": () => {},
      },
      {
        "icon": Icons.language,
        "label": AppLocalizations.of(context)!.language,
        "subtitle": _getLocaleLabel(context),
        "onTap": () => _showLanguageDialog(context),
      },
    ];

    return Scaffold(
      appBar: AppBar(
        title: Text(AppLocalizations.of(context)!.displaySettings),
      ),
      body: SafeArea(
        child: LayoutBuilder(
          builder: (context, constraints) {
            final double maxTileWidth = 400;
            final int crossAxisCount =
                (constraints.maxWidth / maxTileWidth).floor().clamp(1, 4);

            final double tileWidth =
                (constraints.maxWidth - (crossAxisCount - 1) * 16) /
                    crossAxisCount;

            final double minTileHeight = 100;

            return GridView.builder(
                padding: const EdgeInsets.all(16),
                gridDelegate: SliverGridDelegateWithFixedCrossAxisCount(
                  crossAxisCount: crossAxisCount,
                  mainAxisSpacing: 16,
                  crossAxisSpacing: 16,
                  childAspectRatio: tileWidth / minTileHeight,
                ),
                itemCount: items.length,
                itemBuilder: (context, index) {
                  final item = items[index];

                  // Publication count tile
                  if (item["label"] ==
                      AppLocalizations.of(context)!.showPublicationCount) {
                    return Card(
                      elevation: 2,
                      child: InkWell(
                        onTap: () {
                          setState(() =>
                              _showPublicationCount = !_showPublicationCount);
                          _saveShowPublicationCount(_showPublicationCount);
                        },
                        child: Padding(
                          padding: const EdgeInsets.symmetric(
                              horizontal: 16, vertical: 14),
                          child: Row(
                            children: [
                              Icon(item["icon"] as IconData),
                              const SizedBox(width: 12),
                              Expanded(
                                child: Text(
                                  item["label"] as String,
                                ),
                              ),
                              Switch(
                                value: _showPublicationCount,
                                onChanged: (value) {
                                  setState(() => _showPublicationCount = value);
                                  _saveShowPublicationCount(value);
                                },
                              ),
                            ],
                          ),
                        ),
                      ),
                    );
                  }

                  // Default tiles
                  return Card(
                    elevation: 2,
                    child: InkWell(
                      onTap: item["onTap"] as VoidCallback,
                      child: Padding(
                        padding: const EdgeInsets.symmetric(
                            horizontal: 16, vertical: 14),
                        child: Column(
                          crossAxisAlignment: CrossAxisAlignment.start,
                          mainAxisAlignment: MainAxisAlignment.center,
                          children: [
                            Row(
                              children: [
                                Icon(item["icon"] as IconData),
                                const SizedBox(width: 8),
                                Expanded(child: Text(item["label"] as String)),
                              ],
                            ),
                            if (item["subtitle"] != null)
                              Padding(
                                padding:
                                    const EdgeInsets.only(top: 8, left: 32),
                                child: Text(
                                  item["subtitle"] as String,
                                  style: const TextStyle(
                                      fontSize: 13, color: Colors.grey),
                                ),
                              ),
                          ],
                        ),
                      ),
                    ),
                  );
                });
          },
        ),
      ),
    );
  }

  void _showThemeDialog(BuildContext context) {
    final themeProvider = Provider.of<ThemeProvider>(context, listen: false);

    showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.appearance),
          content: RadioGroup<ThemeMode>(
            groupValue: themeProvider.themeMode,
            onChanged: (ThemeMode? value) {
              if (value != null) {
                themeProvider.setThemeMode(value);
                Navigator.of(context).pop();
              }
            },
            child: Column(
              mainAxisSize: MainAxisSize.min,
              children: <Widget>[
                RadioListTile<ThemeMode>(
                  title: Text(AppLocalizations.of(context)!.light),
                  value: ThemeMode.light,
                ),
                RadioListTile<ThemeMode>(
                  title: Text(AppLocalizations.of(context)!.dark),
                  value: ThemeMode.dark,
                ),
                RadioListTile<ThemeMode>(
                  title: Text(AppLocalizations.of(context)!.systemtheme),
                  value: ThemeMode.system,
                ),
              ],
            ),
          ),
        );
      },
    );
  }

  void _showPdfThemeDialog(BuildContext context) {
    showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.pdfTheme),
          content: RadioGroup<int>(
            groupValue: _pdfThemeOption,
            onChanged: (int? value) {
              if (value != null) {
                setState(() => _pdfThemeOption = value);
                _savePdfThemeOption(value);
                Navigator.of(context).pop();
              }
            },
            child: Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                RadioListTile<int>(
                  title: Text(AppLocalizations.of(context)!.light),
                  value: 0,
                ),
                RadioListTile<int>(
                  title: Text(AppLocalizations.of(context)!.dark),
                  value: 1,
                ),
                RadioListTile<int>(
                  title: Text(AppLocalizations.of(context)!.systemtheme),
                  value: 2,
                ),
              ],
            ),
          ),
        );
      },
    );
  }

  String _getPdfThemeSubtitle(int option) {
    switch (option) {
      case 0:
        return AppLocalizations.of(context)!.light;
      case 1:
        return AppLocalizations.of(context)!.dark;
      case 2:
        return AppLocalizations.of(context)!.systemtheme;
      default:
        return 'Unknown';
    }
  }

  void _showPdfOrientationDialog(BuildContext context) {
    showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.pdfReadingOrientation),
          content: RadioGroup<int>(
            groupValue: _pdfOrientationOption,
            onChanged: (int? value) {
              if (value != null) {
                setState(() => _pdfOrientationOption = value);
                _savePdfOrientationOption(value);
                Navigator.of(context).pop();
              }
            },
            child: Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                RadioListTile<int>(
                  title: Text(AppLocalizations.of(context)!.vertical),
                  value: 0,
                ),
                RadioListTile<int>(
                  title: Text(AppLocalizations.of(context)!.horizontal),
                  value: 1,
                ),
              ],
            ),
          ),
        );
      },
    );
  }

  String _getPdfOrientationSubtitle(int option) {
    switch (option) {
      case 0:
        return AppLocalizations.of(context)!.vertical;
      case 1:
        return AppLocalizations.of(context)!.horizontal;
      default:
        return 'Unknown';
    }
  }

  void _showLanguageDialog(BuildContext context) {
    final provider = Provider.of<LocaleProvider>(context, listen: false);
    final currentLang = provider.locale?.languageCode ?? 'system';

    // Sort languages alphabetically
    final sortedLocales = [..._supportedLocales]..sort((a, b) {
        final labelA = _languageLabels[a.languageCode] ?? a.languageCode;
        final labelB = _languageLabels[b.languageCode] ?? b.languageCode;
        return labelA.compareTo(labelB);
      });

    showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.language),
          content: SingleChildScrollView(
            child: RadioGroup<String>(
              groupValue: currentLang,
              onChanged: (value) {
                if (value == 'system') {
                  provider.clearLocale();
                } else {
                  provider.setLocale(value!);
                }
                Navigator.of(context).pop();
              },
              child: Column(
                mainAxisSize: MainAxisSize.min,
                children: [
                  ...sortedLocales.map((locale) {
                    final code = locale.languageCode;
                    return RadioListTile<String>(
                      title: Text(_languageLabels[code] ?? code),
                      value: code,
                    );
                  }),
                  RadioListTile<String>(
                    title: Text(AppLocalizations.of(context)!.system),
                    value: 'system',
                  ),
                ],
              ),
            ),
          ),
        );
      },
    );
  }

  String _getLocaleLabel(BuildContext context) {
    final locale = Provider.of<LocaleProvider>(context).locale;
    if (locale == null) {
      return AppLocalizations.of(context)!.system;
    }
    return _languageLabels[locale.languageCode] ??
        AppLocalizations.of(context)!.system;
  }

  String _getThemeSubtitle(BuildContext context, ThemeMode? themeMode) {
    switch (themeMode) {
      case ThemeMode.light:
        return AppLocalizations.of(context)!.light;
      case ThemeMode.dark:
        return AppLocalizations.of(context)!.dark;
      case ThemeMode.system:
        return AppLocalizations.of(context)!.systemtheme;
      default:
        return 'Unknown';
    }
  }
}
