import type { Mode } from "@ionic/core";

/**
 * Most storage happens asynchronously with Dexie (indexeddb).
 *
 * However, some things we need syncronously (localStorage)
 * to more easily avoid FOUC (e.g. dark vs light mode)
 *
 * So, those critical settings are stored here.
 */
import { isAndroid } from "#/helpers/device";
import { DeepPartial } from "#/helpers/typescript";

import type { SettingsState } from "./settingsSlice";

export const LOCALSTORAGE_KEYS = {
  FONT: {
    FONT_SIZE_MULTIPLIER: "appearance--font-size-multiplier",
    USE_SYSTEM: "appearance--font-use-system",
  },
  DARK: {
    USE_SYSTEM: "appearance--dark-use-system",
    USER_MODE: "appearance--dark-user-mode",
    PURE_BLACK: "appearance--pure-black",
  },
  DEVICE_MODE: "appearance--device-mode",
  THEME: "appearance--theme",
} as const;

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export function get(key: string): any {
  const data = localStorage.getItem(key);
  if (!data) return;
  return JSON.parse(data);
}

export function set(key: string, value: unknown) {
  localStorage.setItem(key, JSON.stringify(value));
}

export function getLocalStorageInitialState(): DeepPartial<SettingsState> {
  return {
    appearance: {
      dark: {
        pureBlack: get(LOCALSTORAGE_KEYS.DARK.PURE_BLACK),
        userDarkMode: get(LOCALSTORAGE_KEYS.DARK.USER_MODE),
        usingSystemDarkMode: get(LOCALSTORAGE_KEYS.DARK.USE_SYSTEM),
      },
      deviceMode: getDeviceMode(),
      font: {
        fontSizeMultiplier: get(LOCALSTORAGE_KEYS.FONT.FONT_SIZE_MULTIPLIER),
        useSystemFontSize: get(LOCALSTORAGE_KEYS.FONT.USE_SYSTEM),
      },
      theme: get(LOCALSTORAGE_KEYS.THEME),
    },
  } as const;
}

export function getDeviceMode(): Mode {
  const mode = get(LOCALSTORAGE_KEYS.DEVICE_MODE);

  if (isAndroid() && !mode) return "md";

  // Default to ios for web and unknown
  if (!mode) return "ios";

  return mode;
}
