import { satisfies } from 'compare-versions';
import * as LemmyV0 from 'lemmy-js-client-v0';
import { z } from 'zod/v4-mini';
import * as LemmyV1 from 'lemmy-js-client-v1';
import createClient from 'openapi-fetch';

class BaseClient {
  static mode;
  static softwareName;
  /**
   * NPM semver range, e.g. ">=1.0.0 <2.0.0"
   */
  static softwareVersionRange;
}

class FediverseError extends Error {
  constructor(message, errorOptions) {
    super(message, errorOptions);
    this.name = "FediverseError";
  }
}
class InvalidPayloadError extends FediverseError {
  constructor(message) {
    super(message);
    this.name = "InvalidPayloadError";
  }
}
class UnexpectedResponseError extends FediverseError {
  constructor(message) {
    super(message);
    this.name = "UnexpectedResponseError";
  }
}
class UnsupportedError extends FediverseError {
  constructor(message) {
    super(message);
    this.name = "UnsupportedError";
  }
}
class UnsupportedSoftwareError extends UnsupportedError {
  constructor(message) {
    super(message);
    this.name = "UnsupportedSoftwareError";
  }
}

function cleanThreadiverseParams(payload) {
  delete payload.mode;
  return payload;
}

const Person = z.object({
  actor_id: z.string(),
  avatar: z.optional(z.string()),
  banner: z.optional(z.string()),
  bio: z.optional(z.string()),
  bot_account: z.boolean(),
  deleted: z.boolean(),
  display_name: z.optional(z.string()),
  id: z.number(),
  local: z.boolean(),
  name: z.string(),
  published: z.string()
});

const PersonAggregates = z.object({
  comment_count: z.number(),
  post_count: z.number()
});
const PersonView = z.object({
  counts: PersonAggregates,
  is_admin: z.boolean(),
  person: Person
});

const Comment = z.object({
  /**
   * The federated activity id / ap_id.
   */
  ap_id: z.string(),
  content: z.string(),
  creator_id: z.number(),
  /**
   * Whether the comment has been deleted by its creator.
   */
  deleted: z.boolean(),
  /**
   * Whether the comment has been distinguished(speaking officially) by a mod.
   */
  distinguished: z.boolean(),
  id: z.number(),
  language_id: z.number(),
  /**
   * Whether the comment is local.
   */
  local: z.boolean(),
  /**
   * The path / tree location of a comment, separated by dots, ending with the comment's id. Ex:
   * 0.24.27
   */
  path: z.string(),
  post_id: z.number(),
  published: z.string(),
  /**
   * Whether the comment has been removed.
   */
  removed: z.boolean(),
  updated: z.optional(z.string())
});

const CommentReply = z.object({
  comment_id: z.number(),
  id: z.number(),
  published: z.string(),
  read: z.boolean(),
  recipient_id: z.number()
});

const CommunityVisibility = z.enum([
  "Public",
  "Unlisted",
  "LocalOnlyPublic",
  "LocalOnlyPrivate",
  "Private"
]);

const Community = z.object({
  actor_id: z.string(),
  banner: z.optional(z.string()),
  deleted: z.boolean(),
  description: z.optional(z.string()),
  hidden: z.boolean(),
  icon: z.optional(z.string()),
  id: z.number(),
  local: z.boolean(),
  name: z.string(),
  nsfw: z.boolean(),
  posting_restricted_to_mods: z.boolean(),
  published: z.string(),
  removed: z.boolean(),
  title: z.string(),
  updated: z.optional(z.string()),
  visibility: CommunityVisibility
});

const Post = z.object({
  /**
   * An optional alt_text, usable for image posts.
   */
  alt_text: z.optional(z.string()),
  /**
   * The federated activity id / ap_id.
   */
  ap_id: z.string(),
  /**
   * An optional post body, in markdown.
   */
  body: z.optional(z.string()),
  community_id: z.number(),
  creator_id: z.number(),
  /**
   * Whether the post is deleted.
   */
  deleted: z.boolean(),
  /**
   * A description for the link.
   */
  embed_description: z.optional(z.string()),
  /**
   * A title for the link.
   */
  embed_title: z.optional(z.string()),
  /**
   * Whether the post is featured to its community.
   */
  featured_community: z.boolean(),
  /**
   * Whether the post is featured to its site.
   */
  featured_local: z.boolean(),
  id: z.number(),
  language_id: z.number(),
  /**
   * Whether the post is local.
   */
  local: z.boolean(),
  /**
   * Whether the post is locked.
   */
  locked: z.boolean(),
  name: z.string(),
  /**
   * Whether the post is NSFW.
   */
  nsfw: z.boolean(),
  published: z.string(),
  /**
   * Whether the post is removed.
   */
  removed: z.boolean(),
  /**
   * A thumbnail picture url.
   */
  thumbnail_url: z.optional(z.string()),
  updated: z.optional(z.string()),
  /**
   * An optional link / url for the post.
   */
  url: z.optional(z.string()),
  url_content_type: z.optional(z.string())
});

const SubscribedType = z.enum([
  "Subscribed",
  "NotSubscribed",
  "Pending",
  "ApprovalRequired"
]);

const Vote = z.number().check(z.gte(-1), z.lte(1));

const CommentAggregates = z.object({
  /**
   * The total number of children in this comment branch.
   */
  child_count: z.number(),
  comment_id: z.number(),
  downvotes: z.number(),
  published: z.string(),
  score: z.number(),
  upvotes: z.number()
});
const CommentView = z.object({
  banned_from_community: z.boolean(),
  comment: Comment,
  community: Community,
  counts: CommentAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  my_vote: z.optional(Vote),
  post: Post,
  saved: z.boolean(),
  subscribed: SubscribedType
});

const CommentReplyView = z.object({
  banned_from_community: z.boolean(),
  comment: Comment,
  comment_reply: CommentReply,
  community: Community,
  counts: CommentAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_blocked: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  my_vote: z.optional(Vote),
  post: Post,
  recipient: Person,
  saved: z.boolean(),
  subscribed: SubscribedType
});

const CommentReport = z.object({
  comment_id: z.number(),
  creator_id: z.number(),
  id: z.number(),
  original_comment_text: z.string(),
  published: z.string(),
  reason: z.string(),
  resolved: z.boolean(),
  resolver_id: z.optional(z.number()),
  updated: z.optional(z.string())
});

const CommentReportView = z.object({
  comment: Comment,
  comment_creator: Person,
  comment_report: CommentReport,
  community: Community,
  counts: CommentAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_blocked: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  my_vote: z.optional(Vote),
  post: Post,
  resolver: z.optional(Person),
  saved: z.boolean(),
  subscribed: SubscribedType
});

const BaseCommunityAggregates = z.object({
  comments: z.number(),
  posts: z.number(),
  subscribers: z.number()
});
const ExtendedCommunityAggregates = z.object({
  comments: z.number(),
  posts: z.number(),
  subscribers: z.number(),
  subscribers_local: z.number(),
  /**
   * The number of users with any activity in the last day.
   */
  users_active_day: z.number(),
  /**
   * The number of users with any activity in the last year.
   */
  users_active_half_year: z.number(),
  /**
   * The number of users with any activity in the last month.
   */
  users_active_month: z.number(),
  /**
   * The number of users with any activity in the last week.
   */
  users_active_week: z.number()
});
const CommunityAggregates = z.union([
  BaseCommunityAggregates,
  ExtendedCommunityAggregates
]);

const CommunityFollowerView = z.object({
  community: Community,
  follower: Person
});

const CommunityModeratorView = z.object({
  community: Community,
  moderator: Person
});

const CommunityView = z.object({
  blocked: z.boolean(),
  community: Community,
  counts: CommunityAggregates,
  subscribed: SubscribedType
});

const ReadableFederationState = z.object({
  /**
   * how many failed attempts have been made to send the next activity
   */
  fail_count: z.number(),
  instance_id: z.number(),
  /**
   * timestamp of the last retry attempt (when the last failing activity was resent)
   */
  last_retry: z.optional(z.string()),
  /**
   * the last successfully sent activity id
   */
  last_successful_id: z.optional(z.number()),
  last_successful_published_time: z.optional(z.string()),
  /**
   * timestamp of the next retry attempt (null if fail count is 0)
   */
  next_retry: z.optional(z.string())
});

const InstanceWithFederationState = z.object({
  domain: z.string(),
  /**
   * if federation to this instance is or was active, show state of outgoing federation to this
   * instance
   */
  federation_state: z.optional(ReadableFederationState),
  id: z.number(),
  published: z.string(),
  software: z.optional(z.string()),
  updated: z.optional(z.string()),
  version: z.optional(z.string())
});

const FederatedInstances = z.object({
  allowed: z.array(InstanceWithFederationState),
  blocked: z.array(InstanceWithFederationState),
  linked: z.array(InstanceWithFederationState)
});

const CaptchaResponse = z.object({
  /**
   * A Base64 encoded png
   */
  png: z.string(),
  /**
   * The UUID for the captcha item.
   */
  uuid: z.string(),
  /**
   * A Base64 encoded wav audio
   */
  wav: z.string()
});
const GetCaptchaResponse = z.object({
  /**
   * Will be None if captchas are disabled.
   */
  ok: z.optional(CaptchaResponse)
});

const GetCommunityResponse = z.object({
  community_view: CommunityView,
  moderators: z.array(CommunityModeratorView)
});

const ModRemovePost = z.object({
  id: z.number(),
  mod_person_id: z.number(),
  post_id: z.number(),
  reason: z.optional(z.string()),
  removed: z.boolean(),
  when_: z.string()
});
const ModLockPost = z.object({
  id: z.number(),
  locked: z.boolean(),
  mod_person_id: z.number(),
  post_id: z.number(),
  when_: z.string()
});
const ModFeaturePost = z.object({
  featured: z.boolean(),
  id: z.number(),
  is_featured_community: z.boolean(),
  mod_person_id: z.number(),
  post_id: z.number(),
  when_: z.string()
});
const ModRemoveComment = z.object({
  comment_id: z.number(),
  id: z.number(),
  mod_person_id: z.number(),
  reason: z.optional(z.string()),
  removed: z.boolean(),
  when_: z.string()
});
const ModRemoveCommunity = z.object({
  community_id: z.number(),
  id: z.number(),
  mod_person_id: z.number(),
  reason: z.optional(z.string()),
  removed: z.boolean(),
  when_: z.string()
});
const ModBanFromCommunity = z.object({
  banned: z.boolean(),
  community_id: z.number(),
  expires: z.optional(z.string()),
  id: z.number(),
  mod_person_id: z.number(),
  other_person_id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const ModBan = z.object({
  banned: z.boolean(),
  expires: z.optional(z.string()),
  id: z.number(),
  mod_person_id: z.number(),
  other_person_id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const ModAddCommunity = z.object({
  community_id: z.number(),
  id: z.number(),
  mod_person_id: z.number(),
  other_person_id: z.number(),
  removed: z.boolean(),
  when_: z.string()
});
const ModTransferCommunity = z.object({
  community_id: z.number(),
  id: z.number(),
  mod_person_id: z.number(),
  other_person_id: z.number(),
  when_: z.string()
});
const ModAdd = z.object({
  id: z.number(),
  mod_person_id: z.number(),
  other_person_id: z.number(),
  removed: z.boolean(),
  when_: z.string()
});
const AdminPurgePerson = z.object({
  admin_person_id: z.number(),
  id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const AdminPurgeCommunity = z.object({
  admin_person_id: z.number(),
  id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const AdminPurgePost = z.object({
  admin_person_id: z.number(),
  community_id: z.number(),
  id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const AdminPurgeComment = z.object({
  admin_person_id: z.number(),
  id: z.number(),
  post_id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const ModHideCommunity = z.object({
  community_id: z.number(),
  hidden: z.boolean(),
  id: z.number(),
  mod_person_id: z.number(),
  reason: z.optional(z.string()),
  when_: z.string()
});
const ModRemovePostView = z.object({
  community: Community,
  mod_remove_post: ModRemovePost,
  moderator: z.optional(Person),
  post: Post
});
const ModLockPostView = z.object({
  community: Community,
  mod_lock_post: ModLockPost,
  moderator: z.optional(Person),
  post: Post
});
const ModFeaturePostView = z.object({
  community: Community,
  mod_feature_post: ModFeaturePost,
  moderator: z.optional(Person),
  post: Post
});
const ModRemoveCommentView = z.object({
  comment: Comment,
  commenter: Person,
  community: Community,
  mod_remove_comment: ModRemoveComment,
  moderator: z.optional(Person),
  post: Post
});
const ModRemoveCommunityView = z.object({
  community: Community,
  mod_remove_community: ModRemoveCommunity,
  moderator: z.optional(Person)
});
const ModBanFromCommunityView = z.object({
  banned_person: Person,
  community: Community,
  mod_ban_from_community: ModBanFromCommunity,
  moderator: z.optional(Person)
});
const ModBanView = z.object({
  banned_person: Person,
  mod_ban: ModBan,
  moderator: z.optional(Person)
});
const ModAddCommunityView = z.object({
  community: Community,
  mod_add_community: ModAddCommunity,
  modded_person: Person,
  moderator: z.optional(Person)
});
const ModTransferCommunityView = z.object({
  community: Community,
  mod_transfer_community: ModTransferCommunity,
  modded_person: Person,
  moderator: z.optional(Person)
});
const ModAddView = z.object({
  mod_add: ModAdd,
  modded_person: Person,
  moderator: z.optional(Person)
});
const AdminPurgePersonView = z.object({
  admin: z.optional(Person),
  admin_purge_person: AdminPurgePerson
});
const AdminPurgeCommunityView = z.object({
  admin: z.optional(Person),
  admin_purge_community: AdminPurgeCommunity
});
const AdminPurgePostView = z.object({
  admin: z.optional(Person),
  admin_purge_post: AdminPurgePost,
  community: Community
});
const AdminPurgeCommentView = z.object({
  admin: z.optional(Person),
  admin_purge_comment: AdminPurgeComment,
  post: Post
});
const ModHideCommunityView = z.object({
  admin: z.optional(Person),
  community: Community,
  mod_hide_community: ModHideCommunity
});
const ModlogItem = z.union([
  ModRemovePostView,
  ModLockPostView,
  ModFeaturePostView,
  ModRemoveCommentView,
  ModRemoveCommunityView,
  ModBanFromCommunityView,
  ModBanView,
  ModAddCommunityView,
  ModTransferCommunityView,
  ModAddView,
  AdminPurgePersonView,
  AdminPurgeCommunityView,
  AdminPurgePostView,
  AdminPurgeCommentView,
  ModHideCommunityView
]);

const GetPersonDetailsResponse = z.object({
  // site?: Site;
  moderates: z.array(CommunityModeratorView),
  person_view: PersonView
});

const LinkMetadata = z.object({
  content_type: z.optional(z.string()),
  description: z.optional(z.string()),
  embed_video_url: z.optional(z.string()),
  image: z.optional(z.string()),
  title: z.optional(z.string())
});
const GetSiteMetadataResponse = z.object({
  metadata: LinkMetadata
});

const Instance = z.object({
  domain: z.string(),
  id: z.number(),
  published: z.string(),
  software: z.optional(z.string()),
  updated: z.optional(z.string()),
  version: z.optional(z.string())
});

const RegistrationMode = z.enum([
  "Closed",
  "RequireApplication",
  "Open"
]);

const SiteAggregates = z.object({
  comments: z.number(),
  communities: z.number(),
  posts: z.number(),
  users: z.number(),
  /**
   * The number of users with any activity in the last day.
   */
  users_active_day: z.number(),
  /**
   * The number of users with any activity in the last half year.
   */
  users_active_half_year: z.number(),
  /**
   * The number of users with any activity in the last month.
   */
  users_active_month: z.number(),
  /**
   * The number of users with any activity in the last week.
   */
  users_active_week: z.number()
});

const MyUserInfo = z.object({
  community_blocks: z.array(Community),
  follows: z.array(CommunityFollowerView),
  instance_blocks: z.array(Instance),
  local_user_view: z.object({
    counts: PersonAggregates,
    local_user: z.object({
      admin: z.boolean(),
      show_nsfw: z.boolean()
    }),
    person: Person
  }),
  moderates: z.array(CommunityModeratorView),
  person_blocks: z.array(Person)
});
const FederationMode = z.enum(["All", "Local", "Disable"]);
const LocalSite = z.object({
  /**
   * An optional registration application questionnaire in markdown.
   */
  application_question: z.optional(z.string()),
  captcha_enabled: z.boolean(),
  /**
   * What kind of comment downvotes your site allows.
   */
  comment_downvotes: FederationMode,
  /**
   * What kind of comment upvotes your site allows.
   */
  comment_upvotes: FederationMode,
  legal_information: z.optional(z.string()),
  /**
   * What kind of post downvotes your site allows.
   */
  post_downvotes: FederationMode,
  /**
   * What kind of post upvotes your site allows.
   */
  post_upvotes: FederationMode,
  registration_mode: RegistrationMode,
  require_email_verification: z.boolean()
});
const Site = z.object({
  actor_id: z.string(),
  banner: z.optional(z.string()),
  description: z.optional(z.string()),
  icon: z.optional(z.string()),
  name: z.string(),
  sidebar: z.optional(z.string()),
  version: z.optional(z.string())
});
const SiteView = z.object({
  counts: z.optional(SiteAggregates),
  local_site: LocalSite,
  site: Site
});
const GetSiteResponse = z.object({
  admins: z.array(PersonView),
  my_user: z.optional(MyUserInfo),
  site_view: SiteView,
  version: z.string()
});

const GetUnreadCountResponse = z.object({
  mentions: z.number(),
  private_messages: z.number(),
  replies: z.number()
});

const LoginResponse = z.object({
  /**
   * This is None in response to `Register` if email verification is enabled, or the server
   * requires registration applications.
   */
  jwt: z.optional(z.string()),
  /**
   * If registration applications are required, this will return true for a signup response.
   *
   * Omitted for piefed
   */
  registration_created: z.optional(z.boolean()),
  /**
   * If email verifications are required, this will return true for a signup response.
   *
   * Omitted for piefed
   */
  verify_email_sent: z.optional(z.boolean())
});

const PersonMention = z.object({
  comment_id: z.number(),
  id: z.number(),
  published: z.string(),
  read: z.boolean(),
  recipient_id: z.number()
});

const PersonMentionView = z.object({
  banned_from_community: z.boolean(),
  comment: Comment,
  community: Community,
  counts: CommentAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_blocked: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  my_vote: z.optional(Vote),
  person_mention: PersonMention,
  post: Post,
  recipient: Person,
  saved: z.boolean(),
  subscribed: SubscribedType
});

const PrivateMessage = z.object({
  ap_id: z.string(),
  content: z.string(),
  creator_id: z.number(),
  deleted: z.boolean(),
  id: z.number(),
  local: z.boolean(),
  published: z.string(),
  read: z.boolean(),
  recipient_id: z.number(),
  updated: z.optional(z.string())
});

const PrivateMessageView = z.object({
  creator: Person,
  private_message: PrivateMessage,
  recipient: Person
});

const Notification = z.union([
  CommentReplyView,
  PersonMentionView,
  PrivateMessageView
]);

const PageCursor = z.union([z.string(), z.number()]);
const PagableResponse = z.object({
  next_page: z.optional(PageCursor),
  prev_page: z.optional(PageCursor)
});
function buildPagableResponse(schema) {
  return z.extend(PagableResponse, {
    data: z.array(schema)
  });
}

const PostAggregates = z.object({
  comments: z.number(),
  downvotes: z.number(),
  /**
   * The time of the newest comment in the post.
   */
  newest_comment_time: z.string(),
  published: z.string(),
  score: z.number(),
  upvotes: z.number()
});
const PostView = z.object({
  banned_from_community: z.boolean(),
  community: Community,
  counts: PostAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_blocked: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  hidden: z.boolean(),
  my_vote: z.optional(Vote),
  post: Post,
  read: z.boolean(),
  saved: z.boolean(),
  subscribed: SubscribedType,
  unread_comments: z.number()
});

const PersonContentItem = z.union([PostView, CommentView]);

const PostReport = z.object({
  creator_id: z.number(),
  id: z.number(),
  /**
   * The original post body.
   */
  original_post_body: z.optional(z.string()),
  /**
   * The original post title.
   */
  original_post_name: z.string(),
  /**
   * The original post url.
   */
  original_post_url: z.optional(z.string()),
  post_id: z.number(),
  published: z.string(),
  reason: z.string(),
  resolved: z.boolean(),
  resolver_id: z.optional(z.number()),
  updated: z.optional(z.string())
});

const PostReportView = z.object({
  community: Community,
  counts: PostAggregates,
  creator: Person,
  creator_banned_from_community: z.boolean(),
  creator_blocked: z.boolean(),
  creator_is_admin: z.boolean(),
  creator_is_moderator: z.boolean(),
  hidden: z.boolean(),
  my_vote: z.optional(Vote),
  post: Post,
  post_creator: Person,
  post_report: PostReport,
  read: z.boolean(),
  resolver: z.optional(Person),
  saved: z.boolean(),
  subscribed: SubscribedType,
  unread_comments: z.number()
});

const ReportItemView = z.union([CommentReportView, PostReportView]);

const ResolveObjectResponse = z.object({
  comment: z.optional(CommentView),
  community: z.optional(CommunityView),
  person: z.optional(PersonView),
  post: z.optional(PostView)
});

const SearchItem = z.union([
  CommentView,
  CommunityView,
  PersonView,
  PostView
]);

const UploadImageResponse = z.object({
  delete_token: z.optional(z.string()),
  url: z.string()
});

const ListPostsResponse = buildPagableResponse(PostView);
const ListCommentsResponse = buildPagableResponse(CommentView);
const ListModlogResponse = buildPagableResponse(ModlogItem);
const ListNotificationsResponse = buildPagableResponse(
  Notification
);
const ListPersonMentionsResponse = buildPagableResponse(
  PersonMentionView
);
const ListPrivateMessagesResponse = buildPagableResponse(
  PrivateMessageView
);
const ListRepliesResponse = buildPagableResponse(
  CommentReplyView
);
const ListCommentReportsResponse = buildPagableResponse(
  CommentReportView
);
const ListCommunitiesResponse = buildPagableResponse(
  CommunityView
);
const ListPersonContentResponse = buildPagableResponse(
  PersonContentItem
);
const ListPostReportsResponse = buildPagableResponse(
  PostReportView
);
const ListReportsResponse = buildPagableResponse(ReportItemView);
const ListSearchResponse = buildPagableResponse(SearchItem);

function buildSafeClient(_Client) {
  const Client = _Client;
  return class SafeClient extends Client {
    async banFromCommunity(...params) {
      return super.banFromCommunity(...params);
    }
    async blockCommunity(...params) {
      const { community_view } = await super.blockCommunity(...params);
      return { community_view: CommunityView.parse(community_view) };
    }
    async blockInstance(...params) {
      return super.blockInstance(...params);
    }
    async blockPerson(...params) {
      const { person_view } = await super.blockPerson(...params);
      return { person_view: PersonView.parse(person_view) };
    }
    async createComment(...params) {
      const { comment_view } = await super.createComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async createCommentReport(...params) {
      return super.createCommentReport(...params);
    }
    async createPost(...params) {
      const { post_view } = await super.createPost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async createPostReport(...params) {
      return super.createPostReport(...params);
    }
    async createPrivateMessage(...params) {
      const { private_message_view } = await super.createPrivateMessage(
        ...params
      );
      return {
        private_message_view: PrivateMessageView.parse(private_message_view)
      };
    }
    async createPrivateMessageReport(...params) {
      return super.createPrivateMessageReport(...params);
    }
    async deleteComment(...params) {
      const { comment_view } = await super.deleteComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async deleteImage(...params) {
      return super.deleteImage(...params);
    }
    async deletePost(...params) {
      const { post_view } = await super.deletePost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async distinguishComment(...params) {
      const { comment_view } = await super.distinguishComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async editComment(...params) {
      const { comment_view } = await super.editComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async editPost(...params) {
      const { post_view } = await super.editPost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async featurePost(...params) {
      const { post_view } = await super.featurePost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async followCommunity(...params) {
      const { community_view } = await super.followCommunity(...params);
      return { community_view: CommunityView.parse(community_view) };
    }
    async getCaptcha(...params) {
      const response = await super.getCaptcha(...params);
      return GetCaptchaResponse.parse(response);
    }
    async getComments(...params) {
      const response = await super.getComments(...params);
      return ListCommentsResponse.parse(response);
    }
    async getCommunity(...params) {
      const response = await super.getCommunity(...params);
      return GetCommunityResponse.parse(response);
    }
    async getFederatedInstances(...params) {
      const { federated_instances } = await super.getFederatedInstances(
        ...params
      );
      return {
        federated_instances: federated_instances ? FederatedInstances.parse(federated_instances) : void 0
      };
    }
    async getModlog(...params) {
      const response = await super.getModlog(...params);
      return ListModlogResponse.parse(response);
    }
    async getNotifications(...params) {
      const response = await super.getNotifications(...params);
      return ListNotificationsResponse.parse(response);
    }
    async getPersonDetails(...params) {
      const response = await super.getPersonDetails(...params);
      return GetPersonDetailsResponse.parse(response);
    }
    async getPersonMentions(...params) {
      const response = await super.getPersonMentions(...params);
      return ListPersonMentionsResponse.parse(response);
    }
    async getPost(...params) {
      const { post_view } = await super.getPost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async getPosts(...params) {
      const response = await super.getPosts(...params);
      return ListPostsResponse.parse(response);
    }
    async getPrivateMessages(...params) {
      const response = await super.getPrivateMessages(...params);
      return ListPrivateMessagesResponse.parse(response);
    }
    async getRandomCommunity(...params) {
      const { community_view } = await super.getRandomCommunity(...params);
      return { community_view: CommunityView.parse(community_view) };
    }
    async getReplies(...params) {
      const response = await super.getReplies(...params);
      return ListRepliesResponse.parse(response);
    }
    async getSite(...params) {
      const response = await super.getSite(...params);
      return GetSiteResponse.parse(response);
    }
    async getSiteMetadata(...params) {
      const response = await super.getSiteMetadata(...params);
      return GetSiteMetadataResponse.parse(response);
    }
    async getUnreadCount(...params) {
      const response = await super.getUnreadCount(...params);
      return GetUnreadCountResponse.parse(response);
    }
    async likeComment(...params) {
      const { comment_view } = await super.likeComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async likePost(...params) {
      const { post_view } = await super.likePost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async listCommentReports(...params) {
      const response = await super.listCommentReports(...params);
      return ListCommentReportsResponse.parse(response);
    }
    async listCommunities(...params) {
      const response = await super.listCommunities(...params);
      return ListCommunitiesResponse.parse(response);
    }
    async listPersonContent(...params) {
      const response = await super.listPersonContent(...params);
      return ListPersonContentResponse.parse(response);
    }
    async listPersonLiked(...params) {
      const response = await super.listPersonLiked(...params);
      return ListPersonContentResponse.parse(response);
    }
    async listPersonSaved(...params) {
      const response = await super.listPersonSaved(...params);
      return ListPersonContentResponse.parse(response);
    }
    async listPostReports(...params) {
      const response = await super.listPostReports(...params);
      return ListPostReportsResponse.parse(response);
    }
    async listReports(...params) {
      const response = await super.listReports(...params);
      return ListReportsResponse.parse(response);
    }
    async lockPost(...params) {
      const { post_view } = await super.lockPost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async login(...params) {
      const response = await super.login(...params);
      return LoginResponse.parse(response);
    }
    async logout(...params) {
      return super.logout(...params);
    }
    async markAllAsRead(...params) {
      return super.markAllAsRead(...params);
    }
    async markCommentReplyAsRead(...params) {
      return super.markCommentReplyAsRead(...params);
    }
    async markPersonMentionAsRead(...params) {
      return super.markPersonMentionAsRead(...params);
    }
    async markPostAsRead(...params) {
      return super.markPostAsRead(...params);
    }
    async markPrivateMessageAsRead(...params) {
      return super.markPrivateMessageAsRead(...params);
    }
    async register(...params) {
      const response = await super.register(...params);
      return LoginResponse.parse(response);
    }
    async removeComment(...params) {
      const { comment_view } = await super.removeComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async removePost(...params) {
      const { post_view } = await super.removePost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async resolveCommentReport(...params) {
      return super.resolveCommentReport(...params);
    }
    async resolveObject(...params) {
      const response = await super.resolveObject(...params);
      return ResolveObjectResponse.parse(response);
    }
    async resolvePostReport(...params) {
      return super.resolvePostReport(...params);
    }
    async saveComment(...params) {
      const { comment_view } = await super.saveComment(...params);
      return { comment_view: CommentView.parse(comment_view) };
    }
    async savePost(...params) {
      const { post_view } = await super.savePost(...params);
      return { post_view: PostView.parse(post_view) };
    }
    async saveUserSettings(...params) {
      return super.saveUserSettings(...params);
    }
    async search(...params) {
      const response = await super.search(...params);
      return ListSearchResponse.parse(response);
    }
    async uploadImage(...params) {
      const response = await super.uploadImage(...params);
      return UploadImageResponse.parse(response);
    }
  };
}

function fromPageParams$2(params) {
  const result = { ...params };
  const page_cursor = result.page_cursor;
  delete result.page_cursor;
  if (typeof page_cursor === "string")
    throw new InvalidPayloadError(
      "lemmyv0 does not support string page_cursor"
    );
  return {
    ...result,
    limit: params.limit,
    page: page_cursor ? Number(page_cursor) : void 0
  };
}
function toBlocks(blocks) {
  return {
    community_blocks: blocks.community_blocks.map((t) => "community" in t ? t.community : t).map(toCommunity$2),
    instance_blocks: blocks.instance_blocks.map(
      (t) => "instance" in t ? t.instance : t
    ),
    person_blocks: blocks.person_blocks.map(
      (t) => "target" in t ? t.target : t
    )
  };
}
function toCommentReportView$1(commentReport) {
  return {
    ...commentReport,
    community: toCommunity$2(commentReport.community)
  };
}
function toCommentView$2(comment) {
  return {
    ...comment,
    banned_from_community: comment.banned_from_community ?? false,
    // v0.13.3
    community: toCommunity$2(comment.community)
  };
}
function toCommunity$2(community) {
  return {
    ...community,
    hidden: community.hidden ?? false,
    // v0.13.3
    visibility: compatCommunityVisibility(community.visibility)
  };
}
function toCommunityFollowerView(communityFollower) {
  return {
    ...communityFollower,
    community: toCommunity$2(communityFollower.community)
  };
}
function toCommunityModeratorView$2(communityModerator) {
  return {
    ...communityModerator,
    community: toCommunity$2(communityModerator.community)
  };
}
function toCommunityView$2(communityView) {
  return {
    ...communityView,
    community: toCommunity$2(communityView.community)
  };
}
function toLocalSite$1(localSite) {
  const downvotes_disabled = localSite.enable_downvotes === false;
  const downvotesMode = downvotes_disabled ? "Disable" : "All";
  return {
    ...localSite,
    comment_downvotes: downvotesMode,
    comment_upvotes: "All",
    post_downvotes: downvotesMode,
    post_upvotes: "All"
  };
}
function toMentionView(personMention) {
  return {
    ...personMention,
    community: toCommunity$2(personMention.community)
  };
}
function toModlogView$1(modlog) {
  if ("community" in modlog) {
    return {
      ...modlog,
      community: toCommunity$2(modlog.community)
    };
  }
  return modlog;
}
function toPageResponse$1(params) {
  const page_cursor = params.page_cursor;
  if (typeof page_cursor === "string")
    throw new InvalidPayloadError(
      "lemmyv0 does not support string page_cursor"
    );
  return {
    next_page: (page_cursor ?? 1) + 1
  };
}
function toPostReportView$1(postReport) {
  return {
    ...postReport,
    community: toCommunity$2(postReport.community)
  };
}
function toPostView$2(post) {
  return {
    ...post,
    banned_from_community: post.banned_from_community ?? false,
    // v0.13.3
    community: toCommunity$2(post.community),
    hidden: post.hidden ?? false
    // v0.13.3
  };
}
function toReplyView(personMention) {
  return {
    ...personMention,
    community: toCommunity$2(personMention.community)
  };
}
function compatCommunityVisibility(visibility) {
  return visibility === "LocalOnly" ? "LocalOnlyPublic" : visibility ?? "Public";
}

function getInboxItemPublished(item) {
  if ("comment_reply" in item) {
    return item.comment_reply.published;
  }
  if ("private_message" in item) {
    return item.private_message.published;
  }
  return item.person_mention.published;
}
function getLogDate(item) {
  switch (true) {
    case "mod_remove_comment" in item:
      return item.mod_remove_comment.when_;
    case "mod_remove_post" in item:
      return item.mod_remove_post.when_;
    case "mod_lock_post" in item:
      return item.mod_lock_post.when_;
    case "mod_feature_post" in item:
      return item.mod_feature_post.when_;
    case "mod_remove_community" in item:
      return item.mod_remove_community.when_;
    case "mod_ban_from_community" in item:
      return item.mod_ban_from_community.when_;
    case "mod_ban" in item:
      return item.mod_ban.when_;
    case "mod_add_community" in item:
      return item.mod_add_community.when_;
    case "mod_transfer_community" in item:
      return item.mod_transfer_community.when_;
    case "mod_add" in item:
      return item.mod_add.when_;
    case "admin_purge_person" in item:
      return item.admin_purge_person.when_;
    case "admin_purge_community" in item:
      return item.admin_purge_community.when_;
    case "admin_purge_post" in item:
      return item.admin_purge_post.when_;
    case "admin_purge_comment" in item:
      return item.admin_purge_comment.when_;
    case "mod_hide_community" in item:
      return item.mod_hide_community.when_;
    default:
      return item;
  }
}
function getPostCommentItemCreatedDate(item) {
  if ("comment" in item) return Date.parse(item.comment.published);
  return Date.parse(item.post.published);
}
const getPublishedDate = (item) => {
  if ("comment" in item) {
    return item.comment.published;
  } else {
    return item.post.published;
  }
};
function sortPostCommentByPublished(a, b) {
  return getPublishedDate(b).localeCompare(getPublishedDate(a));
}

class UnsafeLemmyV0Client {
  static mode = "lemmyv0";
  static softwareName = "lemmy";
  static softwareVersionRange = ">=0.19.0";
  #client;
  constructor(hostname, options) {
    this.#client = new LemmyV0.LemmyHttp(hostname, options);
  }
  async banFromCommunity(...params) {
    await this.#client.banFromCommunity(...params);
  }
  async blockCommunity(...params) {
    const response = await this.#client.blockCommunity(...params);
    return {
      ...response,
      community_view: toCommunityView$2(response.community_view)
    };
  }
  async blockInstance(...params) {
    await this.#client.blockInstance(...params);
  }
  async blockPerson(...params) {
    return this.#client.blockPerson(...params);
  }
  async createComment(...params) {
    const response = await this.#client.createComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async createCommentReport(...params) {
    await this.#client.createCommentReport(...params);
  }
  async createPost(...params) {
    const response = await this.#client.createPost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async createPostReport(...params) {
    await this.#client.createPostReport(...params);
  }
  async createPrivateMessage(...params) {
    return this.#client.createPrivateMessage(...params);
  }
  async createPrivateMessageReport(...params) {
    await this.#client.createPrivateMessageReport(...params);
  }
  async deleteComment(...params) {
    const response = await this.#client.deleteComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async deleteImage(payload, options) {
    await this.#client.deleteImage(
      {
        filename: new URL(payload.url).pathname.split("/").pop(),
        token: payload.delete_token
      },
      options
    );
  }
  async deletePost(...params) {
    const response = await this.#client.deletePost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async distinguishComment(...params) {
    const response = await this.#client.distinguishComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async editComment(...params) {
    const response = await this.#client.editComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async editPost(...params) {
    const response = await this.#client.editPost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async featurePost(...params) {
    const response = await this.#client.featurePost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async followCommunity(...params) {
    const response = await this.#client.followCommunity(...params);
    return {
      ...response,
      community_view: toCommunityView$2(response.community_view)
    };
  }
  async getCaptcha(...params) {
    return this.#client.getCaptcha(...params);
  }
  async getComments(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv0")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const response = await this.#client.getComments(
      cleanThreadiverseParams(fromPageParams$2(payload)),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.comments.map(toCommentView$2)
    };
  }
  async getCommunity(...params) {
    const response = await this.#client.getCommunity(...params);
    return {
      ...response,
      community_view: {
        ...toCommunityView$2(response.community_view)
      },
      moderators: response.moderators.map(toCommunityModeratorView$2)
    };
  }
  async getFederatedInstances(...params) {
    return this.#client.getFederatedInstances(...params);
  }
  async getModlog(payload, options) {
    const response = await this.#client.getModlog(
      fromPageParams$2(payload),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: Object.values(response).flat().map(toModlogView$1).sort((a, b) => Date.parse(getLogDate(b)) - Date.parse(getLogDate(a)))
    };
  }
  async getNotifications(payload, options) {
    const params = fromPageParams$2(payload);
    const [replies, mentions, privateMessages] = await Promise.all([
      this.getReplies(params, options),
      this.getPersonMentions(params, options),
      this.getPrivateMessages(params, options)
    ]);
    const data = [
      ...replies.data,
      ...mentions.data,
      ...privateMessages.data
    ].sort(
      (a, b) => Date.parse(getInboxItemPublished(b)) - Date.parse(getInboxItemPublished(a))
    );
    return {
      ...toPageResponse$1(payload),
      data
    };
  }
  async getPersonDetails(payload, options) {
    const response = await this.#client.getPersonDetails(
      {
        ...payload,
        limit: 1
        // Lemmy melts down if limit is 0
      },
      options
    );
    return {
      ...response,
      moderates: response.moderates.map(toCommunityModeratorView$2)
    };
  }
  async getPersonMentions(payload, options) {
    const response = await this.#client.getPersonMentions(
      { ...payload, sort: "New" },
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.mentions.map(toMentionView)
    };
  }
  async getPost(...params) {
    const response = await this.#client.getPost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async getPosts(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv0")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const page_cursor = payload.page_cursor;
    if (typeof page_cursor === "number")
      throw new InvalidPayloadError("page_cursor must be string");
    const response = await this.#client.getPosts(
      {
        // Only endpoint in lemmy v0 that supports page_cursor
        // Do not call fromPageParams here!
        ...cleanThreadiverseParams(payload),
        page_cursor
      },
      options
    );
    return {
      data: response.posts.map(toPostView$2),
      next_page: response.next_page
    };
  }
  async getPostSortType() {
    return [{ sort: "Top" }, { sort: "All" }];
  }
  async getPrivateMessages(payload, options) {
    const response = await this.#client.getPrivateMessages(
      fromPageParams$2(payload),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.private_messages
    };
  }
  async getRandomCommunity(..._params) {
    throw new UnsupportedError(
      "Get random community is not supported by Lemmy v0"
    );
  }
  async getReplies(payload, options) {
    const response = await this.#client.getReplies(
      fromPageParams$2({ ...payload, sort: "New" }),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.replies.map(toReplyView)
    };
  }
  async getSite(...params) {
    const site = await this.#client.getSite(...params);
    return {
      ...site,
      my_user: site.my_user ? {
        ...site.my_user,
        follows: site.my_user.follows.map(toCommunityFollowerView),
        moderates: site.my_user.moderates.map(
          toCommunityModeratorView$2
        ),
        ...toBlocks(site.my_user)
      } : void 0,
      site_view: {
        ...site.site_view,
        local_site: toLocalSite$1(site.site_view.local_site)
      }
    };
  }
  async getSiteMetadata(...params) {
    return this.#client.getSiteMetadata(...params);
  }
  async getUnreadCount(...params) {
    return this.#client.getUnreadCount(...params);
  }
  async likeComment(...params) {
    const response = await this.#client.likeComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async likePost(...params) {
    const response = await this.#client.likePost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async listCommentReports(payload, options) {
    const response = await this.#client.listCommentReports(
      fromPageParams$2(payload),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.comment_reports.map(toCommentReportView$1)
    };
  }
  async listCommunities(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv0")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const response = await this.#client.listCommunities(
      cleanThreadiverseParams(fromPageParams$2(payload)),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.communities.map(toCommunityView$2)
    };
  }
  async listPersonContent(payload, options) {
    const response = await this.#client.getPersonDetails(
      {
        ...fromPageParams$2(payload),
        sort: "New"
      },
      options
    );
    const data = (() => {
      switch (payload.type) {
        case "All":
        case void 0:
          return [
            ...response.posts.map(toPostView$2),
            ...response.comments.map(toCommentView$2)
          ].sort(
            (a, b) => getPostCommentItemCreatedDate(b) - getPostCommentItemCreatedDate(a)
          );
        case "Comments":
          return response.comments.map(toCommentView$2);
        case "Posts":
          return response.posts.map(toPostView$2);
      }
    })();
    return {
      ...toPageResponse$1(payload),
      data
    };
  }
  async listPersonLiked({ type, ...payload }, options) {
    const v0Payload = {
      ...fromPageParams$2(payload),
      disliked_only: type === "Downvoted",
      liked_only: type === "Upvoted",
      show_read: true
    };
    const [{ posts }, { comments }] = await Promise.all([
      this.#client.getPosts(v0Payload, options),
      this.#client.getComments(v0Payload, options)
    ]);
    return {
      data: [
        ...comments.map(toCommentView$2),
        ...posts.map(toPostView$2)
      ].sort(sortPostCommentByPublished),
      ...toPageResponse$1(payload)
    };
  }
  async listPersonSaved(payload, options) {
    return this.listPersonContent(
      {
        ...payload,
        // @ts-expect-error Dogfood the api
        saved_only: true
      },
      options
    );
  }
  async listPostReports(payload, options) {
    const response = await this.#client.listPostReports(
      fromPageParams$2(payload),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: response.post_reports.map(toPostReportView$1)
    };
  }
  async listReports(payload, options) {
    const params = fromPageParams$2(payload);
    const [{ comment_reports }, { post_reports }] = await Promise.all([
      this.#client.listCommentReports(params, options),
      this.#client.listPostReports(params, options)
    ]);
    return {
      ...toPageResponse$1(payload),
      data: [
        ...comment_reports.map(toCommentReportView$1),
        ...post_reports.map(toPostReportView$1)
      ].sort(
        (a, b) => getPostCommentItemCreatedDate(b) - getPostCommentItemCreatedDate(a)
      )
    };
  }
  async lockPost(...params) {
    const response = await this.#client.lockPost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async login(...params) {
    return this.#client.login(...params);
  }
  async logout(...params) {
    await this.#client.logout(...params);
  }
  async markAllAsRead(...params) {
    await this.#client.markAllAsRead(...params);
  }
  async markCommentReplyAsRead(...params) {
    await this.#client.markCommentReplyAsRead(...params);
  }
  async markPersonMentionAsRead(...params) {
    await this.#client.markPersonMentionAsRead(...params);
  }
  async markPostAsRead(...params) {
    await this.#client.markPostAsRead(...params);
  }
  async markPrivateMessageAsRead(...params) {
    await this.#client.markPrivateMessageAsRead(...params);
  }
  async register(...params) {
    return this.#client.register(...params);
  }
  async removeComment(...params) {
    const response = await this.#client.removeComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async removePost(...params) {
    const response = await this.#client.removePost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async resolveCommentReport(...params) {
    await this.#client.resolveCommentReport(...params);
  }
  async resolveObject(payload, options) {
    const response = await this.#client.resolveObject(payload, options);
    return {
      ...response,
      comment: response.comment ? toCommentView$2(response.comment) : void 0,
      community: response.community ? toCommunityView$2(response.community) : void 0,
      post: response.post ? toPostView$2(response.post) : void 0
    };
  }
  async resolvePostReport(...params) {
    await this.#client.resolvePostReport(...params);
  }
  async saveComment(...params) {
    const response = await this.#client.saveComment(...params);
    return {
      comment_view: toCommentView$2(response.comment_view)
    };
  }
  async savePost(...params) {
    const response = await this.#client.savePost(...params);
    return {
      post_view: toPostView$2(response.post_view)
    };
  }
  async saveUserSettings(...params) {
    await this.#client.saveUserSettings(...params);
  }
  async search(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv0")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const response = await this.#client.search(
      cleanThreadiverseParams(fromPageParams$2(payload)),
      options
    );
    return {
      ...toPageResponse$1(payload),
      data: [
        ...response.comments.map(toCommentView$2),
        ...response.posts.map(toPostView$2),
        ...response.communities.map(toCommunityView$2),
        ...response.users
      ]
    };
  }
  async uploadImage(payload, options) {
    const response = await this.#client.uploadImage(
      { image: payload.file },
      options
    );
    const fileResponse = response.files?.[0];
    if (!fileResponse || !response.url) {
      throw new UnexpectedResponseError("Failed to upload image");
    }
    return {
      delete_token: fileResponse.delete_token,
      url: response.url
    };
  }
}
const LemmyV0Client = buildSafeClient(UnsafeLemmyV0Client);

function fromPageParams$1(params) {
  if (typeof params.page_cursor === "number")
    throw new InvalidPayloadError("page_cursor must be string in lemmyv1");
  return {
    ...params,
    page_cursor: params.page_cursor
  };
}
function toCommentMentionView(personMentionView) {
  return {
    ...personMentionView,
    banned_from_community: !!personMentionView.community_actions?.ban_expires_at,
    comment: toComment$1(personMentionView.comment),
    community: toCommunity$1(personMentionView.community),
    counts: toCommentCounts(personMentionView.comment),
    creator: toPerson$1(personMentionView.creator),
    creator_banned_from_community: !!personMentionView.community_actions?.ban_expires_at,
    creator_blocked: !!personMentionView.person_actions?.blocked_at,
    person_mention: {
      ...personMentionView.person_comment_mention,
      published: personMentionView.person_comment_mention.published_at
    },
    post: toPost$1(personMentionView.post),
    recipient: toPerson$1(personMentionView.recipient),
    saved: !!personMentionView.comment_actions?.saved_at,
    subscribed: toFollowState(
      personMentionView.community_actions?.follow_state
    )
  };
}
function toCommentReplyView$1(commentReply) {
  return {
    ...commentReply,
    banned_from_community: !!commentReply.community_actions?.ban_expires_at,
    comment: toComment$1(commentReply.comment),
    comment_reply: {
      ...commentReply.comment_reply,
      published: commentReply.comment_reply.published_at
    },
    community: toCommunity$1(commentReply.community),
    counts: toCommentCounts(commentReply.comment),
    creator: toPerson$1(commentReply.creator),
    creator_blocked: !!commentReply.person_actions?.blocked_at,
    post: toPost$1(commentReply.post),
    recipient: toPerson$1(commentReply.recipient),
    saved: !!commentReply.comment_actions?.saved_at,
    subscribed: toFollowState(commentReply.community_actions?.follow_state)
  };
}
function toCommentReportView(commentReport) {
  return {
    ...commentReport,
    comment: toComment$1(commentReport.comment),
    comment_creator: toPerson$1(commentReport.comment_creator),
    comment_report: {
      ...commentReport.comment_report,
      published: commentReport.comment_report.published_at
    },
    community: toCommunity$1(commentReport.community),
    counts: toCommentCounts(commentReport.comment),
    creator: toPerson$1(commentReport.creator),
    creator_banned_from_community: !!commentReport.community_actions?.ban_expires_at,
    creator_blocked: !!commentReport.person_actions?.blocked_at,
    creator_is_moderator: false,
    my_vote: commentReport.comment_actions?.like_score,
    post: toPost$1(commentReport.post),
    resolver: commentReport.resolver ? toPerson$1(commentReport.resolver) : void 0,
    saved: !!commentReport.comment_actions?.saved_at,
    subscribed: toFollowState(commentReport.community_actions?.follow_state)
  };
}
function toCommentView$1(commentView) {
  return {
    ...commentView,
    banned_from_community: !!commentView.community_actions?.ban_expires_at,
    comment: toComment$1(commentView.comment),
    community: toCommunity$1(commentView.community),
    counts: toCommentCounts(commentView.comment),
    creator: toPerson$1(commentView.creator),
    post: toPost$1(commentView.post),
    subscribed: toFollowState(commentView.community_actions?.follow_state),
    ...toCommentViewActions(commentView.comment_actions)
  };
}
function toCommunityCounts(community) {
  return {
    comments: community.comments,
    posts: community.posts,
    subscribers: community.subscribers,
    subscribers_local: community.subscribers_local,
    users_active_day: community.users_active_day,
    users_active_half_year: community.users_active_half_year,
    users_active_month: community.users_active_month,
    users_active_week: community.users_active_week
  };
}
function toCommunityModeratorView$1(communityModerator) {
  return {
    ...communityModerator,
    community: toCommunity$1(communityModerator.community),
    moderator: toPerson$1(communityModerator.moderator)
  };
}
function toCommunityView$1(communityView) {
  return {
    ...communityView,
    community: toCommunity$1(communityView.community),
    counts: toCommunityCounts(communityView.community),
    ...toViewUserActions(communityView.community_actions)
  };
}
function toFederatedInstances(federatedInstances) {
  return {
    ...federatedInstances,
    allowed: federatedInstances.allowed.map(toInstanceWithFederationState),
    blocked: federatedInstances.blocked.map(toInstanceWithFederationState),
    linked: federatedInstances.linked.map(toInstanceWithFederationState)
  };
}
function toInboxCombinedView(inboxItem) {
  switch (inboxItem.type_) {
    case "CommentMention":
      return toCommentMentionView(inboxItem);
    case "CommentReply":
      return toCommentReplyView$1(inboxItem);
    case "PrivateMessage":
      return toPrivateMessageView$1(inboxItem);
    default:
      return;
  }
}
function toModlogView(modlog) {
  switch (modlog.type_) {
    case "AdminAllowInstance":
    case "AdminBlockInstance":
    case "ModChangeCommunityVisibility":
      return void 0;
    // Currently Threadiverse doesn't support these types
    case "AdminPurgeComment":
      return toAdminPurgeCommentView(modlog);
    case "AdminPurgeCommunity":
      return toAdminPurgeCommunityView(modlog);
    case "AdminPurgePerson":
      return toAdminPurgePersonView(modlog);
    case "AdminPurgePost":
      return toAdminPurgePostView(modlog);
    case "ModAdd":
      return toModAddView(modlog);
    case "ModAddCommunity":
      return toModAddCommunityView(modlog);
    case "ModBan":
      return toModBanView(modlog);
    case "ModBanFromCommunity":
      return toModBanFromCommunityView(modlog);
    case "ModFeaturePost":
      return toModFeaturePostView(modlog);
    case "ModLockPost":
      return toModLockPostView(modlog);
    case "ModRemoveComment":
      return toModRemoveCommentView(modlog);
    case "ModRemoveCommunity":
      return toModRemoveCommunityView(modlog);
    case "ModRemovePost":
      return toModRemovePostView(modlog);
    case "ModTransferCommunity":
      return toModTransferCommunityView(modlog);
  }
}
function toPersonView$1(personView) {
  return {
    ...personView,
    counts: toPersonCounts(personView.person),
    person: toPerson$1(personView.person)
  };
}
function toPostReportView(postReport) {
  return {
    ...postReport,
    community: toCommunity$1(postReport.community),
    counts: toPostCounts(postReport.post),
    creator: toPerson$1(postReport.creator),
    creator_banned_from_community: !!postReport.community_actions?.ban_expires_at,
    creator_blocked: !!postReport.person_actions?.blocked_at,
    creator_is_moderator: false,
    hidden: !!postReport.post_actions?.hidden_at,
    post: toPost$1(postReport.post),
    post_creator: toPerson$1(postReport.post_creator),
    post_report: {
      ...postReport.post_report,
      published: postReport.post_report.published_at
    },
    read: !!postReport.post_actions?.read_at,
    resolver: postReport.resolver ? toPerson$1(postReport.resolver) : void 0,
    saved: !!postReport.post_actions?.saved_at,
    subscribed: toFollowState(postReport.community_actions?.follow_state),
    unread_comments: postReport.post_actions?.read_comments_at ? postReport.post.comments - (postReport.post_actions.read_comments_amount ?? 0) : 0
  };
}
function toPostView$1(postView) {
  return {
    ...postView,
    banned_from_community: !!postView.community_actions?.ban_expires_at,
    community: toCommunity$1(postView.community),
    counts: toPostCounts(postView.post),
    creator: toPerson$1(postView.creator),
    creator_blocked: !!postView.person_actions?.blocked_at,
    post: toPost$1(postView.post),
    ...toPostViewUserActions(postView.post_actions, postView.post.comments),
    ...toViewUserActions(postView.community_actions)
  };
}
function toPrivateMessageView$1(privateMessage) {
  return {
    ...privateMessage,
    creator: toPerson$1(privateMessage.creator),
    private_message: {
      ...privateMessage.private_message,
      published: privateMessage.private_message.published_at
    },
    recipient: toPerson$1(privateMessage.recipient)
  };
}
function toReportView(report) {
  switch (report.type_) {
    case "Comment":
      return toCommentReportView(report);
    case "Post":
      return toPostReportView(report);
  }
}
function toSearchItem(item) {
  switch (item.type_) {
    case "Comment":
      return toCommentView$1(item);
    case "Community":
      return toCommunityView$1(item);
    case "MultiCommunity":
      return;
    // Not supported
    case "Person":
      return toPersonView$1(item);
    case "Post":
      return toPostView$1(item);
  }
}
function toSiteView(siteView) {
  return {
    ...siteView,
    site: toSite$1(siteView.site)
  };
}
function toAdminPurgeCommentView(adminPurgeComment) {
  return {
    ...adminPurgeComment,
    admin: adminPurgeComment.admin ? toPerson$1(adminPurgeComment.admin) : void 0,
    admin_purge_comment: {
      ...adminPurgeComment.admin_purge_comment,
      when_: adminPurgeComment.admin_purge_comment.published_at
    },
    post: toPost$1(adminPurgeComment.post)
  };
}
function toAdminPurgeCommunityView(adminPurgeCommunity) {
  return {
    ...adminPurgeCommunity,
    admin: adminPurgeCommunity.admin ? toPerson$1(adminPurgeCommunity.admin) : void 0,
    admin_purge_community: {
      ...adminPurgeCommunity.admin_purge_community,
      when_: adminPurgeCommunity.admin_purge_community.published_at
    }
  };
}
function toAdminPurgePersonView(adminPurgePerson) {
  return {
    ...adminPurgePerson,
    admin: adminPurgePerson.admin ? toPerson$1(adminPurgePerson.admin) : void 0,
    admin_purge_person: {
      ...adminPurgePerson.admin_purge_person,
      when_: adminPurgePerson.admin_purge_person.published_at
    }
  };
}
function toAdminPurgePostView(adminPurgePost) {
  return {
    ...adminPurgePost,
    admin: adminPurgePost.admin ? toPerson$1(adminPurgePost.admin) : void 0,
    admin_purge_post: {
      ...adminPurgePost.admin_purge_post,
      when_: adminPurgePost.admin_purge_post.published_at
    },
    community: toCommunity$1(adminPurgePost.community)
  };
}
function toComment$1(comment) {
  return {
    ...comment,
    published: comment.published_at
  };
}
function toCommentCounts(comment) {
  return {
    child_count: comment.child_count,
    comment_id: comment.id,
    downvotes: comment.downvotes,
    published: comment.published_at,
    score: comment.score,
    upvotes: comment.upvotes
  };
}
function toCommentViewActions(commentActions) {
  return {
    my_vote: commentActions?.like_score,
    saved: !!commentActions?.saved_at
  };
}
function toCommunity$1(community) {
  return {
    ...community,
    actor_id: community.ap_id,
    hidden: false,
    // TODO what does this mean, v0 vs v1?
    published: community.published_at
  };
}
function toFollowState(followState) {
  switch (followState) {
    case "Accepted":
      return "Subscribed";
    case "ApprovalRequired":
    case "Pending":
      return followState;
    case void 0:
      return "NotSubscribed";
  }
}
function toInstanceWithFederationState(instance) {
  return {
    ...instance,
    published: instance.published_at
  };
}
function toModAddCommunityView(modAddCommunity) {
  return {
    ...modAddCommunity,
    community: toCommunity$1(modAddCommunity.community),
    mod_add_community: {
      ...modAddCommunity.mod_add_community,
      when_: modAddCommunity.mod_add_community.published_at
    },
    modded_person: toPerson$1(modAddCommunity.other_person),
    moderator: modAddCommunity.moderator ? toPerson$1(modAddCommunity.moderator) : void 0
  };
}
function toModAddView(modAdd) {
  return {
    ...modAdd,
    mod_add: {
      ...modAdd.mod_add,
      when_: modAdd.mod_add.published_at
    },
    modded_person: toPerson$1(modAdd.other_person),
    moderator: modAdd.moderator ? toPerson$1(modAdd.moderator) : void 0
  };
}
function toModBanFromCommunityView(modBanFromCommunity) {
  return {
    ...modBanFromCommunity,
    banned_person: toPerson$1(modBanFromCommunity.other_person),
    community: toCommunity$1(modBanFromCommunity.community),
    mod_ban_from_community: {
      ...modBanFromCommunity.mod_ban_from_community,
      when_: modBanFromCommunity.mod_ban_from_community.published_at
    },
    moderator: modBanFromCommunity.moderator ? toPerson$1(modBanFromCommunity.moderator) : void 0
  };
}
function toModBanView(modBan) {
  return {
    ...modBan,
    banned_person: toPerson$1(modBan.other_person),
    mod_ban: {
      ...modBan.mod_ban,
      when_: modBan.mod_ban.published_at
    },
    moderator: modBan.moderator ? toPerson$1(modBan.moderator) : void 0
  };
}
function toModFeaturePostView(modFeaturePost) {
  return {
    ...modFeaturePost,
    community: toCommunity$1(modFeaturePost.community),
    mod_feature_post: {
      ...modFeaturePost.mod_feature_post,
      when_: modFeaturePost.mod_feature_post.published_at
    },
    moderator: modFeaturePost.moderator ? toPerson$1(modFeaturePost.moderator) : void 0,
    post: toPost$1(modFeaturePost.post)
  };
}
function toModLockPostView(modLockPost) {
  return {
    ...modLockPost,
    community: toCommunity$1(modLockPost.community),
    mod_lock_post: {
      ...modLockPost.mod_lock_post,
      when_: modLockPost.mod_lock_post.published_at
    },
    moderator: modLockPost.moderator ? toPerson$1(modLockPost.moderator) : void 0,
    post: toPost$1(modLockPost.post)
  };
}
function toModRemoveCommentView(modRemoveComment) {
  return {
    ...modRemoveComment,
    comment: toComment$1(modRemoveComment.comment),
    commenter: toPerson$1(modRemoveComment.other_person),
    community: toCommunity$1(modRemoveComment.community),
    mod_remove_comment: {
      ...modRemoveComment.mod_remove_comment,
      when_: modRemoveComment.mod_remove_comment.published_at
    },
    moderator: modRemoveComment.moderator ? toPerson$1(modRemoveComment.moderator) : void 0,
    post: toPost$1(modRemoveComment.post)
  };
}
function toModRemoveCommunityView(modRemoveCommunity) {
  return {
    ...modRemoveCommunity,
    community: toCommunity$1(modRemoveCommunity.community),
    mod_remove_community: {
      ...modRemoveCommunity.mod_remove_community,
      when_: modRemoveCommunity.mod_remove_community.published_at
    },
    moderator: modRemoveCommunity.moderator ? toPerson$1(modRemoveCommunity.moderator) : void 0
  };
}
function toModRemovePostView(modRemovePost) {
  return {
    ...modRemovePost,
    community: toCommunity$1(modRemovePost.community),
    mod_remove_post: {
      ...modRemovePost.mod_remove_post,
      when_: modRemovePost.mod_remove_post.published_at
    },
    moderator: modRemovePost.moderator ? toPerson$1(modRemovePost.moderator) : void 0,
    post: toPost$1(modRemovePost.post)
  };
}
function toModTransferCommunityView(modTransferCommunity) {
  return {
    ...modTransferCommunity,
    community: toCommunity$1(modTransferCommunity.community),
    mod_transfer_community: {
      ...modTransferCommunity.mod_transfer_community,
      when_: modTransferCommunity.mod_transfer_community.published_at
    },
    modded_person: toPerson$1(modTransferCommunity.other_person),
    moderator: modTransferCommunity.moderator ? toPerson$1(modTransferCommunity.moderator) : void 0
  };
}
function toPerson$1(person) {
  return {
    ...person,
    actor_id: person.ap_id,
    published: person.published_at
  };
}
function toPersonCounts(person) {
  return {
    comment_count: person.comment_count,
    post_count: person.post_count
  };
}
function toPost$1(post) {
  return {
    ...post,
    published: post.published_at
  };
}
function toPostCounts(post) {
  return {
    comments: post.comments,
    downvotes: post.downvotes,
    newest_comment_time: post.newest_comment_time_at,
    published: post.published_at,
    score: post.score,
    upvotes: post.upvotes
  };
}
function toPostViewUserActions(postActions, totalComments) {
  return {
    hidden: !!postActions?.hidden_at,
    read: !!postActions?.read_at,
    saved: !!postActions?.saved_at,
    unread_comments: postActions?.read_comments_at ? totalComments - (postActions.read_comments_amount ?? 0) : 0
  };
}
function toSite$1(site) {
  return {
    ...site,
    actor_id: site.ap_id
  };
}
function toViewUserActions(userActions) {
  return {
    blocked: !!userActions?.blocked_at,
    subscribed: toFollowState(userActions?.follow_state)
  };
}

function isPostCommentReport(report) {
  return report.type_ === "Post" || report.type_ === "Comment";
}

class UnsafeLemmyV1Client {
  static mode = "lemmyv1";
  static softwareName = "lemmy";
  static softwareVersionRange = ">=1.0.0-alpha.5";
  #client;
  constructor(hostname, options) {
    this.#client = new LemmyV1.LemmyHttp(hostname, options);
  }
  async banFromCommunity(...params) {
    await this.#client.banFromCommunity(...params);
  }
  async blockCommunity(...params) {
    const response = await this.#client.blockCommunity(...params);
    return {
      ...response,
      community_view: toCommunityView$1(response.community_view)
    };
  }
  async blockInstance(...params) {
    await this.#client.userBlockInstance(...params);
  }
  async blockPerson(...params) {
    const response = await this.#client.blockPerson(...params);
    return {
      ...response,
      person_view: toPersonView$1(response.person_view)
    };
  }
  async createComment(...params) {
    const response = await this.#client.createComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async createCommentReport(...params) {
    await this.#client.createCommentReport(...params);
  }
  async createPost(...params) {
    const response = await this.#client.createPost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async createPostReport(...params) {
    await this.#client.createPostReport(...params);
  }
  async createPrivateMessage(...params) {
    const response = await this.#client.createPrivateMessage(...params);
    return {
      private_message_view: toPrivateMessageView$1(
        response.private_message_view
      )
    };
  }
  async createPrivateMessageReport(...params) {
    await this.#client.createPrivateMessageReport(...params);
  }
  async deleteComment(...params) {
    const response = await this.#client.deleteComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async deleteImage(payload, options) {
    await this.#client.deleteMedia(
      { filename: payload.url },
      // delete_token is not needed. Lemmy verifies account ownership
      options
    );
  }
  async deletePost(...params) {
    const response = await this.#client.deletePost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async distinguishComment(...params) {
    const response = await this.#client.distinguishComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async editComment(...params) {
    const response = await this.#client.editComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async editPost(...params) {
    const response = await this.#client.editPost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async featurePost(...params) {
    const response = await this.#client.featurePost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async followCommunity(...params) {
    const response = await this.#client.followCommunity(...params);
    return {
      community_view: toCommunityView$1(response.community_view)
    };
  }
  async getCaptcha(...params) {
    return this.#client.getCaptcha(...params);
  }
  async getComments(payload, options) {
    const response = await this.#client.getComments(
      fromPageParams$1(payload),
      options
    );
    return {
      ...response,
      data: response.comments.map(toCommentView$1)
    };
  }
  async getCommunity(...params) {
    const response = await this.#client.getCommunity(...params);
    return {
      ...response,
      community_view: toCommunityView$1(response.community_view),
      moderators: response.moderators.map(toCommunityModeratorView$1)
    };
  }
  async getFederatedInstances(...params) {
    const response = await this.#client.getFederatedInstances(...params);
    return {
      federated_instances: response.federated_instances ? toFederatedInstances(response.federated_instances) : void 0
    };
  }
  async getModlog(payload, options) {
    const response = await this.#client.getModlog(
      fromPageParams$1(payload),
      options
    );
    return {
      ...response,
      data: response.modlog.map(toModlogView).filter((m) => !!m)
    };
  }
  async getNotifications(payload, options) {
    const response = await this.#client.listInbox(
      fromPageParams$1(payload),
      options
    );
    return {
      data: response.inbox.map(toInboxCombinedView).filter((r) => !!r),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async getPersonDetails(...params) {
    const response = await this.#client.getPersonDetails(...params);
    return {
      ...response,
      moderates: response.moderates.map(toCommunityModeratorView$1),
      person_view: toPersonView$1(response.person_view)
    };
  }
  async getPersonMentions(payload, options) {
    const response = await this.#client.listInbox(
      {
        ...fromPageParams$1(payload),
        type_: "CommentMention"
      },
      options
    );
    return {
      // Type set to CommentMention in request, so safe to cast
      data: response.inbox.map(
        toCommentMentionView
      ),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async getPost(...params) {
    const response = await this.#client.getPost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async getPosts(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv1")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const response = await this.#client.getPosts(
      cleanThreadiverseParams(fromPageParams$1(payload)),
      options
    );
    return {
      ...response,
      data: response.posts.map(toPostView$1)
    };
  }
  async getPrivateMessages(payload, options) {
    const response = await this.#client.listInbox(
      {
        ...fromPageParams$1(payload),
        type_: "PrivateMessage"
      },
      options
    );
    return {
      // Type set to PrivateMessage in request, so safe to cast
      data: response.inbox.map(
        toPrivateMessageView$1
      ),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async getRandomCommunity(..._params) {
    throw new UnsupportedError(
      "Get random community is not supported by Lemmy v0"
    );
  }
  async getReplies(payload, options) {
    const response = await this.#client.listInbox(
      {
        ...fromPageParams$1(payload),
        type_: "CommentReply"
      },
      options
    );
    return {
      data: response.inbox.map(
        toCommentReplyView$1
      ),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async getSite(...params) {
    const [siteResponse, myUserResponse] = await Promise.all([
      this.#client.getSite(...params),
      this.#client.getHeader("Authorization") && this.#client.getMyUser(...params)
    ]);
    return {
      ...siteResponse,
      ...myUserResponse,
      admins: siteResponse.admins.map(toPersonView$1),
      site_view: toSiteView(siteResponse.site_view)
    };
  }
  async getSiteMetadata(...params) {
    return this.#client.getSiteMetadata(...params);
  }
  async getUnreadCount(...params) {
    const response = await this.#client.getUnreadCount(...params);
    return {
      mentions: 0,
      private_messages: 0,
      replies: response.count
    };
  }
  async likeComment(...params) {
    const response = await this.#client.likeComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async likePost(...params) {
    const response = await this.#client.likePost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async listCommentReports(payload, options) {
    const response = await this.#client.listReports(
      {
        ...fromPageParams$1(payload),
        type_: "Comments"
      },
      options
    );
    return {
      // Type set to Comments in request, so safe to cast
      data: response.reports.map(
        toCommentReportView
      ),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listCommunities(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv1")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const response = await this.#client.listCommunities(
      cleanThreadiverseParams(fromPageParams$1(payload)),
      options
    );
    return {
      data: response.communities.map(toCommunityView$1),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listPersonContent(payload, options) {
    const response = await this.#client.listPersonContent(
      fromPageParams$1(payload),
      options
    );
    return {
      data: response.content.map((item) => {
        if (item.type_ === "Comment") return toCommentView$1(item);
        return toPostView$1(item);
      }),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listPersonLiked(payload, options) {
    const response = await this.#client.listPersonLiked(
      fromPageParams$1(payload),
      options
    );
    return {
      data: response.liked.map((item) => {
        switch (item.type_) {
          case "Comment":
            return toCommentView$1(item);
          case "Post":
            return toPostView$1(item);
        }
      }),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listPersonSaved(payload, options) {
    const response = await this.#client.listPersonSaved(
      fromPageParams$1(payload),
      options
    );
    return {
      data: response.saved.map((item) => {
        if (item.type_ === "Comment") return toCommentView$1(item);
        return toPostView$1(item);
      }),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listPostReports(payload, options) {
    const response = await this.#client.listReports(
      {
        ...fromPageParams$1(payload),
        type_: "Posts"
      },
      options
    );
    return {
      // Type set to Posts in request, so safe to cast
      data: response.reports.map(
        toPostReportView
      ),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async listReports(payload, options) {
    const response = await this.#client.listReports(
      fromPageParams$1(payload),
      options
    );
    return {
      data: response.reports.filter(isPostCommentReport).map(toReportView),
      next_page: response.next_page,
      prev_page: response.prev_page
    };
  }
  async lockPost(...params) {
    const response = await this.#client.lockPost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async login(...params) {
    return this.#client.login(...params);
  }
  async logout(...params) {
    await this.#client.logout(...params);
  }
  async markAllAsRead(...params) {
    await this.#client.markAllNotificationsAsRead(...params);
  }
  async markCommentReplyAsRead(...params) {
    await this.#client.markCommentReplyAsRead(...params);
  }
  async markPersonMentionAsRead(payload, options) {
    await this.#client.markCommentMentionAsRead(
      {
        ...payload,
        person_comment_mention_id: payload.person_mention_id
      },
      options
    );
  }
  async markPostAsRead(...params) {
    await this.#client.markManyPostAsRead(...params);
  }
  async markPrivateMessageAsRead(...params) {
    await this.#client.markPrivateMessageAsRead(...params);
  }
  async register(...params) {
    return this.#client.register(...params);
  }
  async removeComment(...params) {
    const response = await this.#client.removeComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async removePost(...params) {
    const response = await this.#client.removePost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async resolveCommentReport(...params) {
    await this.#client.resolveCommentReport(...params);
  }
  async resolveObject(payload, options) {
    const response = await this.#client.resolveObject(payload, options);
    const compatResponse = {};
    const result = response.results[0];
    if (!result) return compatResponse;
    switch (result.type_) {
      case "Comment":
        compatResponse.comment = toCommentView$1(result);
        break;
      case "Community":
        compatResponse.community = toCommunityView$1(result);
        break;
      case "Post":
        compatResponse.post = toPostView$1(result);
        break;
      case "Person":
        compatResponse.person = toPersonView$1(result);
    }
    return compatResponse;
  }
  async resolvePostReport(...params) {
    await this.#client.resolvePostReport(...params);
  }
  async saveComment(...params) {
    const response = await this.#client.saveComment(...params);
    return {
      comment_view: toCommentView$1(response.comment_view)
    };
  }
  async savePost(...params) {
    const response = await this.#client.savePost(...params);
    return {
      post_view: toPostView$1(response.post_view)
    };
  }
  async saveUserSettings(...params) {
    await this.#client.saveUserSettings(...params);
  }
  async search(payload, options) {
    if (payload.mode && payload.mode !== "lemmyv1")
      throw new InvalidPayloadError(
        `Connected to lemmyv1, ${payload.mode} is not supported`
      );
    const result = await this.#client.search(
      cleanThreadiverseParams(fromPageParams$1(payload)),
      options
    );
    return {
      ...result,
      data: result.results.map(toSearchItem).filter((item) => !!item)
    };
  }
  async uploadImage(payload, options) {
    const response = await this.#client.uploadImage(
      { image: payload.file },
      options
    );
    return {
      delete_token: "",
      url: response.image_url
    };
  }
}
const LemmyV1Client = buildSafeClient(UnsafeLemmyV1Client);

const toPageResponse = toPageResponse$1;
const fromPageParams = fromPageParams$2;
function toComment(comment, creator_id) {
  return {
    ...comment,
    content: comment.body,
    creator_id,
    distinguished: comment.distinguished ?? false
  };
}
function toCommentReplyView(reply) {
  return {
    ...reply,
    banned_from_community: false,
    // TODO this isn't being returned rn
    comment: toComment(reply.comment, reply.creator.id),
    community: toCommunity(reply.community),
    creator: toPerson(reply.creator),
    post: toPost(reply.post),
    recipient: toPerson(reply.recipient)
  };
}
function toCommentView(comment) {
  return {
    ...comment,
    comment: toComment(comment.comment, comment.creator.id),
    community: toCommunity(comment.community),
    creator: toPerson(comment.creator),
    post: toPost(comment.post),
    // TODO: is this correct? Piefed types are wide (string) here
    subscribed: comment.subscribed
  };
}
function toCommunity(community) {
  return {
    ...community,
    banner: community.banner ?? void 0,
    icon: community.icon ?? void 0,
    posting_restricted_to_mods: community.restricted_to_mods,
    visibility: "Public"
  };
}
function toCommunityModeratorView(view) {
  return {
    ...view,
    community: toCommunity(view.community),
    moderator: toPerson(view.moderator)
  };
}
function toCommunityView(community) {
  return {
    ...community,
    community: toCommunity(community.community),
    counts: {
      comments: community.counts.post_reply_count,
      posts: community.counts.post_count,
      subscribers: community.counts.subscriptions_count,
      users_active_day: community.counts.active_6monthly,
      users_active_half_year: community.counts.active_6monthly,
      users_active_month: community.counts.active_monthly,
      users_active_week: community.counts.active_weekly
    }
  };
}
function toGetCommunityResponse(response) {
  return {
    community_view: toCommunityView(response.community_view),
    moderators: response.moderators.map(toCommunityModeratorView)
  };
}
function toLocalSite(site) {
  return {
    captcha_enabled: false,
    comment_downvotes: "All",
    comment_upvotes: "All",
    post_downvotes: "All",
    post_upvotes: "All",
    registration_mode: site.registration_mode,
    require_email_verification: false
  };
}
function toPerson(person) {
  return {
    ...person,
    avatar: person.avatar ?? void 0,
    // TODO piefed types are wrong, this is returned as null if not set
    banner: person.banner ?? void 0,
    // TODO piefed types are wrong, this is returned as null if not set
    bio: person.about ?? void 0,
    // TODO piefed types are wrong, this is returned as null if not set
    bot_account: person.bot,
    display_name: person.title ?? void 0,
    // TODO piefed types are wrong, this is returned as null if not set
    name: person.user_name,
    published: person.published
  };
}
function toPersonMentionView(mention) {
  return {
    ...mention,
    banned_from_community: false,
    // TODO isn't being returned rn
    comment: toComment(mention.comment, mention.creator.id),
    community: toCommunity(mention.community),
    creator: toPerson(mention.creator),
    person_mention: mention.comment_reply,
    post: toPost(mention.post),
    recipient: toPerson(mention.recipient)
  };
}
function toPersonView(personView) {
  return {
    ...personView,
    person: toPerson(personView.person)
  };
}
function toPost(post) {
  return {
    ...post,
    creator_id: post.user_id,
    featured_community: post.sticky,
    featured_local: false,
    name: post.title
  };
}
function toPostView(post) {
  return {
    ...post,
    community: toCommunity(post.community),
    creator: toPerson(post.creator),
    creator_blocked: false,
    // TODO piefed does not return this
    post: toPost(post.post)
  };
}
function toPrivateMessageView(message) {
  return {
    ...message,
    creator: toPerson(message.creator),
    recipient: toPerson(message.recipient)
  };
}
function toSite(site) {
  return {
    ...site,
    icon: site.icon ?? void 0
  };
}

async function validateResponse(response) {
  if (!response.ok) {
    const data = await response.json();
    if ("error" in data && typeof data.error === "string") {
      throw new Error(data.error);
    }
  }
}
const piefedMiddleware = {
  async onResponse({ response }) {
    await validateResponse(response);
  }
};
class UnsafePiefedClient {
  static mode = "piefed";
  static softwareName = "piefed";
  // Piefed is not versioned atm
  static softwareVersionRange = "*";
  #client;
  #customFetch;
  #headers;
  #url;
  constructor(url, options) {
    this.#customFetch = options.fetchFunction ?? globalThis.fetch;
    this.#url = url;
    const headers = options.headers?.Authorization ? {
      Authorization: options.headers?.Authorization
    } : void 0;
    this.#headers = headers;
    this.#client = createClient({
      baseUrl: url,
      fetch: options.fetchFunction,
      // TODO: piefed doesn't allow CORS headers other than Authorization
      headers
    });
    this.#client.use(piefedMiddleware);
  }
  async banFromCommunity(payload, options) {
    await this.#client.POST("/api/alpha/community/moderate/ban", {
      ...options,
      // @ts-expect-error TODO: fix this
      body: payload
    });
  }
  async blockCommunity(payload, options) {
    const response = await this.#client.POST("/api/alpha/community/block", {
      ...options,
      body: { ...payload }
    });
    return {
      community_view: toCommunityView(response.data.community_view)
    };
  }
  async blockInstance(payload, options) {
    await this.#client.POST("/api/alpha/site/block", {
      ...options,
      body: { ...payload }
    });
  }
  async blockPerson(payload, options) {
    const response = await this.#client.POST("/api/alpha/user/block", {
      ...options,
      body: { ...payload }
    });
    return {
      ...response.data,
      person_view: toPersonView(response.data.person_view)
    };
  }
  async createComment(payload, options) {
    const response = await this.#client.POST("/api/alpha/comment", {
      ...options,
      body: {
        ...payload,
        body: payload.content
      }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async createCommentReport(payload, options) {
    await this.#client.POST("/api/alpha/comment/report", {
      ...options,
      body: { ...payload, report_remote: true }
    });
  }
  async createPost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post", {
      ...options,
      body: {
        ...payload,
        title: payload.name
      }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async createPostReport(payload, options) {
    await this.#client.POST("/api/alpha/post/report", {
      ...options,
      body: { ...payload }
    });
  }
  async createPrivateMessage(payload, options) {
    const response = await this.#client.POST("/api/alpha/private_message", {
      ...options,
      body: { ...payload }
    });
    return {
      private_message_view: toPrivateMessageView(
        response.data.private_message_view
      )
    };
  }
  async createPrivateMessageReport(..._params) {
    throw new UnsupportedError(
      "Create private message report is not supported by piefed"
    );
  }
  async deleteComment(payload, options) {
    const response = await this.#client.POST("/api/alpha/comment/delete", {
      ...options,
      body: { ...payload }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async deleteImage(..._params) {
    throw new UnsupportedError("Delete image is not supported by piefed");
  }
  async deletePost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post/delete", {
      ...options,
      body: { ...payload }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async distinguishComment(..._params) {
    throw new UnsupportedError(
      "Distinguish comment is not supported by piefed"
    );
  }
  async editComment(payload, options) {
    const response = await this.#client.PUT("/api/alpha/comment", {
      ...options,
      body: {
        ...payload,
        body: payload.content
        // TODO: piefed types say this is required, but it's not
      }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async editPost(payload, options) {
    const response = await this.#client.PUT("/api/alpha/post", {
      ...options,
      body: {
        ...payload,
        title: payload.name
      }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async featurePost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post/feature", {
      ...options,
      body: { ...payload }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async followCommunity(payload, options) {
    const response = await this.#client.POST("/api/alpha/community/follow", {
      ...options,
      body: { ...payload }
    });
    return {
      community_view: toCommunityView(response.data.community_view)
    };
  }
  async getCaptcha(..._params) {
    throw new UnsupportedError("Get captcha is not supported by piefed");
  }
  async getComments(payload, options) {
    if (payload.mode && payload.mode !== "piefed")
      throw new InvalidPayloadError(
        `Connected to piefed, ${payload.mode} is not supported`
      );
    const query = cleanThreadiverseParams(
      fromPageParams(payload)
    );
    const response = await this.#client.GET("/api/alpha/comment/list", {
      ...options,
      params: { query }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.comments.map(toCommentView)
    };
  }
  async getCommunity(payload, options) {
    const response = await this.#client.GET("/api/alpha/community", {
      ...options,
      params: { query: payload }
    });
    return toGetCommunityResponse(response.data);
  }
  async getFederatedInstances(..._params) {
    const response = await this.#client.GET("/api/alpha/federated_instances");
    return response.data;
  }
  async getModlog(..._params) {
    throw new UnsupportedError("Get modlog is not supported by piefed");
  }
  async getNotifications(...params) {
    const [replies, mentions, privateMessages] = await Promise.all([
      this.getReplies(...params),
      this.getPersonMentions(...params),
      this.getPrivateMessages(...params)
    ]);
    const data = [
      ...replies.data,
      ...mentions.data,
      ...privateMessages.data
    ].sort(
      (a, b) => Date.parse(getInboxItemPublished(b)) - Date.parse(getInboxItemPublished(a))
    );
    return {
      ...toPageResponse(params[0]),
      data
    };
  }
  async getPersonDetails(payload, options) {
    const response = await this.#client.GET("/api/alpha/user", {
      ...options,
      params: { query: payload }
    });
    return {
      ...response.data,
      moderates: response.data.moderates.map(toCommunityModeratorView),
      person_view: toPersonView(response.data.person_view)
    };
  }
  async getPersonMentions(payload, options) {
    const response = await this.#client.GET("/api/alpha/user/mentions", {
      ...options,
      params: { query: fromPageParams(payload) }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.replies.map(toPersonMentionView)
    };
  }
  async getPost(payload, options) {
    const query = payload;
    const response = await this.#client.GET("/api/alpha/post", {
      ...options,
      params: { query }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async getPosts(payload, options) {
    if (payload.mode && payload.mode !== "piefed")
      throw new InvalidPayloadError(
        `Connected to piefed, ${payload.mode} is not supported`
      );
    const query = cleanThreadiverseParams(
      fromPageParams(payload)
    );
    const response = await this.#client.GET("/api/alpha/post/list", {
      ...options,
      params: { query }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.posts.map(toPostView)
    };
  }
  async getPrivateMessages(payload, options) {
    const response = await this.#client.GET("/api/alpha/private_message/list", {
      ...options,
      params: { query: fromPageParams(payload) }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.private_messages.map(toPrivateMessageView)
    };
  }
  async getRandomCommunity(..._params) {
    throw new UnsupportedError(
      "Get random community is not supported by piefed"
    );
  }
  async getReplies(payload, options) {
    const response = await this.#client.GET("/api/alpha/user/replies", {
      ...options,
      params: { query: fromPageParams(payload) }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.replies.map(toCommentReplyView)
    };
  }
  async getSite(options) {
    const response = await this.#client.GET("/api/alpha/site", {
      ...options
    });
    return {
      ...response.data,
      // TODO: piefed.ca is missing admins in the response for some reason????
      admins: (response.data.admins ?? []).map(toPersonView),
      my_user: response.data.my_user ? {
        ...response.data.my_user,
        community_blocks: response.data.my_user?.community_blocks.map(
          ({ community }) => toCommunity(community)
        ),
        follows: response.data.my_user.follows.map((f) => ({
          community: toCommunity(f.community),
          follower: toPerson(f.follower)
        })),
        instance_blocks: response.data.my_user?.instance_blocks.map(
          ({ instance }) => instance
        ),
        local_user_view: {
          ...response.data.my_user.local_user_view,
          local_user: {
            admin: false,
            // TODO: piefed doesn't expose admin status in site response
            show_nsfw: response.data.my_user.local_user_view.local_user.show_nsfw
          },
          person: toPerson(
            response.data.my_user.local_user_view.person
          )
        },
        moderates: response.data.my_user.moderates.map(
          toCommunityModeratorView
        ),
        person_blocks: response.data.my_user?.person_blocks.map(
          ({ target }) => toPerson(target)
        )
      } : void 0,
      site_view: {
        local_site: toLocalSite(response.data.site),
        site: toSite(response.data.site)
      }
    };
  }
  async getSiteMetadata(..._params) {
    throw new UnsupportedError("Get site metadata is not supported by piefed");
  }
  async getUnreadCount(options) {
    const response = await this.#client.GET("/api/alpha/user/unread_count", {
      ...options
    });
    return response.data;
  }
  async likeComment(payload, options) {
    const response = await this.#client.POST("/api/alpha/comment/like", {
      ...options,
      body: {
        ...payload,
        private: false
      }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async likePost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post/like", {
      ...options,
      body: {
        ...payload
      }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async listCommentReports(..._params) {
    throw new UnsupportedError(
      "List comment reports is not supported by piefed"
    );
  }
  async listCommunities(payload, options) {
    const response = await this.#client.GET("/api/alpha/community/list", {
      ...options,
      // @ts-expect-error TODO: fix this
      params: { query: fromPageParams(payload) }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.communities.map(toCommunityView)
    };
  }
  async listPersonContent(payload, options) {
    switch (payload.type) {
      case "All":
      case void 0: {
        const response = await Promise.all([
          this.#listPersonPosts(payload, options),
          this.#listPersonComments(payload, options)
        ]).then(
          ([posts, comments]) => [...posts.data, ...comments.data].sort(
            (a, b) => getPostCommentItemCreatedDate(b) - getPostCommentItemCreatedDate(a)
          )
        );
        return {
          ...toPageResponse(payload),
          data: response
        };
      }
      case "Comments":
        return this.#listPersonComments(payload, options);
      case "Posts":
        return this.#listPersonPosts(payload, options);
    }
  }
  async listPersonLiked(..._params) {
    throw new UnsupportedError("List person liked is not supported by piefed");
  }
  async listPersonSaved(payload, options) {
    const response = await this.#client.GET("/api/alpha/user", {
      ...options,
      params: {
        query: { ...fromPageParams(payload), saved_only: true }
      }
    });
    const data = (() => {
      switch (payload.type) {
        case "All":
        case void 0:
          return [
            ...response.data.posts.map(toPostView),
            ...response.data.comments.map(toCommentView)
          ].sort(
            (a, b) => getPostCommentItemCreatedDate(b) - getPostCommentItemCreatedDate(a)
          );
        case "Comments":
          return response.data.comments.map(toCommentView);
        case "Posts":
          return response.data.posts.map(toPostView);
      }
    })();
    return {
      ...toPageResponse(payload),
      data
    };
  }
  async listPostReports(..._params) {
    throw new UnsupportedError("List post reports is not supported by piefed");
  }
  async listReports(..._params) {
    throw new UnsupportedError("List reports is not supported by piefed");
  }
  async lockPost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post/lock", {
      ...options,
      body: { ...payload }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async login(payload, options) {
    const response = await this.#client.POST("/api/alpha/user/login", {
      ...options,
      body: { password: payload.password, username: payload.username_or_email }
    });
    return response.data;
  }
  async logout(_options) {
  }
  async markAllAsRead(options) {
    await this.#client.POST("/api/alpha/user/mark_all_as_read", options);
  }
  async markCommentReplyAsRead(payload, options) {
    await this.#client.POST("/api/alpha/comment/mark_as_read", {
      ...options,
      body: payload
    });
  }
  async markPersonMentionAsRead(payload, options) {
    await this.#client.POST("/api/alpha/comment/mark_as_read", {
      ...options,
      body: {
        comment_reply_id: payload.person_mention_id,
        read: payload.read
      }
    });
  }
  async markPostAsRead(payload, options) {
    await this.#client.POST("/api/alpha/post/mark_as_read", {
      ...options,
      body: payload
    });
  }
  async markPrivateMessageAsRead(payload, options) {
    await this.#client.POST("/api/alpha/private_message/mark_as_read", {
      ...options,
      body: payload
    });
  }
  async register(..._params) {
    throw new UnsupportedError("Register is not supported by piefed");
  }
  async removeComment(payload, options) {
    const response = await this.#client.POST("/api/alpha/comment/remove", {
      ...options,
      body: { ...payload }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async removePost(payload, options) {
    const response = await this.#client.POST("/api/alpha/post/remove", {
      ...options,
      body: { ...payload }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async resolveCommentReport(..._params) {
    throw new UnsupportedError(
      "Resolve comment report is not supported by piefed"
    );
  }
  async resolveObject(payload, options) {
    const response = await this.#client.GET("/api/alpha/resolve_object", {
      ...options,
      params: { query: payload }
    });
    if (!response.data) throw new Error(response.error.error);
    return {
      ...response.data,
      comment: response.data.comment ? toCommentView(response.data.comment) : void 0,
      community: response.data.community ? toCommunityView(response.data.community) : void 0,
      person: response.data.person ? toPersonView(response.data.person) : void 0,
      post: response.data.post ? toPostView(response.data.post) : void 0
    };
  }
  async resolvePostReport(..._params) {
    throw new UnsupportedError(
      "Resolve post report is not supported by piefed"
    );
  }
  async saveComment(payload, options) {
    const response = await this.#client.PUT("/api/alpha/comment/save", {
      ...options,
      body: { ...payload }
    });
    return {
      comment_view: toCommentView(response.data.comment_view)
    };
  }
  async savePost(payload, options) {
    const response = await this.#client.PUT("/api/alpha/post/save", {
      ...options,
      body: { ...payload }
    });
    return {
      post_view: toPostView(response.data.post_view)
    };
  }
  async saveUserSettings(..._params) {
    throw new UnsupportedError("Save user settings is not supported by piefed");
  }
  async search(payload, options) {
    const response = await this.#client.GET("/api/alpha/search", {
      ...options,
      // @ts-expect-error TODO: fix this
      params: { query: fromPageParams(payload) }
    });
    return {
      ...toPageResponse(payload),
      data: [
        ...response.data.communities.map(toCommunityView),
        ...response.data.posts.map(toPostView),
        ...response.data.users.map(toPersonView),
        ...response.data.comments.map(toCommentView)
      ]
    };
  }
  async uploadImage(payload, options) {
    const formData = new FormData();
    formData.append("file", payload.file);
    const response = await this.#customFetch(
      `${this.#url}/api/alpha/upload/image`,
      {
        ...options,
        body: formData,
        headers: this.#headers,
        method: "POST"
      }
    );
    await validateResponse(response);
    const data = await response.json();
    return {
      url: data.url
    };
  }
  async #listPersonComments(payload, options) {
    const response = await this.#client.GET("/api/alpha/comment/list", {
      ...options,
      params: { query: { ...fromPageParams(payload), sort: "New" } }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.comments.map(toCommentView)
    };
  }
  async #listPersonPosts(payload, options) {
    const response = await this.#client.GET("/api/alpha/post/list", {
      ...options,
      params: { query: { ...fromPageParams(payload), sort: "New" } }
    });
    return {
      ...toPageResponse(payload),
      data: response.data.posts.map(toPostView)
    };
  }
}
const PiefedClient = buildSafeClient(UnsafePiefedClient);

async function resolveSoftware(url, options) {
  const fetch = options?.fetchFunction ?? globalThis.fetch;
  const fetchOptions = {
    headers: {
      // ...options?.headers, // TODO: Piefed doesn't allow many headers for CORS
      Accept: "application/json"
    }
  };
  const response = await fetch(`${url}/.well-known/nodeinfo`, fetchOptions);
  const data = await response.json();
  const nodeinfoLink = resolveNodeinfoLink(data);
  if (nodeinfoLink) {
    const nodeinfoResponse = await fetch(nodeinfoLink, fetchOptions);
    const nodeinfoData = await nodeinfoResponse.json();
    return nodeinfoData.software;
  }
  throw new Error("No supported nodeinfo (2.1 or 2.0) found");
}
function resolveNodeinfoLink(data) {
  return data.links.find(
    (link) => link.rel.match(
      /^http:\/\/nodeinfo\.diaspora\.software\/ns\/schema\/2\.\d+$/
    )
  )?.href;
}

const discoveryCache = /* @__PURE__ */ new Map();
class ThreadiverseClient {
  /**
   * Important: First match wins.
   */
  static get supportedSoftware() {
    return [LemmyV1Client, LemmyV0Client, PiefedClient];
  }
  get software() {
    if (!this.delegateClient || !getBaseClientConstructor(this.delegateClient).softwareName || !this.discoveredSoftware)
      throw new Error(
        "Client not initialized. Wait for getSoftware() or any other async method to resolve first"
      );
    return {
      name: getBaseClientConstructor(this.delegateClient).softwareName,
      version: this.discoveredSoftware.version
    };
  }
  delegateClient;
  discoveredSoftware;
  hostname;
  options;
  constructor(hostname, options) {
    this.hostname = hostname;
    this.options = options;
  }
  static resolveClient(software) {
    for (const Client of ThreadiverseClient.supportedSoftware) {
      if (Client.softwareName === software.name && (Client.softwareVersionRange === "*" || satisfies(software.version, Client.softwareVersionRange))) {
        return Client;
      }
    }
  }
  async banFromCommunity(...params) {
    const client = await this.ensureClient();
    return client.banFromCommunity(...params);
  }
  async blockCommunity(...params) {
    const client = await this.ensureClient();
    return client.blockCommunity(...params);
  }
  async blockInstance(...params) {
    const client = await this.ensureClient();
    return client.blockInstance(...params);
  }
  async blockPerson(...params) {
    const client = await this.ensureClient();
    return client.blockPerson(...params);
  }
  async createComment(...params) {
    const client = await this.ensureClient();
    return client.createComment(...params);
  }
  async createCommentReport(...params) {
    const client = await this.ensureClient();
    return client.createCommentReport(...params);
  }
  async createPost(...params) {
    const client = await this.ensureClient();
    return client.createPost(...params);
  }
  async createPostReport(...params) {
    const client = await this.ensureClient();
    return client.createPostReport(...params);
  }
  async createPrivateMessage(...params) {
    const client = await this.ensureClient();
    return client.createPrivateMessage(...params);
  }
  async createPrivateMessageReport(...params) {
    const client = await this.ensureClient();
    return client.createPrivateMessageReport(...params);
  }
  async deleteComment(...params) {
    const client = await this.ensureClient();
    return client.deleteComment(...params);
  }
  async deleteImage(...params) {
    const client = await this.ensureClient();
    return client.deleteImage(...params);
  }
  async deletePost(...params) {
    const client = await this.ensureClient();
    return client.deletePost(...params);
  }
  async distinguishComment(...params) {
    const client = await this.ensureClient();
    return client.distinguishComment(...params);
  }
  async editComment(...params) {
    const client = await this.ensureClient();
    return client.editComment(...params);
  }
  async editPost(...params) {
    const client = await this.ensureClient();
    return client.editPost(...params);
  }
  async featurePost(...params) {
    const client = await this.ensureClient();
    return client.featurePost(...params);
  }
  async followCommunity(...params) {
    const client = await this.ensureClient();
    return client.followCommunity(...params);
  }
  async getCaptcha(...params) {
    const client = await this.ensureClient();
    return client.getCaptcha(...params);
  }
  async getComments(...params) {
    const client = await this.ensureClient();
    return client.getComments(...params);
  }
  async getCommunity(...params) {
    const client = await this.ensureClient();
    return client.getCommunity(...params);
  }
  async getFederatedInstances(...params) {
    const client = await this.ensureClient();
    return client.getFederatedInstances(...params);
  }
  async getMode() {
    const client = await this.ensureClient();
    return getBaseClientConstructor(client).mode;
  }
  async getModlog(...params) {
    const client = await this.ensureClient();
    return client.getModlog(...params);
  }
  async getNotifications(...params) {
    const client = await this.ensureClient();
    return client.getNotifications(...params);
  }
  async getPersonDetails(...params) {
    const client = await this.ensureClient();
    return client.getPersonDetails(...params);
  }
  async getPersonMentions(...params) {
    const client = await this.ensureClient();
    return client.getPersonMentions(...params);
  }
  async getPost(...params) {
    const client = await this.ensureClient();
    return client.getPost(...params);
  }
  async getPosts(...params) {
    const client = await this.ensureClient();
    return client.getPosts(...params);
  }
  async getPrivateMessages(...params) {
    const client = await this.ensureClient();
    return client.getPrivateMessages(...params);
  }
  async getRandomCommunity(...params) {
    const client = await this.ensureClient();
    return client.getRandomCommunity(...params);
  }
  async getReplies(...params) {
    const client = await this.ensureClient();
    return client.getReplies(...params);
  }
  async getSite(...params) {
    const client = await this.ensureClient();
    return client.getSite(...params);
  }
  async getSiteMetadata(...params) {
    const client = await this.ensureClient();
    return client.getSiteMetadata(...params);
  }
  async getSoftware() {
    const client = await this.ensureClient();
    if (!this.discoveredSoftware) throw new Error("Internal error");
    return {
      name: getBaseClientConstructor(client).softwareName,
      version: this.discoveredSoftware.version
    };
  }
  async getUnreadCount(...params) {
    const client = await this.ensureClient();
    return client.getUnreadCount(...params);
  }
  async likeComment(...params) {
    const client = await this.ensureClient();
    return client.likeComment(...params);
  }
  async likePost(...params) {
    const client = await this.ensureClient();
    return client.likePost(...params);
  }
  async listCommentReports(...params) {
    const client = await this.ensureClient();
    return client.listCommentReports(...params);
  }
  async listCommunities(...params) {
    const client = await this.ensureClient();
    return client.listCommunities(...params);
  }
  async listPersonContent(...params) {
    const client = await this.ensureClient();
    return client.listPersonContent(...params);
  }
  async listPersonLiked(...params) {
    const client = await this.ensureClient();
    return client.listPersonLiked(...params);
  }
  async listPersonSaved(...params) {
    const client = await this.ensureClient();
    return client.listPersonSaved(...params);
  }
  async listPostReports(...params) {
    const client = await this.ensureClient();
    return client.listPostReports(...params);
  }
  async listReports(...params) {
    const client = await this.ensureClient();
    return client.listReports(...params);
  }
  async lockPost(...params) {
    const client = await this.ensureClient();
    return client.lockPost(...params);
  }
  async login(...params) {
    const client = await this.ensureClient();
    return client.login(...params);
  }
  async logout(...params) {
    const client = await this.ensureClient();
    return client.logout(...params);
  }
  async markAllAsRead(...params) {
    const client = await this.ensureClient();
    return client.markAllAsRead(...params);
  }
  async markCommentReplyAsRead(...params) {
    const client = await this.ensureClient();
    return client.markCommentReplyAsRead(...params);
  }
  async markPersonMentionAsRead(...params) {
    const client = await this.ensureClient();
    return client.markPersonMentionAsRead(...params);
  }
  async markPostAsRead(...params) {
    const client = await this.ensureClient();
    return client.markPostAsRead(...params);
  }
  async markPrivateMessageAsRead(...params) {
    const client = await this.ensureClient();
    return client.markPrivateMessageAsRead(...params);
  }
  async register(...params) {
    const client = await this.ensureClient();
    return client.register(...params);
  }
  async removeComment(...params) {
    const client = await this.ensureClient();
    return client.removeComment(...params);
  }
  async removePost(...params) {
    const client = await this.ensureClient();
    return client.removePost(...params);
  }
  async resolveCommentReport(...params) {
    const client = await this.ensureClient();
    return client.resolveCommentReport(...params);
  }
  async resolveObject(...params) {
    const client = await this.ensureClient();
    return client.resolveObject(...params);
  }
  async resolvePostReport(...params) {
    const client = await this.ensureClient();
    return client.resolvePostReport(...params);
  }
  async saveComment(...params) {
    const client = await this.ensureClient();
    return client.saveComment(...params);
  }
  async savePost(...params) {
    const client = await this.ensureClient();
    return client.savePost(...params);
  }
  async saveUserSettings(...params) {
    const client = await this.ensureClient();
    return client.saveUserSettings(...params);
  }
  async search(...params) {
    const client = await this.ensureClient();
    return client.search(...params);
  }
  async uploadImage(...params) {
    const client = await this.ensureClient();
    return client.uploadImage(...params);
  }
  async ensureClient() {
    if (this.delegateClient) {
      return this.delegateClient;
    }
    if (!this.discoveredSoftware) {
      if (!discoveryCache.has(this.hostname)) {
        const resolver = resolveSoftware(this.hostname, this.options);
        discoveryCache.set(this.hostname, resolver);
        try {
          await resolver;
        } catch (e) {
          discoveryCache.delete(this.hostname);
          throw e;
        }
      }
      this.discoveredSoftware = await discoveryCache.get(this.hostname);
    }
    const delegateClient = (() => {
      const Client = ThreadiverseClient.resolveClient(this.discoveredSoftware);
      if (!Client) {
        throw new UnsupportedSoftwareError(
          `${this.discoveredSoftware.name} v${this.discoveredSoftware.version} is not supported`
        );
      }
      return new Client(this.hostname, this.options);
    })();
    this.delegateClient = delegateClient;
    return delegateClient;
  }
}
function getBaseClientConstructor(client) {
  return client.constructor;
}

export { BaseClient, FediverseError, InvalidPayloadError, LemmyV0Client, LemmyV1Client, PiefedClient, ThreadiverseClient, UnexpectedResponseError, UnsupportedError, UnsupportedSoftwareError };
