#!/usr/bin/env node
"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const commander_1 = require("commander");
const chalk_1 = __importDefault(require("chalk"));
const loglevel_1 = __importDefault(require("loglevel"));
const index_1 = __importDefault(require("../index"));
const package_json_1 = __importDefault(require("../../package.json"));
const logger = loglevel_1.default.getLogger('find-process');
let type, keyword = '';
commander_1.program
    .version(package_json_1.default.version)
    .option('-t, --type <type>', 'find process by keyword type (pid|port|name)')
    .option('-p, --port', 'find process by port')
    .arguments('<keyword>')
    .action(function (kw) {
    keyword = kw;
})
    .on('--help', () => {
    console.log();
    console.log('  Examples:');
    console.log();
    console.log('    $ find-process node          # find by name "node"');
    console.log('    $ find-process 111           # find by pid "111"');
    console.log('    $ find-process -p 80         # find by port "80"');
    console.log('    $ find-process -t port 80    # find by port "80"');
    console.log();
})
    .showHelpAfterError()
    .parse(process.argv);
const opts = commander_1.program.opts();
// check keyword
if (!keyword) {
    console.error(chalk_1.default.red('Error: search keyword cannot be empty!'));
    commander_1.program.outputHelp();
    process.exit(1);
}
// check type
if (opts.port) {
    type = 'port';
}
else if (!opts.type) {
    // pid or port
    if (/^\d+$/.test(String(keyword))) {
        type = 'pid';
        keyword = Number(keyword);
    }
    else {
        type = 'name';
    }
}
else {
    type = opts.type;
}
logger.debug('find process by: type = %s, keyword = "%s"', type, keyword);
(0, index_1.default)(type, keyword)
    .then((list) => {
    if (list.length) {
        console.log('Found %s process' + (list.length === 1 ? '' : 'es') + '\n', list.length);
        for (const item of list) {
            console.log(chalk_1.default.cyan('[%s]'), item.name || 'unknown');
            console.log('pid: %s', chalk_1.default.white(item.pid));
            console.log('cmd: %s', chalk_1.default.white(item.cmd));
            console.log();
        }
    }
    else {
        console.log('No process found');
    }
}, (err) => {
    console.error(chalk_1.default.red(err.stack || err));
    process.exit(1);
});
//# sourceMappingURL=find-process.js.map