import { JSONSchema4 } from '@typescript-eslint/utils/json-schema'
/**
 * JSON schema for the newlines between option. Validates configuration for
 * adding newlines between different groups.
 */
export declare let newlinesBetweenJsonSchema: JSONSchema4
/** JSON schema for the newlines inside option. */
export declare let newlinesInsideJsonSchema: JSONSchema4
/**
 * Builds JSON schema for custom groups array configuration.
 *
 * Creates a schema that validates an array of custom group definitions.
 * Supports both single custom groups and "anyOf" groups containing multiple
 * subgroups. Each group must have a groupName and can include various matching
 * criteria.
 *
 * @example
 *   // Valid configuration:
 *   ;[
 *     {
 *       groupName: 'react',
 *       anyOf: [{ elementNamePattern: 'use*' }, { selector: 'hook' }],
 *     },
 *     {
 *       groupName: 'utils',
 *       elementNamePattern: '*Utils',
 *     },
 *   ]
 *
 * @param options - Configuration options.
 * @param options.additionalFallbackSortProperties - Extra properties for
 *   fallback sort.
 * @param options.singleCustomGroupJsonSchema - Schema for individual custom
 *   group properties.
 * @returns JSON schema for custom groups array validation.
 */
export declare function buildCustomGroupsArrayJsonSchema({
  additionalFallbackSortProperties,
  singleCustomGroupJsonSchema,
  allowedAdditionalTypeValues,
}: {
  additionalFallbackSortProperties: Record<string, JSONSchema4> | undefined
  singleCustomGroupJsonSchema: Record<string, JSONSchema4> | undefined
  allowedAdditionalTypeValues: undefined | string[]
}): JSONSchema4
export declare function buildGroupsJsonSchema({
  allowedAdditionalTypeValues,
}: {
  allowedAdditionalTypeValues: undefined | string[]
}): JSONSchema4
export declare function buildCommonGroupsJsonSchemas({
  additionalFallbackSortProperties,
  singleCustomGroupJsonSchema,
  allowedAdditionalTypeValues,
}?: {
  additionalFallbackSortProperties?: Record<string, JSONSchema4>
  singleCustomGroupJsonSchema?: Record<string, JSONSchema4>
  allowedAdditionalTypeValues?: string[]
}): Record<string, JSONSchema4>
/**
 * Builds JSON schema for custom group modifiers configuration.
 *
 * Creates a schema that validates an array of modifiers that must be present on
 * an element for it to match a custom group.
 *
 * @example
 *   // For TypeScript class members:
 *   buildCustomGroupModifiersJsonSchema([
 *     'static',
 *     'private',
 *     'readonly',
 *     'async',
 *   ])
 *
 * @param modifiers - Array of valid modifier names.
 * @returns JSON schema for modifiers array validation.
 */
export declare function buildCustomGroupModifiersJsonSchema(
  modifiers: readonly string[],
): JSONSchema4
/**
 * Builds JSON schema for custom group selector configuration.
 *
 * Creates a schema that validates a selector string used to match specific
 * types of elements in a custom group.
 *
 * @example
 *   // For class members:
 *   buildCustomGroupSelectorJsonSchema([
 *     'property',
 *     'method',
 *     'constructor',
 *     'accessor',
 *   ])
 *
 * @param selectors - Array of valid selector names.
 * @returns JSON schema for selector validation.
 */
export declare function buildCustomGroupSelectorJsonSchema(
  selectors: readonly string[],
): JSONSchema4
