import { TSESLint } from '@typescript-eslint/utils'
import { CommonPartitionOptions } from '../types/common-partition-options.js'
import { CommonGroupsOptions } from '../types/common-groups-options.js'
import { CommonOptions } from '../types/common-options.js'
/**
 * Global settings for the Perfectionist plugin.
 *
 * These settings can be configured in ESLint configuration under the
 * 'perfectionist' key and apply to all Perfectionist rules unless overridden by
 * rule-specific options.
 */
export type Settings = Partial<
  Pick<
    CommonGroupsOptions<unknown, unknown, string>,
    'newlinesBetween' | 'newlinesInside'
  > &
    CommonPartitionOptions &
    CommonOptions
>
/**
 * Extracts and validates Perfectionist settings from ESLint configuration.
 *
 * Retrieves global Perfectionist settings that apply to all rules. Validates
 * that only allowed settings are provided and throws an error if invalid
 * options are detected. This ensures configuration errors are caught early with
 * clear error messages.
 *
 * The settings are accessed from the 'perfectionist' key in ESLint's shared
 * configuration settings.
 *
 * @example
 *   // Valid usage:
 *   const settings = getSettings(context.settings)
 *   // Returns: { type: 'natural', order: 'asc', ignoreCase: true }
 *
 * @example
 *   // Invalid setting throws error:
 *   getSettings({
 *     perfectionist: {
 *       type: 'natural',
 *       invalidOption: true, // This will throw
 *     },
 *   })
 *   // Throws: Error: Invalid Perfectionist setting(s): invalidOption
 *
 * @param settings - ESLint shared configuration settings object.
 * @returns Validated Perfectionist settings or empty object if none configured.
 * @throws {Error} If invalid settings are provided.
 */
export declare function getSettings(
  settings?: TSESLint.SharedConfigurationSettings,
): Settings
