import { AnyOfCustomGroup } from '../types/common-groups-options.js'
import { RegexOption } from '../types/common-options.js'
/**
 * Parameters for testing if an element matches a custom group.
 *
 * Contains all the properties of an element that can be used for matching
 * against custom group criteria.
 */
export interface DoesCustomGroupMatchParameters {
  /**
   * Optional value of the element. Used for matching against
   * elementValuePattern in custom groups.
   */
  elementValue?: string | null
  /**
   * Optional list of decorator names applied to the element. Used for
   * matching against decoratorNamePattern in custom groups.
   */
  decorators?: string[]
  /**
   * List of modifiers applied to the element (e.g., 'static', 'private',
   * 'async'). Must include all modifiers specified in the custom group.
   */
  modifiers: string[]
  /**
   * List of selectors that describe the element type. Used for matching
   * against the selector field in custom groups.
   */
  selectors: string[]
  /**
   * Name of the element. Used for matching against elementNamePattern in
   * custom groups.
   */
  elementName: string
}
/**
 * Base structure for a single custom group configuration.
 *
 * Defines matching criteria that an element must satisfy to belong to this
 * custom group. All specified criteria must match for the element to be
 * considered part of the group.
 */
interface BaseSingleCustomGroup {
  /**
   * Pattern to match against decorator names. Element must have at least one
   * decorator matching this pattern.
   */
  decoratorNamePattern?: RegexOption
  /**
   * Pattern to match against the element's value. Used for matching literal
   * values, initializers, or expressions.
   */
  elementValuePattern?: RegexOption
  /**
   * Regular expression pattern to match the element's name. Elements matching
   * this pattern will be included in this custom group.
   */
  elementNamePattern?: RegexOption
  /**
   * List of required modifiers. Element must have ALL specified modifiers to
   * match.
   */
  modifiers?: string[]
  /** Required selector type. Element must have this exact selector to match. */
  selector?: string
}
/**
 * Checks whether an element matches the criteria of a custom group.
 *
 * Supports both single custom groups and "anyOf" groups (where matching any
 * subgroup is sufficient). For single groups, all specified criteria must
 * match. For "anyOf" groups, at least one subgroup must match.
 *
 * @example
 *   // Single custom group
 *   doesCustomGroupMatch({
 *     customGroup: {
 *       selector: 'property',
 *       modifiers: ['static'],
 *       elementNamePattern: 'on*',
 *     },
 *     elementName: 'onClick',
 *     selectors: ['property'],
 *     modifiers: ['static', 'readonly'],
 *     elementValue: null,
 *     decorators: [],
 *   })
 *   // Returns: true
 *
 * @example
 *   // AnyOf custom group
 *   doesCustomGroupMatch({
 *     customGroup: {
 *       anyOf: [
 *         { selector: 'method' },
 *         { selector: 'property', modifiers: ['static'] },
 *       ],
 *     },
 *     elementName: 'foo',
 *     selectors: ['method'],
 *     modifiers: [],
 *     elementValue: null,
 *   })
 *   // Returns: true (matches first subgroup)
 *
 * @template SingleCustomGroup - Type of individual custom group configuration.
 * @param props - Combined parameters including the custom group and element
 *   properties.
 * @returns True if the element matches the custom group criteria, false
 *   otherwise.
 */
export declare function doesCustomGroupMatch<
  SingleCustomGroup extends BaseSingleCustomGroup,
>(
  props: {
    customGroup: AnyOfCustomGroup<SingleCustomGroup> | SingleCustomGroup
  } & DoesCustomGroupMatchParameters,
): boolean
export {}
