import { JSONSchema4 } from '@typescript-eslint/utils/json-schema'
import { TSESTree } from '@typescript-eslint/types'
import { CommonPartitionOptions } from '../../types/common-partition-options.js'
import { CommonGroupsOptions } from '../../types/common-groups-options.js'
import { CommonOptions, TypeOption } from '../../types/common-options.js'
import { SortingNode } from '../../types/sorting-node.js'
/**
 * Configuration options for the sort-named-imports rule.
 *
 * Controls how named imports are sorted within import statements.
 */
export type Options = Partial<
  {
    /**
     * Whether to ignore import aliases when sorting. When true, sorts by the
     * original name rather than the alias.
     *
     * @default false
     */
    ignoreAlias: boolean
  } & CommonGroupsOptions<
    SingleCustomGroup,
    Record<string, never>,
    TypeOption
  > &
    CommonOptions<TypeOption> &
    CommonPartitionOptions
>[]
/** Extended sorting node for named import specifiers. */
export type SortNamedImportsSortingNode = SortingNode<TSESTree.ImportClause>
/**
 * Union type of all available modifiers for named imports.
 *
 * Modifiers distinguish between type imports and value imports.
 */
export type Modifier = (typeof allModifiers)[number]
/**
 * Union type of all available selectors for named imports.
 *
 * Currently only includes the 'import' selector.
 */
export type Selector = (typeof allSelectors)[number]
/** Additional configuration for a single custom group. */
interface SingleCustomGroup {
  /**
   * Array of modifiers that imports must have to match this group. Can
   * include 'type' for type imports or 'value' for value imports.
   */
  modifiers?: Modifier[]
  /**
   * The selector type this group matches. Currently only 'import' is
   * available for named imports.
   */
  selector?: Selector
}
/**
 * Array of all available selectors for named imports.
 *
 * Used for validation and configuration in the ESLint rule.
 */
export declare let allSelectors: readonly ['import']
/**
 * Array of all available modifiers for named imports.
 *
 * Used for validation and configuration in the ESLint rule.
 */
export declare let allModifiers: readonly ['value', 'type']
/**
 * JSON Schema definitions for single custom group configurations.
 *
 * Provides additional schema properties specific to the sort-named-imports
 * rule, extending the base custom group schema with element name patterns.
 */
export declare let singleCustomGroupJsonSchema: Record<string, JSONSchema4>
export {}
