import { AST_NODE_TYPES } from '@typescript-eslint/utils'
import {
  buildUseConfigurationIfJsonSchema,
  buildCommonJsonSchemas,
  buildRegexJsonSchema,
} from '../utils/json-schemas/common-json-schemas.js'
import {
  ORDER_ERROR,
  GROUP_ORDER_ERROR,
  EXTRA_SPACING_ERROR,
  MISSED_SPACING_ERROR,
} from '../utils/report-errors.js'
import { validateNewlinesAndPartitionConfiguration } from '../utils/validate-newlines-and-partition-configuration.js'
import { buildDefaultOptionsByGroupIndexComputer } from '../utils/build-default-options-by-group-index-computer.js'
import { defaultComparatorByOptionsComputer } from '../utils/compare/default-comparator-by-options-computer.js'
import {
  singleCustomGroupJsonSchema,
  allSelectors,
  allModifiers,
} from './sort-jsx-props/types.js'
import { partitionByNewLineJsonSchema } from '../utils/json-schemas/common-partition-json-schemas.js'
import { buildCommonGroupsJsonSchemas } from '../utils/json-schemas/common-groups-json-schemas.js'
import { validateCustomSortConfiguration } from '../utils/validate-custom-sort-configuration.js'
import { filterOptionsByAllNamesMatch } from '../utils/filter-options-by-all-names-match.js'
import { validateGroupsConfiguration } from '../utils/validate-groups-configuration.js'
import { generatePredefinedGroups } from '../utils/generate-predefined-groups.js'
import { getEslintDisabledLines } from '../utils/get-eslint-disabled-lines.js'
import { isNodeEslintDisabled } from '../utils/is-node-eslint-disabled.js'
import { doesCustomGroupMatch } from '../utils/does-custom-group-match.js'
import { isNodeOnSingleLine } from '../utils/is-node-on-single-line.js'
import { sortNodesByGroups } from '../utils/sort-nodes-by-groups.js'
import { createEslintRule } from '../utils/create-eslint-rule.js'
import { reportAllErrors } from '../utils/report-all-errors.js'
import { shouldPartition } from '../utils/should-partition.js'
import { computeGroup } from '../utils/compute-group.js'
import { rangeToDiff } from '../utils/range-to-diff.js'
import { getSettings } from '../utils/get-settings.js'
import { isSortable } from '../utils/is-sortable.js'
import { complete } from '../utils/complete.js'
import { matches } from '../utils/matches.js'
let cachedGroupsByModifiersAndSelectors = /* @__PURE__ */ new Map()
const ORDER_ERROR_ID = 'unexpectedJSXPropsOrder'
const GROUP_ORDER_ERROR_ID = 'unexpectedJSXPropsGroupOrder'
const EXTRA_SPACING_ERROR_ID = 'extraSpacingBetweenJSXPropsMembers'
const MISSED_SPACING_ERROR_ID = 'missedSpacingBetweenJSXPropsMembers'
let defaultOptions = {
  fallbackSort: { type: 'unsorted' },
  newlinesInside: 'newlinesBetween',
  specialCharacters: 'keep',
  newlinesBetween: 'ignore',
  partitionByNewLine: false,
  useConfigurationIf: {},
  type: 'alphabetical',
  ignoreCase: true,
  customGroups: [],
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
  groups: [],
}
const sortJsxProps = createEslintRule({
  create: context => ({
    JSXElement: node => {
      if (!isSortable(node.openingElement.attributes)) {
        return
      }
      let settings = getSettings(context.settings)
      let { sourceCode, id } = context
      let matchedContextOptions = computeMatchedContextOptions({
        sourceCode,
        context,
        node,
      })
      let options = complete(matchedContextOptions, settings, defaultOptions)
      validateCustomSortConfiguration(options)
      validateGroupsConfiguration({
        selectors: allSelectors,
        modifiers: allModifiers,
        options,
      })
      validateNewlinesAndPartitionConfiguration(options)
      let eslintDisabledLines = getEslintDisabledLines({
        ruleName: id,
        sourceCode,
      })
      let formattedMembers = node.openingElement.attributes.reduce(
        (accumulator, attribute) => {
          if (attribute.type === AST_NODE_TYPES.JSXSpreadAttribute) {
            accumulator.push([])
            return accumulator
          }
          let name = getNodeName({ attribute })
          let selectors = []
          let modifiers = []
          if (attribute.value === null) {
            modifiers.push('shorthand')
          }
          if (!isNodeOnSingleLine(attribute)) {
            modifiers.push('multiline')
          }
          selectors.push('prop')
          let predefinedGroups = generatePredefinedGroups({
            cache: cachedGroupsByModifiersAndSelectors,
            selectors,
            modifiers,
          })
          let group = computeGroup({
            customGroupMatcher: customGroup =>
              doesCustomGroupMatch({
                elementValue: attribute.value
                  ? sourceCode.getText(attribute.value)
                  : null,
                elementName: name,
                customGroup,
                selectors,
                modifiers,
              }),
            predefinedGroups,
            options,
          })
          let sortingNode = {
            isEslintDisabled: isNodeEslintDisabled(
              attribute,
              eslintDisabledLines,
            ),
            size: rangeToDiff(attribute, sourceCode),
            node: attribute,
            group,
            name,
          }
          let lastSortingNode = accumulator.at(-1)?.at(-1)
          if (
            shouldPartition({
              lastSortingNode,
              sortingNode,
              sourceCode,
              options,
            })
          ) {
            accumulator.push([])
          }
          accumulator.at(-1).push({
            ...sortingNode,
            partitionId: accumulator.length,
          })
          return accumulator
        },
        [[]],
      )
      for (let currentNodes of formattedMembers) {
        let createSortNodesExcludingEslintDisabled = function (nodes) {
          return function (ignoreEslintDisabledNodes) {
            return sortNodesByGroups({
              optionsByGroupIndexComputer:
                buildDefaultOptionsByGroupIndexComputer(options),
              comparatorByOptionsComputer: defaultComparatorByOptionsComputer,
              ignoreEslintDisabledNodes,
              groups: options.groups,
              nodes,
            })
          }
        }
        reportAllErrors({
          availableMessageIds: {
            missedSpacingBetweenMembers: MISSED_SPACING_ERROR_ID,
            extraSpacingBetweenMembers: EXTRA_SPACING_ERROR_ID,
            unexpectedGroupOrder: GROUP_ORDER_ERROR_ID,
            unexpectedOrder: ORDER_ERROR_ID,
          },
          sortNodesExcludingEslintDisabled:
            createSortNodesExcludingEslintDisabled(currentNodes),
          nodes: currentNodes,
          options,
          context,
        })
      }
    },
  }),
  meta: {
    schema: {
      items: {
        properties: {
          ...buildCommonJsonSchemas(),
          ...buildCommonGroupsJsonSchemas({
            singleCustomGroupJsonSchema,
          }),
          useConfigurationIf: buildUseConfigurationIfJsonSchema({
            additionalProperties: {
              tagMatchesPattern: buildRegexJsonSchema(),
            },
          }),
          partitionByNewLine: partitionByNewLineJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
      uniqueItems: true,
      type: 'array',
    },
    messages: {
      [MISSED_SPACING_ERROR_ID]: MISSED_SPACING_ERROR,
      [EXTRA_SPACING_ERROR_ID]: EXTRA_SPACING_ERROR,
      [GROUP_ORDER_ERROR_ID]: GROUP_ORDER_ERROR,
      [ORDER_ERROR_ID]: ORDER_ERROR,
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-jsx-props',
      description: 'Enforce sorted JSX props.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-jsx-props',
})
function computeMatchedContextOptions({ sourceCode, context, node }) {
  return filterOptionsByAllNamesMatch({
    nodeNames: node.openingElement.attributes
      .filter(attribute => attribute.type !== AST_NODE_TYPES.JSXSpreadAttribute)
      .map(attribute => getNodeName({ attribute })),
    contextOptions: context.options,
  }).find(options => {
    if (!options.useConfigurationIf?.tagMatchesPattern) {
      return true
    }
    return matches(
      sourceCode.getText(node.openingElement.name),
      options.useConfigurationIf.tagMatchesPattern,
    )
  })
}
function getNodeName({ attribute }) {
  return attribute.name.type === AST_NODE_TYPES.JSXNamespacedName
    ? `${attribute.name.namespace.name}:${attribute.name.name.name}`
    : attribute.name.name
}
export { sortJsxProps as default }
