import {
  ORDER_ERROR,
  GROUP_ORDER_ERROR,
  EXTRA_SPACING_ERROR,
  MISSED_SPACING_ERROR,
  DEPENDENCY_ORDER_ERROR,
} from '../utils/report-errors.js'
import {
  partitionByNewLineJsonSchema,
  partitionByCommentJsonSchema,
} from '../utils/json-schemas/common-partition-json-schemas.js'
import { validateNewlinesAndPartitionConfiguration } from '../utils/validate-newlines-and-partition-configuration.js'
import { buildDefaultOptionsByGroupIndexComputer } from '../utils/build-default-options-by-group-index-computer.js'
import {
  buildRegexJsonSchema,
  buildCommonJsonSchemas,
} from '../utils/json-schemas/common-json-schemas.js'
import { defaultComparatorByOptionsComputer } from '../utils/compare/default-comparator-by-options-computer.js'
import {
  singleCustomGroupJsonSchema,
  allSelectors,
  allModifiers,
} from './sort-classes/types.js'
import { buildCommonGroupsJsonSchemas } from '../utils/json-schemas/common-groups-json-schemas.js'
import { validateCustomSortConfiguration } from '../utils/validate-custom-sort-configuration.js'
import { getOverloadSignatureGroups } from './sort-classes/get-overload-signature-groups.js'
import { validateGroupsConfiguration } from '../utils/validate-groups-configuration.js'
import { generatePredefinedGroups } from '../utils/generate-predefined-groups.js'
import { sortNodesByDependencies } from '../utils/sort-nodes-by-dependencies.js'
import { getEslintDisabledLines } from '../utils/get-eslint-disabled-lines.js'
import { isNodeEslintDisabled } from '../utils/is-node-eslint-disabled.js'
import { doesCustomGroupMatch } from '../utils/does-custom-group-match.js'
import { sortNodesByGroups } from '../utils/sort-nodes-by-groups.js'
import { getNodeDecorators } from '../utils/get-node-decorators.js'
import { createEslintRule } from '../utils/create-eslint-rule.js'
import { getDecoratorName } from '../utils/get-decorator-name.js'
import { reportAllErrors } from '../utils/report-all-errors.js'
import { shouldPartition } from '../utils/should-partition.js'
import { getGroupIndex } from '../utils/get-group-index.js'
import { computeGroup } from '../utils/compute-group.js'
import { rangeToDiff } from '../utils/range-to-diff.js'
import { getSettings } from '../utils/get-settings.js'
import { isSortable } from '../utils/is-sortable.js'
import { complete } from '../utils/complete.js'
import { matches } from '../utils/matches.js'
let cachedGroupsByModifiersAndSelectors = /* @__PURE__ */ new Map()
const ORDER_ERROR_ID = 'unexpectedClassesOrder'
const GROUP_ORDER_ERROR_ID = 'unexpectedClassesGroupOrder'
const EXTRA_SPACING_ERROR_ID = 'extraSpacingBetweenClassMembers'
const MISSED_SPACING_ERROR_ID = 'missedSpacingBetweenClassMembers'
const DEPENDENCY_ORDER_ERROR_ID = 'unexpectedClassesDependencyOrder'
let defaultOptions = {
  groups: [
    'index-signature',
    ['static-property', 'static-accessor-property'],
    ['static-get-method', 'static-set-method'],
    ['protected-static-property', 'protected-static-accessor-property'],
    ['protected-static-get-method', 'protected-static-set-method'],
    ['private-static-property', 'private-static-accessor-property'],
    ['private-static-get-method', 'private-static-set-method'],
    'static-block',
    ['property', 'accessor-property'],
    ['get-method', 'set-method'],
    ['protected-property', 'protected-accessor-property'],
    ['protected-get-method', 'protected-set-method'],
    ['private-property', 'private-accessor-property'],
    ['private-get-method', 'private-set-method'],
    'constructor',
    ['static-method', 'static-function-property'],
    ['protected-static-method', 'protected-static-function-property'],
    ['private-static-method', 'private-static-function-property'],
    ['method', 'function-property'],
    ['protected-method', 'protected-function-property'],
    ['private-method', 'private-function-property'],
    'unknown',
  ],
  ignoreCallbackDependenciesPatterns: [],
  fallbackSort: { type: 'unsorted' },
  newlinesInside: 'newlinesBetween',
  partitionByComment: false,
  partitionByNewLine: false,
  newlinesBetween: 'ignore',
  specialCharacters: 'keep',
  type: 'alphabetical',
  ignoreCase: true,
  customGroups: [],
  locales: 'en-US',
  alphabet: '',
  order: 'asc',
}
const sortClasses = createEslintRule({
  create: context => ({
    ClassBody: node => {
      if (!isSortable(node.body)) {
        return
      }
      let settings = getSettings(context.settings)
      let options = complete(context.options.at(0), settings, defaultOptions)
      validateCustomSortConfiguration(options)
      validateGroupsConfiguration({
        modifiers: allModifiers,
        selectors: allSelectors,
        options,
      })
      validateNewlinesAndPartitionConfiguration(options)
      let { sourceCode, id } = context
      let eslintDisabledLines = getEslintDisabledLines({
        ruleName: id,
        sourceCode,
      })
      let className = node.parent.id?.name
      function getDependencyName(props) {
        return `${props.isStatic ? 'static ' : ''}${props.isPrivateHash ? '#' : ''}${props.nodeNameWithoutStartingHash}`
      }
      let classMethodsDependencyNames = new Set(
        node.body
          .map(member => {
            if (
              (member.type === 'MethodDefinition' ||
                member.type === 'TSAbstractMethodDefinition') &&
              'name' in member.key
            ) {
              return getDependencyName({
                isPrivateHash: member.key.type === 'PrivateIdentifier',
                nodeNameWithoutStartingHash: member.key.name,
                isStatic: member.static,
              })
            }
            return null
          })
          .filter(Boolean),
      )
      function extractDependencies(expression, isMemberStatic) {
        let dependencies = []
        function checkNode(nodeValue) {
          if (
            nodeValue.type === 'MemberExpression' &&
            (nodeValue.object.type === 'ThisExpression' ||
              (nodeValue.object.type === 'Identifier' &&
                nodeValue.object.name === className)) &&
            (nodeValue.property.type === 'Identifier' ||
              nodeValue.property.type === 'PrivateIdentifier')
          ) {
            let isStaticDependency =
              isMemberStatic || nodeValue.object.type === 'Identifier'
            let dependencyName = getDependencyName({
              isPrivateHash: nodeValue.property.type === 'PrivateIdentifier',
              nodeNameWithoutStartingHash: nodeValue.property.name,
              isStatic: isStaticDependency,
            })
            if (!classMethodsDependencyNames.has(dependencyName)) {
              dependencies.push(dependencyName)
            }
          }
          if (nodeValue.type === 'Property') {
            traverseNode(nodeValue.key)
            traverseNode(nodeValue.value)
          }
          if (nodeValue.type === 'ConditionalExpression') {
            traverseNode(nodeValue.test)
            traverseNode(nodeValue.consequent)
            traverseNode(nodeValue.alternate)
          }
          if (
            'expression' in nodeValue &&
            typeof nodeValue.expression !== 'boolean'
          ) {
            traverseNode(nodeValue.expression)
          }
          if ('object' in nodeValue) {
            traverseNode(nodeValue.object)
          }
          if ('callee' in nodeValue) {
            traverseNode(nodeValue.callee)
          }
          if ('init' in nodeValue && nodeValue.init) {
            traverseNode(nodeValue.init)
          }
          if ('body' in nodeValue && nodeValue.body) {
            traverseNode(nodeValue.body)
          }
          if ('left' in nodeValue) {
            traverseNode(nodeValue.left)
          }
          if ('right' in nodeValue) {
            traverseNode(nodeValue.right)
          }
          if ('elements' in nodeValue) {
            let elements = nodeValue.elements.filter(
              currentNode => currentNode !== null,
            )
            for (let element of elements) {
              traverseNode(element)
            }
          }
          if ('argument' in nodeValue && nodeValue.argument) {
            traverseNode(nodeValue.argument)
          }
          if ('arguments' in nodeValue) {
            let shouldIgnore = false
            if (nodeValue.type === 'CallExpression') {
              let functionName =
                'name' in nodeValue.callee ? nodeValue.callee.name : null
              shouldIgnore =
                functionName !== null &&
                matches(
                  functionName,
                  options.ignoreCallbackDependenciesPatterns,
                )
            }
            if (!shouldIgnore) {
              for (let argument of nodeValue.arguments) {
                traverseNode(argument)
              }
            }
          }
          if ('declarations' in nodeValue) {
            for (let declaration of nodeValue.declarations) {
              traverseNode(declaration)
            }
          }
          if ('properties' in nodeValue) {
            for (let property of nodeValue.properties) {
              traverseNode(property)
            }
          }
          if ('expressions' in nodeValue) {
            for (let nodeExpression of nodeValue.expressions) {
              traverseNode(nodeExpression)
            }
          }
        }
        function traverseNode(nodeValue) {
          if (Array.isArray(nodeValue)) {
            for (let nodeItem of nodeValue) {
              traverseNode(nodeItem)
            }
          } else {
            checkNode(nodeValue)
          }
        }
        traverseNode(expression)
        return dependencies
      }
      let overloadSignatureGroups = getOverloadSignatureGroups(node.body)
      let formattedNodes = node.body.reduce(
        (accumulator, member) => {
          let name
          let dependencies = []
          if (member.type === 'StaticBlock') {
            name = 'static'
          } else if (member.type === 'TSIndexSignature') {
            name = sourceCode.text.slice(
              member.range.at(0),
              member.typeAnnotation?.range.at(0) ?? member.range.at(1),
            )
          } else if (member.key.type === 'Identifier') {
            ;({ name } = member.key)
          } else {
            name = sourceCode.getText(member.key)
          }
          let isPrivateHash =
            'key' in member && member.key.type === 'PrivateIdentifier'
          let decorated = false
          let decorators = []
          if ('decorators' in member) {
            decorators = getNodeDecorators(member).map(decorator =>
              getDecoratorName({ sourceCode, decorator }),
            )
            decorated = decorators.length > 0
          }
          let memberValue
          let modifiers = []
          let selectors = []
          let addSafetySemicolonWhenInline = true
          switch (member.type) {
            case 'TSAbstractPropertyDefinition':
            case 'PropertyDefinition':
              if ('static' in member && member.static) {
                modifiers.push('static')
              }
              if ('declare' in member && member.declare) {
                modifiers.push('declare')
              }
              if (member.type === 'TSAbstractPropertyDefinition') {
                modifiers.push('abstract')
              }
              if (decorated) {
                modifiers.push('decorated')
              }
              if ('override' in member && member.override) {
                modifiers.push('override')
              }
              if ('readonly' in member && member.readonly) {
                modifiers.push('readonly')
              }
              if (
                'accessibility' in member &&
                member.accessibility === 'protected'
              ) {
                modifiers.push('protected')
              } else if (
                ('accessibility' in member &&
                  member.accessibility === 'private') ||
                isPrivateHash
              ) {
                modifiers.push('private')
              } else {
                modifiers.push('public')
              }
              if ('optional' in member && member.optional) {
                modifiers.push('optional')
              }
              if (
                member.value?.type === 'ArrowFunctionExpression' ||
                member.value?.type === 'FunctionExpression'
              ) {
                if (member.value.async) {
                  modifiers.push('async')
                }
                selectors.push('function-property')
              } else if (member.value) {
                memberValue = sourceCode.getText(member.value)
                dependencies = extractDependencies(member.value, member.static)
              }
              selectors.push('property')
              break
            case 'TSAbstractMethodDefinition':
            case 'MethodDefinition':
              if (member.static) {
                modifiers.push('static')
              }
              if (member.type === 'TSAbstractMethodDefinition') {
                modifiers.push('abstract')
              } else if (!node.parent.declare) {
                addSafetySemicolonWhenInline = false
              }
              if (decorated) {
                modifiers.push('decorated')
              }
              if (member.override) {
                modifiers.push('override')
              }
              if (member.accessibility === 'protected') {
                modifiers.push('protected')
              } else if (member.accessibility === 'private' || isPrivateHash) {
                modifiers.push('private')
              } else {
                modifiers.push('public')
              }
              if (member.optional) {
                modifiers.push('optional')
              }
              if (member.value.async) {
                modifiers.push('async')
              }
              if (member.kind === 'constructor') {
                selectors.push('constructor')
              }
              if (member.kind === 'get') {
                selectors.push('get-method')
              }
              if (member.kind === 'set') {
                selectors.push('set-method')
              }
              selectors.push('method')
              break
            case 'TSAbstractAccessorProperty':
            case 'AccessorProperty':
              if (member.static) {
                modifiers.push('static')
              }
              if (member.type === 'TSAbstractAccessorProperty') {
                modifiers.push('abstract')
              }
              if (decorated) {
                modifiers.push('decorated')
              }
              if (member.override) {
                modifiers.push('override')
              }
              if (member.accessibility === 'protected') {
                modifiers.push('protected')
              } else if (member.accessibility === 'private' || isPrivateHash) {
                modifiers.push('private')
              } else {
                modifiers.push('public')
              }
              selectors.push('accessor-property')
              break
            case 'TSIndexSignature':
              if (member.static) {
                modifiers.push('static')
              }
              if (member.readonly) {
                modifiers.push('readonly')
              }
              selectors.push('index-signature')
              break
            case 'StaticBlock':
              addSafetySemicolonWhenInline = false
              selectors.push('static-block')
              dependencies = extractDependencies(member, true)
              break
          }
          let predefinedGroups = generatePredefinedGroups({
            cache: cachedGroupsByModifiersAndSelectors,
            selectors,
            modifiers,
          })
          let group = computeGroup({
            customGroupMatcher: customGroup =>
              doesCustomGroupMatch({
                elementValue: memberValue,
                elementName: name,
                customGroup,
                decorators,
                modifiers,
                selectors,
              }),
            predefinedGroups,
            options,
          })
          let overloadSignatureGroupMemberIndex =
            overloadSignatureGroups.findIndex(overloadSignatures =>
              overloadSignatures.includes(member),
            )
          let overloadSignatureGroupMember =
            overloadSignatureGroups[overloadSignatureGroupMemberIndex]?.at(-1)
          let sortingNode = {
            dependencyNames: [
              getDependencyName({
                nodeNameWithoutStartingHash: name.startsWith('#')
                  ? name.slice(1)
                  : name,
                isStatic: modifiers.includes('static'),
                isPrivateHash,
              }),
            ],
            overloadSignaturesGroupId:
              overloadSignatureGroupMemberIndex === -1
                ? null
                : overloadSignatureGroupMemberIndex,
            size: overloadSignatureGroupMember
              ? rangeToDiff(overloadSignatureGroupMember, sourceCode)
              : rangeToDiff(member, sourceCode),
            isEslintDisabled: isNodeEslintDisabled(member, eslintDisabledLines),
            addSafetySemicolonWhenInline,
            node: member,
            dependencies,
            group,
            name,
          }
          let lastSortingNode = accumulator.at(-1)?.at(-1)
          if (
            shouldPartition({
              lastSortingNode,
              sortingNode,
              sourceCode,
              options,
            })
          ) {
            accumulator.push([])
          }
          accumulator.at(-1).push({
            ...sortingNode,
            partitionId: accumulator.length,
          })
          return accumulator
        },
        [[]],
      )
      function sortNodesExcludingEslintDisabled(ignoreEslintDisabledNodes) {
        let nodesSortedByGroups = formattedNodes.flatMap(nodes2 =>
          sortNodesByGroups({
            isNodeIgnored: sortingNode =>
              getGroupIndex(options.groups, sortingNode) ===
              options.groups.length,
            optionsByGroupIndexComputer:
              buildDefaultOptionsByGroupIndexComputer(options),
            comparatorByOptionsComputer: defaultComparatorByOptionsComputer,
            ignoreEslintDisabledNodes,
            groups: options.groups,
            nodes: nodes2,
          }),
        )
        return sortNodesByDependencies(nodesSortedByGroups, {
          ignoreEslintDisabledNodes,
        })
      }
      let nodes = formattedNodes.flat()
      reportAllErrors({
        newlinesBetweenValueGetter: ({
          computedNewlinesBetween,
          right,
          left,
        }) => {
          if (
            left.overloadSignaturesGroupId !== null &&
            left.overloadSignaturesGroupId === right.overloadSignaturesGroupId
          ) {
            return 0
          }
          return computedNewlinesBetween
        },
        availableMessageIds: {
          missedSpacingBetweenMembers: MISSED_SPACING_ERROR_ID,
          unexpectedDependencyOrder: DEPENDENCY_ORDER_ERROR_ID,
          extraSpacingBetweenMembers: EXTRA_SPACING_ERROR_ID,
          unexpectedGroupOrder: GROUP_ORDER_ERROR_ID,
          unexpectedOrder: ORDER_ERROR_ID,
        },
        sortNodesExcludingEslintDisabled,
        options,
        context,
        nodes,
      })
    },
  }),
  meta: {
    schema: [
      {
        properties: {
          ...buildCommonJsonSchemas(),
          ...buildCommonGroupsJsonSchemas({
            singleCustomGroupJsonSchema,
          }),
          ignoreCallbackDependenciesPatterns: buildRegexJsonSchema(),
          partitionByComment: partitionByCommentJsonSchema,
          partitionByNewLine: partitionByNewLineJsonSchema,
        },
        additionalProperties: false,
        type: 'object',
      },
    ],
    messages: {
      [DEPENDENCY_ORDER_ERROR_ID]: DEPENDENCY_ORDER_ERROR,
      [MISSED_SPACING_ERROR_ID]: MISSED_SPACING_ERROR,
      [EXTRA_SPACING_ERROR_ID]: EXTRA_SPACING_ERROR,
      [GROUP_ORDER_ERROR_ID]: GROUP_ORDER_ERROR,
      [ORDER_ERROR_ID]: ORDER_ERROR,
    },
    docs: {
      url: 'https://perfectionist.dev/rules/sort-classes',
      description: 'Enforce sorted classes.',
      recommended: true,
    },
    type: 'suggestion',
    fixable: 'code',
  },
  defaultOptions: [defaultOptions],
  name: 'sort-classes',
})
export { sortClasses as default }
